/**
 * @file TCA9538.cpp
 * @author Joel von Rotz (joel.vonrotz@maxonmotor.com)
 * @brief 
 * @version 0.1
 * @date 2019-07-30
 * 
 * @copyright Copyright (c) 2019, maxon motor ag - all rights reserved
 * 
 */
#include "TCA9538.h"

/**
 * @brief Construct a new TCA9538 object
 * 
 * @param i2c_device  Reference of a mbed I2C-object
 * @param address     Based on the given address-configurations in the datasheet
 */
TCA9538::TCA9538(I2C& i2c_device, uint8_t address) : 
  m_i2c_address(address << 1),
  m_i2c_device(i2c_device)
{

}

/**
 * @brief Configures the desired Register
 * 
 * The register of the IO-expander can be configured by calling this set-function with
 * a register-value and the value, which will be written onto the register.
 * 
 * Possible values for the registers are:
 * 
 * - TCA9538::INPUT
 * - TCA9538::OUTPUT
 * - TCA9538::POLARITY
 * - TCA9538::CONFIG
 * 
 * @param index   the desired register to be changed
 * @param value   the value written on the given register
 * @return true   no success
 * @return false  success
 */
bool  TCA9538::set(tca9538_registers_t index, uint8_t value)
{
  bool status;

  if(index == INPUT)
  {
    return 1;
  }

  char data[2] = {index, value};
  status = m_i2c_device.write(m_i2c_address, data, 2);
  return status;
}

/**
 * @brief Returns Value of desired Register
 * 
 * Returns the value from the given register. The parameter value is the same
 * as in TCA9538::set(tca9538_registers_t index, uint8_t value).
 * 
 * @param index     Register to be read from
 * @return uint8_t  The Value of the desired Register
 */
uint8_t TCA9538::get(tca9538_registers_t index)
{
  bool status;
  char value;
  char char_index = index;

  status  = m_i2c_device.write(m_i2c_address, &char_index, 1, true);
  status |= m_i2c_device.read(m_i2c_address, &value, 1, false);
  return status;
}
