/**
 * @brief Declaration of the BNO055 class 
 * 
 * @file bno055.h
 * @author Joel von Rotz
 * @date 18.07.2018
 */
#ifndef BNO055_H
#define BNO055_H

#include "mbed.h"
#include "bno055_config.h"
#include "bno055_registermap.h"
#include "bno055_definitions.h"
#include "i2c_master.h"


/**
 * @brief This is a library for the sensor BNO055, a 9 axis sensor by <a href="https://www.bosch-sensortec.com/">Bosch Sensortec</a>, written by <a href="https://os.mbed.com/users/joelvonrotz/">Joel von Rotz</a>
 * <img src="https://ae01.alicdn.com/kf/HTB1n3aPNpXXXXaFXVXXq6xXFXXXi/BNO055-gyroscope-Absolute-Orientation-9-Axis-Sensor.jpg_640x640.jpg" alt="Image of a flying Potato here!">
 * Currently, this library is compatible with the mbed LPC1768 only and probably won't change in the future, unless somebody takes their time to port the library to other microcontroller (would very much appreciate it though).
 * \n\n             
 * <center><a href="https://os.mbed.com/users/joelvonrotz/code/I2C-Master/">This Library requires The <strong>TWI_Master</strong> Library to work</a></center>
 * \n\n
 * <h2>Example Programm Without Debug</h2>
 * @code
 * #include "mbed.h"
 * #include "i2c_master.h"
 * #include "bno055.h"
 * 
 * Serial  pc(USBTX,USBRX,115200);
 * 
 * I2C_Master i2c_master(p9, p10);
 * DigitalOut bno055_reset(p11);
 * BNO055     bno055(0x28,         //7-bit slave address
 *                   i2c_master,   //reference of the i2c-master object
 *                   bno055_reset, //reference of the reset-pin connected with the mbed
 *                   true);        //enable external 32kHz oscillator
 * 
 * 
 * int main() {
 *   bno055.setOperationMode(OPERATION_MODE_NDOF);
 *   
 *   while(1) {
 *     bno055.getEulerDegrees();
 *     pc.printf("x: %4.3f y: %4.3f z: %4.3f\n\r", bno055.euler.x,bno055.euler.y,bno055.euler.z);
 *     wait_ms(100);
 *   }
 * }
 * @endcode
 * \n
 * <h2>Example Programm With Debug</h2>
 * @code
 * #define DEBUGGING_ENABLED
 * 
 * #include "mbed.h"
 * #include "i2c_master.h"
 * #include "bno055.h"
 * 
 * Serial  pc(USBTX,USBRX,115200);
 * 
 * I2C_Master i2c_master(p9, p10);
 * DigitalOut bno055_reset(p11);
 * BNO055     bno055(0x28,         //7-bit slave address
 *                   i2c_master,   //reference of the i2c-master object
 *                   bno055_reset, //reference of the reset-pin connected with the mbed
 *                   true,         //enable external 32kHz oscillator
 *                   pc);          //debugging Serial object
 * 
 * int main() {
 *   bno055.setOperationMode(OPERATION_MODE_NDOF);
 *   
 *   while(1) {
 *     bno055.getEulerDegrees();
 *     pc.printf("x: %4.3f y: %4.3f z: %4.3f\n\r", bno055.euler.x,bno055.euler.y,bno055.euler.z);
 *     wait_ms(100);
 *   }
 * }
 * @endcode
 * 
 * During the execution of the programm, open your terminal. Everything should now be printed.
 * 
 */
class BNO055
{
public:
    #ifdef DEBUGGING_ENABLED
        BNO055(uint8_t slave_address, I2C_Master& i2c_master, DigitalOut& ResetPin, bool external_clk, Serial& DEBUG_SERIAL);
    #else
        BNO055(uint8_t slave_address, I2C_Master& i2c_master, DigitalOut& ResetPin, bool external_clk);
    #endif
//=---------------------------------------
    struct bno055_format_s
    {
        uint8_t                 units;
        bno055_orientation_t    orientation;
        bno055_temperature_t    temperature;
        bno055_euler_t          euler;
        bno055_gyro_t           gyroscope;
        bno055_acceleration_t   acceleration;
    };

    struct bno055_id_s
    {
        uint8_t     chip;
        uint8_t     accel;
        uint8_t     gyro;
        uint8_t     magneto;
        uint8_t     bl_rev;
        uint16_t    sw_rev;
    };
    
    struct bno055_mode_s
    {
        uint8_t     power;
        uint8_t     operation;
    };

    struct bno055_system_s
    {
        uint8_t     status;
        uint8_t     error;
    };

    struct bno055_axis_s
    {
        uint8_t     map;
        uint8_t     sign;
    };

    struct bno055_interrupt_s
    {
        uint8_t     status;
        struct bno055_gyro_interrupt_s
        {
            bool    high_rate;
            bool    any_motion;
        }gyroscope;

        struct bno055_accel_interrupt_s
        {
            bool    no_motion;
            bool    any_motion;
            bool    high_g;
        }acceleration;
    };

    struct bno055_quaternion_s
    {
        float w;
        float x;
        float y;
        float z;
    };

    struct bno055_data_s
    {
        float x;
        float y;
        float z;
    };

    struct bno055_calibration_s
    {
        uint8_t system;
        uint8_t gyro;
        uint8_t acceleration;
        uint8_t magneto;
        uint8_t status;
    };

//=---------------------------------------

    
    float          temperature;
    uint16_t        orientation;

    bno055_format_s     format;
    bno055_id_s         id;
    bno055_mode_s       mode;
    bno055_system_s     system;
    bno055_axis_s       axis;

    bno055_data_s       gravity_vector;
    bno055_data_s       linear_accel;
    bno055_data_s       accel;
    bno055_data_s       magneto;
    bno055_data_s       gyro;
    bno055_data_s       euler;

    bno055_quaternion_s     quaternion;
    bno055_calibration_s    calibration;

    bno055_interrupt_s  interrupt;
    
//=---------------------------------------

    void        setUnitFormat(bno055_orientation_t  new_orientation_format  = WINDOWS,
                              bno055_temperature_t  new_temperature_format  = CELSIUS,
                              bno055_euler_t        new_euler_format        = DEGREES,
                              bno055_gyro_t         new_gyroscope_format    = DEGREE_PER_SEC,
                              bno055_acceleration_t new_acceleration_format = ACCELERATION 
                              );

    void        getUnitFormat(void);
    void        getIDs(void);
    void        setPowerMode(bno055_powermode_t new_power_mode  = POWER_NORMAL);
    uint8_t     getPowerMode(void);
    void        setOperationMode(bno055_opr_mode_t new_opr_mode = OPERATION_MODE_CONFIGMODE);
    uint8_t     getOperationMode(void);
    void        setPage(bno055_page_t new_page = PAGE_0);
    uint8_t     getPage(void);
    uint8_t     getSystemStatus(void);
    uint8_t     getSystemError(void);
    void        useExternalOscillator(bool enabled);
    void        resetSW(void);
    void        resetHW(void);
    void        assignAxis( bno055_axis_t x_axis = X_AXIS,
                            bno055_axis_t y_axis = Y_AXIS,
                            bno055_axis_t z_axis = Z_AXIS);
    void        setAxisSign(bno055_axis_sign_t x_sign = POSITIVE,
                            bno055_axis_sign_t y_sign = POSITIVE,
                            bno055_axis_sign_t z_sign = POSITIVE);
    void        setOrientation(bno055_remap_options_t orientation_placement = REMAP_OPTION_P1);
    void        getOrientation(void);
    void        getCalibrationStatus(void);

    void        getInterruptFlag(void);
    void        setEnableInterrupts(bno055_enable_t accel_no_motion,
                                    bno055_enable_t accel_any_motion,
                                    bno055_enable_t accel_high_g,
                                    bno055_enable_t gyro_high_rate,
                                    bno055_enable_t gyro_any_motion
                                    );
    uint8_t     getEnabledInterrupts(void);
    
    void        setInterruptMask(bno055_enable_t accel_no_motion,
                                 bno055_enable_t accel_any_motion,
                                 bno055_enable_t accel_high_g,
                                 bno055_enable_t gyro_high_rate,
                                 bno055_enable_t gyro_any_motion
                                 );
    uint8_t     getInterruptMask(void);
    
    void        configAccelerationInterrupt(bno055_config_int_axis_t    high_axis,
                                            bno055_config_int_axis_t    am_nm_axis,
                                            bno055_any_motion_sample_t  am_dur
                                            );
    void        configGyroscopeInterrupt(bno055_config_int_axis_t hg_z_axis,
                                         bno055_config_int_axis_t hg_y_axis,
                                         bool   high_rate_unfilt,
                                         bool   any_motion_unfilt
                                         );

    void        getAcceleration(void);
    void        getMagnetometer(void);
    void        getGyroscope(void);
    void        getEulerDegrees(void);
    void        getQuaternion(void);
    void        getLinearAcceleration(void);
    void        getGravityVector(void);
    void        getTemperature(void);

private:
    I2C_Master&     m_i2c_master;
    DigitalOut&     m_ResetPin;
    
    //DEBUG SERIAL
    #ifdef DEBUGGING_ENABLED
        Serial&     m_DEBUG_SERIAL;
    #endif

    uint8_t     m_bno055_address;

    uint8_t     page;
    uint8_t     arrayIndex;
    float       sensor_data_converted;
    uint8_t     register_data[8];
    int16_t     sensor_data[4];

    void        get(bno055_reg_t  address, bno055_data_s& data, const float format, bool is_euler);
    void        get(bno055_reg_t  address, bno055_quaternion_s& data, const float format);

    
};

#endif /* BNO055_H */


