/*
  By Joseph Ellsworth CTO of A2WH
  Take a look at A2WH.com Producing Water from Air using Solar Energy
  March-2016 License: https://developer.mbed.org/handbook/MIT-Licence
  Please contact us http://a2wh.com for help with custom design projects.

  See Also:
    http://www.ti.com/lit/ds/symlink/hdc1080.pdf
    https://github.com/closedcube/ClosedCube_HDC1080_Arduino/blob/master/src/ClosedCube_HDC1080.cpp
    http://e.pavlin.si/2016/06/04/hdlc-like-data-link-via-rs485/
*/

/**
#include "mbed.h"
#include <stdint.h>

//Pin Defines for I2C Bus
#define D_SDA                  PB_7 // specific for Nucleo-F303K8
#define D_SCL                  PB_6 // specific for Nucleo-F303K8
I2C hdc_i2c(D_SDA, D_SCL);
#include "hdc1080.h"

// Host PC Communication channels
Serial pc(USBTX, USBRX); // tx, rx
DigitalOut myled(LED1);

int main()
{
    pc.baud(9600);
    while(1) {
        printf("\r\\nHDC1080 Test\r\n");
        myled = !myled;
        hdc_begin();
        uint16_t manId = hdc_readManufactId();
        float tempC = hdc_readTemp();     
        float humid = hdc_readHumid();  
        unsigned long  serNum = hdc_readSerial();
        printf("manId=x%x, tempC=%0.3f humid=%0.3f serfNum=%ld\r\n",
          manId, tempC, humid, serNum);
        wait(3.0f);
    }
}
**/



#ifndef hdc1080_h
#define hdc1080_h
#include "mbed.h"

#define hdc_chip_addr (0x40 << 1) // Page #8.5.1.1 - 1000000
// left shift 1 bit for 7 bit address required by
// I2C library
//#define hdc_chip_addr 0x40 // Page #8.5.1.1 - 1000000
const int hdc_off_temp  = 0x00;
const int hdc_off_humid = 0x01;
const int hdc_off_config = 0x02;
const int hdc_off_man_id = 0xFE;
const int hdc_off_serial_first = 0xFB;
const int hdc_off_serial_mid = 0xFC;
const int hdc_off_serial_last = 0xFD;
char hdc_comm = hdc_off_man_id;
const float hdc_chip_error = -255;
const unsigned long hdc_chip_err_l = 0;
char hdc_buff[5];

void hdc_begin()
{
    //printf("hdc_begin\r\n");
    memset(hdc_buff,0,3);
    hdc_buff[0] = hdc_off_config;
    int res = hdc_i2c.write(hdc_chip_addr, hdc_buff, 2);
    #ifdef DEBUG3
    printf("hdc_begin write res=%d\r\n", res);
    #endif
}

uint16_t hdc_readData16(int chip_addr, int offset)
{
    memset(hdc_buff,0,3);
    // send chip address onto buss
    hdc_buff[0] = offset;
    int res = hdc_i2c.write(chip_addr, hdc_buff, 1);
    if (res != 0) {
        #ifdef DEBUG3
        printf("error talking to chip %d offst=%d\r\n", chip_addr, offset);
        #endif
        return 0;
    }
    // read data from chip
    wait(0.015);
    memset(hdc_buff,0,3);
    res = hdc_i2c.read(hdc_chip_addr, hdc_buff,2);
    if (res != 0) {
        #ifdef DEBUG3
        printf("error reading chip %d offst=%d\r\n", chip_addr, offset);
        #endif
        return 0;
    }
    return  hdc_buff[0] << 8 | hdc_buff[1];
}

/* Read temperature from hdc_1080 chip.  Returns float
containing the Celcius temperature or hdc_chip_error if
error occurs reading the sensor */
float hdc_readTemp()
{
    uint16_t  rawT = hdc_readData16(hdc_chip_addr, hdc_off_temp);
    if (rawT == 0) {
        #ifdef DEBUG3
        printf("error reading hdc chip temp\r\n");
        #endif
        return hdc_chip_error;
    } else {
        float temp = ((float) rawT / pow(2.0f, 16.0f)) * 165.0f - 40.0f;
        #ifdef DEBUG3
        printf("temp=%0.3f\r\n", temp);
        #endif
        return temp;
    }
}

/* Read humidity from hdc_1080 chip.  Returns a float
containing the humidity or hdc_chip_error if error 
occurs reading the sensor */
float hdc_readHumid()
{
    uint16_t  rawH = hdc_readData16(hdc_chip_addr, hdc_off_humid);
    if (rawH == 0) {
        #ifdef DEBUG3
        printf("error reading hdc chip humid\r\n");
        #endif
        return hdc_chip_error;
    } else {
        float humid = ((float) rawH / pow(2.0f, 16.0f)) * 100.0f;
        #ifdef DEBUG3
        printf("humid humid=%0.3f\r\n", humid);
        #endif
        return humid;
    }
}

int hdc_readManufactId()
{   
    uint16_t  rawid = hdc_readData16(hdc_chip_addr, hdc_off_man_id);
    if (rawid == 0) {
        #ifdef DEBUG3
        printf("error reading hdc chip manId\r\n");
        #endif
        return (int) hdc_chip_error;
    } else {  
        #ifdef DEBUG3      
        printf("man id=%x\r\n", (int) rawid);
        #endif
        return rawid;
    }    
}

unsigned long hdc_readSerial()
{    
    wait(0.015);
    memset(hdc_buff,0,4);
    hdc_buff[0] = hdc_off_man_id;
    int res = hdc_i2c.write(hdc_chip_addr, hdc_buff, 1);
    if (res != 0) {      
      #ifdef DEBUG3
      printf("Error writing chip addr res=%d\r\n", res);
      #endif
      return (unsigned long) hdc_chip_err_l;
    }      

    wait(0.015);
    memset(hdc_buff,0,4);
    res = hdc_i2c.read(hdc_chip_addr, hdc_buff,4);
    if (res != 0) {
      #ifdef DEBUG3
      printf("Errot reading chip serial res=%d#\r\n", res);
      #endif
      return (unsigned long) hdc_chip_err_l;
    }
      
    unsigned long rawser = hdc_buff[0] << 16 | hdc_buff[1] << 8 | hdc_buff[0];
    #ifdef DEBUG3
    printf("ser=%lu\r\n", rawser);
    #endif
    return rawser;
}

#endif
