#include "mbed.h"
#include "M2XStreamClient.h"
#include "MTSSerial.h"
#include "Wifi.h"
#include "include_me.h"
#include "math.h"

using namespace mts;

/* This example shows how to do a basic connectivity test to AT&T M2X Cloud 
 * using the MTS Wifi shield board. You will need to change the network
 * SSID and security key. You may need to chage the security type.
 */

using namespace mts;

const char key[] = "your_api_key";    // Replace with your M2X API key
const char feed[] = "your_feed_id";   // Replace with your blueprint Feed ID
const char stream[] = "your_stream_name"; // Replace with your stream name  
char name[] = "your_device_location_name"; // Name of current location of datasource

double latitude = 30.3748076;
double longitude = -97.7386896; // You can also read those values from a GPS
double elevation = 400.00;

DigitalOut myled(D7);   
AnalogIn tempSensor(A0);    

// Note: This callback function needs to be debugged further as it does not print 
// the actual fetched reports, instead both "at" and "value" pointers print the 
// timestamps of each report.    
void on_data_point_found(const char* at, const char* value, int index, void* context) {
  printf("Found a data point, index: %d\r\n", index);
  printf("At: %s Value: %s \r\n", at, value);   
}

void on_location_found(const char* name,
                       double latitude,
                       double longitude,
                       double elevation,
                       const char* timestamp,
                       int index,
                       void* context) {
  printf("Found a location, index: %d\r\n", index);
  printf("Name: %s  Latitude: %lf  Longitude: %lf\r\n", name, latitude, longitude);
  printf("Elevation: %lf  Timestamp: %s\r\n", elevation, timestamp);
}


int main()
{
    char amb_temp[6];
    int response;
    int a;
    int adc_scale = 4095; 
    int B = 3975;
    float resistance; 
    float temperature;
    float temperature_f;    
          
    //Set the network parameters
    std::string ssid =  "your_wifi_ssid"; 
    std::string securityKey = "your_wifi_passphrase"; 
    Wifi::SecurityType securityType = Wifi::WPA2;

    //Wait for wifi module to boot up
    for (int i = 10; i >= 0; i = i - 2) {
        wait(2);
        printf("Waiting %d seconds...\n\r", i);
    }

    //Setup serial interface to WiFi module
    MTSSerial* serial = new MTSSerial(D8, D2, 256, 4096);   
    serial->baud(9600);

    Transport::setTransport(Transport::WIFI);
    
    //Setup Wifi class
    Wifi* wifi = Wifi::getInstance();
    printf("Init: %s\n\r", wifi->init(serial) ? "SUCCESS" : "FAILURE");

    //Setup and check connection
    printf("Set Network: %s\n\r", getCodeNames(wifi->setNetwork(ssid, securityType, securityKey)).c_str());
    printf("Set DHCP: %s\n\r", getCodeNames(wifi->setDeviceIP("DHCP")).c_str());
    while (! wifi->connect()) {
        printf("Connect: Failure\r\n");
        wait(1);
    }
    printf("Connect: Success\r\n");
    printf("Is Connected: %s\n\r", wifi->isConnected() ? "True" : "False");
    
    printf("Ping Server: %s\n\r", wifi->ping("8.8.8.8") ? "Success" : "Failed");
    wait(1);
    

    // Initialize the M2X client
    Client client;    
    M2XStreamClient m2xClient(&client, key);

    // update location
    response = m2xClient.updateLocation(feed, name, latitude, longitude, elevation);
    printf("updateLocation response code: %d\r\n", response);
    if (response == -1) while (true) ;
    
    response = m2xClient.updateLocation(feed, name, latitude, longitude, elevation);
    printf("updateLocation response code: %d\r\n", response);
    if (response == -1) while (true) ;    
            
    //while(0)    
    for (int i = 0; i < 5; i++)
    {    
        myled = 1; // LED is ON    
        a = tempSensor.read_u16();
               
        resistance = (float)(adc_scale-a)*10000/a; //get the resistance of the sensor;              
        temperature = 1/(log(resistance/10000)/B+1/298.15)-273.15;  //convert to temperature via datasheet        
        temperature_f = (1.8 * temperature) + 32.0;
        sprintf(amb_temp, "%0.2f", temperature_f);  
        
        printf("Temp Sensor Analog Reading is 0x%X = %d   ", a, a);         
        printf("Current Temperature: %f C  %f F \n\r", temperature, temperature_f); 
                
        response = m2xClient.post(feed, stream, amb_temp);
        printf("Post response code: %d\r\n", response);
        if (response == -1) while (true) ;           
                
        myled = 0; // LED is OFF        

        delay(5000);        
    }           
    
    
    // fetch location
    response = m2xClient.readLocation(feed, on_location_found, NULL);
    printf("readLocation response code: %d\r\n", response);
    if (response == -1) while (true) ;  
        
    // fetch temperature
    response = m2xClient.fetchValues(feed, stream, on_data_point_found, NULL);
    printf("Fetch response code: %d\r\n", response);
    if (response == -1) while (true) ;
        
}

