#ifndef HighPWM_H
#define HighPWM_H

#include "mbed.h"

#ifndef F_CLK
#define F_CLK   96000000
#endif

#define us_2_F_CLK (F_CLK/1000000)
#define ms_2_F_CLK (F_CLK/1000)
#define s_2_F_CLK (F_CLK)

/** Library that allows faster and/or higher resolution PWM output
  *
  * This library is based original on FastPWM by Erik Olieman but without floating point (double) calculation in each call of pulsewidth_xx.
  * This change was nessecary to get a performant motion controller.
  *
  * Library can directly replace standard mbed PWM library. Only limitation is that the maximum PWM period is four times shorter
  * The maximum achievable period is roughly 40 seconds, I dont think that should be a problem.
  * Do take into account all PWM objects (based on defailt PWM lib) will run four times faster than default.
  *
  * Contrary to the default mbed library, this library takes doubles instead of floats. The compiler will autocast if needed,
  * but do take into account it is done for a reason, your accuracy will otherwise be limitted by the floating point precision.
  *
  * Using this library for a RC servo the resolution of steps (min to max) is increased from 1000 to 96000.
  *
  * This library can be also used as a analoge output (with external low-pass filter) with a frequency of e.g. 20kHz and a resolution 4800 steps (similar to a 12 bit DAC) instead of 50 steps (similar to a 5 bit DAC) on original PWM lib.
  *
  * In your program you can define F_CLK if you use a different clock frequency than the default one. 
  *
  * Only works on LPC1768 for now. If you want support for the other one, send a PM and I will have a look, but I cannot even compile for it.
  */
class HighPWM {
public:
    /**
    * Create a HighPWM object connected to the specified pin
    *
    * @param pin - PWM pin to connect to
    */
    HighPWM(PinName pin);
    
    /**
    * Set the PWM frequency, specified in Hz (double), keeping the duty cycle the same.
    */
    void frequency(double frq);
    
    /**
    * Set the PWM period, specified in seconds (double), keeping the duty cycle the same.
    */
    void period(double seconds);
    
    /**
    * Set the PWM period, specified in micro-seconds (double), keeping the duty cycle the same.
    */
    void period_us(double us);
    
    /**
    * Set the PWM period, specified in milli-seconds (int), keeping the duty cycle the same.
    */
    void period_ms(unsigned int ms);
    
    /**
    * Set the PWM period, specified in micro-seconds (int), keeping the duty cycle the same.
    */
    void period_us(unsigned int us);
    
    /**
    * Set the PWM period, specified in cpu ticks based on 96MHz (int), keeping the duty cycle the same.
    */
    void period_ticks(unsigned int ticks);
    
    /**
    * Set the PWM pulsewidth, specified in seconds (double), keeping the period the same.
    */
    void pulsewidth(double seconds);
    
    /**
    * Set the PWM pulsewidth, specified in micro seconds (double), keeping the period the same.
    */
    void pulsewidth_us(double us);
    
    /**
    * Set the PWM pulsewidth, specified in milli-seconds (int), keeping the period the same.
    */
    void pulsewidth_ms(unsigned int ms);
    
    /**
    * Set the PWM pulsewidth, specified in micro-seconds (int), keeping the period the same.
    */
    void pulsewidth_us(unsigned int us);
    
    /**
    * Set the PWM pulsewidth, specified in cpu ticks based on 96MHz (int), keeping the period the same.
    */
    void pulsewidth_ticks(unsigned int ticks);
    
    /**
    * Set the ouput duty-cycle, specified as a percentage (double)
    *
    * @param duty - A double value representing the output duty-cycle, specified as a percentage.  The value should lie between 0.0 (representing on 0%) and 1.0 (representing on 100%).
    */
    void write(double duty);
    
    /**
    * Return the ouput duty-cycle, specified as a percentage (double)
    *
    * @param return - A double value representing the output duty-cycle, specified as a percentage.
    */
    double read( void );
    
    /**
    * An operator shorthand for write()
    */
    HighPWM& operator= (double value);
    
    /**
    * An operator shorthand for read()
    */
    operator double();
    
private:
    PwmOut PWMObject;
    //double _duty;
    int _period;
    int _pulsewidth;
    unsigned int PWMUnit;
    
    __IO uint32_t *MR;

};
#endif