/*
 * Author: MIGUEL URCO
 * Date: 10/11/2014
 * Notes: Checks the Ethernet cable connection
*/
#if 1

#include "mbed.h"
#include "rtos.h"
#include "EthernetInterface.h"
#include "SerialDriver.h"
#include "I2CLCD.h"

#include "k64f_EthLink.h"
#include "EthUtils.h"
#include "JroIpdata.h"
#include "JroSIR.h"
#include "CR2.h"

#define mbed_reset        NVIC_SystemReset

#define BUFFER_SIZE     256
static char rx_buffer[BUFFER_SIZE];
static char tx_buffer[BUFFER_SIZE]; 

//THREADS
Thread *ser_thread_ptr;
Thread *eth_thread_ptr;
Thread *lcd_thread_ptr;
Mutex cr2_mutex;
Mutex eth_mutex;
Mutex lcd_mutex;

//SERIAL
#define SERIAL_BAUDRATE     1000000
const char* UART_MSG_OK = "CR2 Successful";
const char* UART_MSG_KO = "CR2 Unsuccessful";
const char* UART_MSG_NC = "CR2 Not connected";

Serial screen(USBTX, USBRX);
SerialDriver uart(D1, D0);


//LCD
I2C i2c_device(I2C_SDA, I2C_SCL);
I2CLCD lcd(i2c_device, 0x78, LCD20x2);

//const char LCD_WELCOME[]    =   "  JICAMARCA - CR2   ";
const char LCD_IP_INI[]     =   "IP : checking...    ";
const char LCD_IP_NC[]      =   "IP : Not connected  ";
//const char LCD_IP_NI[]      =   "IP : Not initialized";

const char LCD_CR2_INI[]    =   "CR2: checking...    ";
const char LCD_CR2_NC[]     =   "CR2: Not detected   ";
const char LCD_CR2_ENA[]    =   "     Outputs Enabled     ";
const char LCD_CR2_DIS[]    =   "     Otuputs Disabled    ";

//ETHERNET
#define ECHO_SERVER_PORT   2000
const char DEVICE_NAME[] = "jrodds";

IpData ipData(tx_buffer);

k64fEthLink eth_link;
EthernetInterface eth;
TCPSocketServer server;

//CR2
SPI spi_device(D11, D12, D13);

DigitalOut    cr2_mreset(D4);
DigitalOut    cr2_outramp(D5);
DigitalOut    cr2_sp_mode(D6);
DigitalOut    cr2_cs(D7);
DigitalOut    cr2_io_reset(D9);
DigitalInOut  cr2_updclk(D10);

CR2 cr2_device(&spi_device, &cr2_mreset, &cr2_outramp, &cr2_sp_mode, &cr2_cs, &cr2_io_reset, &cr2_updclk);

//LEDS
DigitalOut LedR(LED1);          
DigitalOut LedG(LED2);          
DigitalOut LedB(LED3);  

DigitalIn  IPResetButton(PTB2, PullUp);

int printAddr(char *_ip, char *_mask, char *_gateway){
    lcd.printf(_ip, 0, 0);
    lcd.printf(_mask, 0, 1);

    screen.putc(0x0A);
    screen.putc(0x0D);
    
    for (int i=0; i<20; i++){
        screen.putc(_ip[i]);
    }
    screen.putc(0x0A);
    screen.putc(0x0D);
    
    for (int i=0; i<20; i++){
        screen.putc(_mask[i]);
    }  
     
    screen.putc(0x0A);
    screen.putc(0x0D);
    
    for (int i=0; i<20; i++){
        screen.putc(_gateway[i]);
    }

    screen.putc(0x0A);
    screen.putc(0x0D);
    
    return 1;
}

int wasIPResetPressed(){
    
    if (IPResetButton)
        return 0;
    
    //Wait until button will be released
    while(!IPResetButton){
        Thread::wait(50);
    }
    
    return 1;
}


void waitSerialData_thread(void const *args){
    
    int n;
    bool successful;
    
    //Thread::signal_wait(0x1);
    
    LedG = 1;
    
    uart.baud(SERIAL_BAUDRATE);
    
    while(1){
        LedG = 0;
        successful = false;
        
        if (uart.isRxBufferEmpty()){
            Thread::wait(100);
            continue;
            }
            
        Thread::wait(10);
        n = uart.read(rx_buffer, 255, false);
        
        //******************** BLINK LED *****************************
        for (int i=0; i<n; i++){
            LedG = !LedG;
            Thread::wait(10);
        }
        lcd.printf("Serial command received", 0, 1);
        //******************** CR2 NOT INITIALIZED *******************
        if (!cr2_device.wasInitialized()){
            for (int i=0; i<strlen(UART_MSG_NC); i++){   
                uart.putc(UART_MSG_NC[i]);
            }
            continue;
        }
        //Lock cr2_device before execute any command
        cr2_mutex.lock();
        
        //********************* SUCCESSFUL DATA **********************
        //if (n == 40)   
            //if (cr2_device.setAllDevice(rx_buffer) == 1)
                //successful = true;
        
        //******************** REPLY UART*****************************
        if (successful)      
            //for (int i=0; i<strlen(UART_MSG_OK); i++)
            uart.printf(UART_MSG_OK);
        
        else
            uart.printf(UART_MSG_KO);
        
        cr2_mutex.unlock();
        
    }
    
}

void waitEthData_thread(void const *args){
    
    TCPSocketConnection client;
    //int status;
    int n, totalSize=0;
    char _ip[MAX_IP_LEN], _mask[MAX_IP_LEN], _gateway[MAX_IP_LEN];
    
    //Lock ethernet resource until initial configuration will be loaded
    eth_mutex.lock();
    
    readIpConfig(_ip, _mask, _gateway);
    
    LedB = 1;
    
    if (eth.init(_ip, _mask, _gateway) != 0){
        //mbed_reset();
    }
    
    eth.setName(DEVICE_NAME);
    eth.connect();
    Thread::wait(200);
    eth_mutex.unlock();
    
    //If ethernet connection fails then try again
    while(eth_link.GetELink() != 0){
        eth.connect();
        Thread::wait(200);
    }
    
    server.bind(ECHO_SERVER_PORT);
    server.listen(1);
    
    LedB = 0;

    while(1)
    {
        LedB = 0;
        n = 0;
        totalSize = 0;
        
        server.accept(client);
        
        client.set_blocking(false, 500); // Timeout after (0.5)s
        
        while (true) {
            LedR = !LedR;
            n = client.receive(rx_buffer, sizeof(rx_buffer));
            if (n <= 0) break;
            totalSize += n;
            Thread::wait(10);
        }
        
        LedB = 1;
    
        if (totalSize < 1){
            client.close();
            continue;
        }
        
        if (ipData.decode(rx_buffer, totalSize) == 0){
            client.send(ipData.getKOData(0x00), ipData.getKODataLen());
            client.close();
            continue;
        }
        lcd.printf("Eth command received", 0, 1);
        //******************** CR2 NOT INITIALIZED *******************
        if (!cr2_device.wasInitialized()){
            client.send(ipData.getNIData(ipData.getCmd()), ipData.getNIDataLen());
            client.close();
            continue;
        }
        
        //******************** REPLY REQ *****************************
        
        //********* IP CONFIG ****
        if (ipData.getCmd() == CMD_CHANGE_IP){
            
            if ( splitIpConf(ipData.getPayload(), _ip, _mask, _gateway) ){
                //changing ip and reseting device
                client.send(ipData.getOKData(ipData.getCmd()), ipData.getOKDataLen());
                Thread::wait(200);
                client.close();
                
                saveIpConfig(_ip, _mask, _gateway);
                eth.setNewAddr(_ip, _mask, _gateway);
                Thread::wait(500);
                //mbed_reset();
                continue;
            }
            else{
                client.send(ipData.getKOData(ipData.getCmd()), ipData.getKODataLen());
                Thread::wait(200);
                client.close();
                continue;
            }
        }
        
        //Lock cr2_device before execute any command
        cr2_mutex.lock();
        
        // ********** OTHER COMMANDS
        cr2_device.writeBlock(ipData.getPayloadLen(),ipData.getPayload());
        //cr2_device.setCommand(ipData.getCmd(), ipData.getPayload(), ipData.getPayloadLen());
        ipData.encode(ipData.getCmd(), cr2_device.getCmdAnswer(), cr2_device.getCmdAnswerLen());
        
        client.send(ipData.getTxData(), ipData.getTxDataLen());
        Thread::wait(10);
        client.close();
        
        cr2_mutex.unlock();
    }
}

void lcdView_thread(void const *args){
    
    char c=0;
    char lcd_status[2]="\xF7";
    
    lcd.printf(LCD_CR2_INI, 0, 0);
    lcd.printf(LCD_IP_INI, 0, 1);
    Thread::wait(2000);
    
    while(1){
        /*
        if (lcd_mutex.trylock()==false){
            Thread::wait(1000);
            continue;
        }
        */
        
        if (cr2_device.wasInitialized()){
            
            //RF ENABLED OR DISABLED
            if (c==1){
                //if(cr2_device.isRFEnabled()){
                    //lcd.printf(LCD_CR2_ENA, 0, 0);
                //}
                //else{
                    lcd.printf(LCD_CR2_DIS, 0, 0);
                //}
                lcd_status[0] = 0x01;
                lcd.printf(lcd_status, 19, 0);
            
            }
            //FREQUENCY
            if (c==3){
                char tmp[21];
                sprintf(tmp, "Fout = %7.5f*Clk", cr2_device.getFreqFactor1());
                lcd.printf(tmp, 0, 0);
            }
            if (c==5){
                char tmp[21];
                sprintf(tmp, "Modulation = %s", cr2_device.getModeStr() );
                lcd.printf(tmp, 0, 0);
            }
        }
        else{
            lcd.printf(LCD_CR2_NC, 0, 0);
            lcd_status[0] = lcd_status[0] ^ 0xFF;
            lcd.printf(lcd_status, 19, 0);
        }
        
        //Verify if ethernet resource is free
        if (eth_mutex.trylock()){
            if (eth_link.GetELink() == 0){
                if (c==0){
                    lcd.printf("IP :                ", 0, 1);
                    lcd.printf(eth.getIPAddress(), 5, 1);
                }
                if (c==2){
                    lcd.printf("MSK:                ", 0, 1);
                    lcd.printf(eth.getNetworkMask(), 5, 1);
                }      
                if (c==4){
                    lcd.printf("GW :                ", 0, 1);
                    lcd.printf(eth.getGateway(), 5, 1);
                }     
            }
            else{
                lcd.printf(LCD_IP_NC, 0, 1);
            }
            eth_mutex.unlock();
        }
        else{
            lcd.printf(LCD_IP_INI, 0, 1);
        }
        c++;
        if (c>5) c=0;
        //lcd_mutex.unlock();
        Thread::wait(1000);
    }
    
}

int main() 
{   
    screen.baud(9600);

    screen.putc(0x0A);
    screen.putc(0x0D);
    screen.putc(0x33);
    screen.putc(0x33);
    
    ser_thread_ptr = new Thread(&waitSerialData_thread);
    eth_thread_ptr = new Thread(&waitEthData_thread);
    lcd_thread_ptr = new Thread(&lcdView_thread);
    
    LedR = 1;
    
    screen.putc(0x33);
    screen.putc(0x32);
    
    while(true){
        if (cr2_device.init())
            break;
            
        LedR = !LedR;
        Thread::wait(250);
        
        if (wasIPResetPressed()){
            eraseIpConfig();
            mbed_reset();
        }
    }
    
    LedR = 0;
    /*
    screen.putc(0x33);
    screen.putc(0x31);
    */
    LedR = 1;
    
    //cr2_device.defaultSettings();
    LedR = 0;
    /*
    screen.putc(0x33);
    screen.putc(0x30);
    */
    Thread::wait(1000);
    
    //int c=0;
    while(true){
        
        //if (cr2_device.isRFEnabled()){
            //LedR = 0;
        //}
        //else{
            LedR = 1;
        //}
        
        if (wasIPResetPressed()){
            eraseIpConfig();
            mbed_reset();
        }
        
        Thread::wait(200);
    }
    
}

#endif