#include "SimplifiedTouchListener.h"



/*
    This class handles the touch panel on the LPC4088. 
    
    Unlike the original TouchListener class, all it does in response to the user touching the screen 
    is to raise a thread signal, nothing else.
*/

// Note that SimplifiedTouchListener is a singleton - we do not need or want there to be more than one instance of it
// (there is only one board, and only one touch panel)
SimplifiedTouchListener * SimplifiedTouchListener::theSimplifiedTouchListener = NULL;

#ifdef GET_TOUCH_COORDS_AT_TOUCH_TIME // defined in header
touch_coordinate_t SimplifiedTouchListener::touchCoords;
#endif // GET_TOUCH_COORDS_AT_TOUCH_TIME

#ifdef MULTI_TOUCH_TECHNIQUE_5
Timer SimplifiedTouchListener::touchTimer;
const uint32_t SimplifiedTouchListener::touchIntervalMilliSec = 500; // i.e. 0.5 second
#endif


// Singleton class - return the one and only instance, first creating it if necessary
SimplifiedTouchListener *SimplifiedTouchListener::GetInstance(osThreadId newAppThread, TouchPanel* newTouchPanel)
{
    if (theSimplifiedTouchListener == NULL) {
        theSimplifiedTouchListener = new SimplifiedTouchListener(newAppThread, newTouchPanel);
    }
    return theSimplifiedTouchListener;
}

// Singleton class - private constructor
SimplifiedTouchListener::SimplifiedTouchListener(osThreadId newAppThread, TouchPanel* newTouchPanel)
{
    gotTouchEvent = false;
    
    appThread = newAppThread;
    touchPanel = newTouchPanel;
    
#define DO_THE_ATTACH
#ifdef DO_THE_ATTACH
    functionPointer.attach(&ListenerFunction);
    touchPanel->setListener(&functionPointer);
#endif // DO_THE_ATTACH

#ifdef MULTI_TOUCH_TECHNIQUE_5
    touchTimer.stop();
    touchTimer.reset();
    touchTimer.start();
#endif
}
    

/*
    This is the function that the system invokes whenever it sees that the user has touched the LPC4088 touch panel.
    In response, we set a thread signal - nothing else. Whatever is waiting on this thread signal can then decide 
    how to respond to the touch event.
*/
void SimplifiedTouchListener::ListenerFunction(void)
{
#if defined MULTI_TOUCH_TECHNIQUE_5
    if(touchTimer.read_ms() > touchIntervalMilliSec) {
#endif
        if (theSimplifiedTouchListener != NULL) {
            theSimplifiedTouchListener->gotTouchEvent = true;
#ifdef GET_TOUCH_COORDS_AT_TOUCH_TIME // Make sure the touch coords match the current touch event
            theSimplifiedTouchListener->touchPanel->read(touchCoords); // touchCoords passed by reference (see TouchPanel.h)
#endif // GET_TOUCH_COORDS_AT_TOUCH_TIME
            osSignalSet(theSimplifiedTouchListener->appThread, TOUCH_EVENT);
        }
#if defined MULTI_TOUCH_TECHNIQUE_5
        touchTimer.stop();
        touchTimer.reset();
        touchTimer.start();
    }
#endif
}

/*
    Tells the caller whether or not we got a touch event.
    
    Caller may have applied a timeout to his signal_wait, so when signal_wait returns, cannot tell 
    if it did so because it got the signal or because it timed out. Calling this function will tell him.
    
    Note that this is a 'once per touch event' call - we clear the 'got touch event' flag in this function.
    Until the next touch event, subsequent calls will return false.
    
    If we got a touch event, we return the coordinates in the arguments provided. 
    If not, we do nothing to the arguments.
    
    Args: pointers to variables to contain the X, Y and Z touch coordinates (the touch panel returns all three)
    
    Return value: true if this is the first call after a touch event, false otherwise.
*/
bool SimplifiedTouchListener::GotTouchEvent(short *x, short *y, short *z)
{
    if(gotTouchEvent) {
        gotTouchEvent = false;

#ifndef GET_TOUCH_COORDS_AT_TOUCH_TIME
        touch_coordinate_t touchCoords;
        touchPanel->read(touchCoords); // touchCoords passed by reference (see TouchPanel.h)
#endif // GET_TOUCH_COORDS_AT_TOUCH_TIME
// #else we have already got the coords, in the touch event listener function

        *x = touchCoords.x;
        *y = touchCoords.y;
        *z = touchCoords.z;

        return true;
    }
    
    // 'else'...
    return false;
}


// static TouchListener members
FunctionPointer SimplifiedTouchListener::functionPointer;

