#ifndef NUDGEANDDAMPPAGEHANDLER_H
#define NUDGEANDDAMPPAGEHANDLER_H

#include "mbed.h"
#include "DMBoard.h"

#include "USBHostGC.h"

#include "GuiLib.h"

/*
    A struct to relate the index of an easyGUI touch area on one of the nudge and damp pages
    with the easyGUI variable to which it corresponds, and other parameters 
    which the NumericKeypadPageHandler will need to edit the variable
*/
typedef struct structNudgeAndDampEasyGUITouchAreaAndVariables {
    int touchAreaIndex;
    GuiConst_TEXT * easyGUIVariablePtr;
    int maxValue;
    // all values are integers (so there will be zero places of decimals for all of them)
    int easyGUICallingPage; // We are handling more than one page here
    char variableTitle[41];
} NudgeAndDampEasyGUITouchAreaAndVariables;


/*
    This class handles user interaction with the easyGUI pages that deal with the nudge and damp factors 
    for the various components. These pages are all so similar that it seems wasteful 
    to have a separate class for each. 
    
    This class also deals with the Fan Power page, since that is very similar to the Nudge and Damp pages 
    (and the Fan Power parameters are handled in the "Heating Control" frame in the 500 Series GC VB6 application, 
    just like the Nudge and Damp pages).
    
    Note that this class is a singleton - we do not need or want there to be more than one instance of it
    (we do not want multiple values for the various nudge and damp factors, etc, nor will we show 
    more than one easyGUI page to the user at the same time).
*/

class NudgeAndDampPageHandler {
public:
    /**
    * Static method to create (if necessary) and retrieve the single NudgeAndDampPageHandler instance
    */
    static NudgeAndDampPageHandler * GetInstance(USBDeviceConnected* newUsbDevice, USBHostGC* newUsbHostGC);
    
    /**
    * Version of the above that does not create the instance. Provided for callers that do not have 
    * the 'usbDevice' and 'usbHostGC' pointers, and that just want access to the instance
    */
    static NudgeAndDampPageHandler * GetInstance(void);
    
    bool TouchAreaIsOnNudgeAndDampPage(int touchAreaIndex);

    bool DealWithTouch(int touchAreaIndex);
    
    void DisplayingEasyGUIPage(int pageNumber, bool updateEasyGUIVariables);
    
    static bool PageIsANudgeAndDampPage(int pageNumber);

private:    
    static NudgeAndDampPageHandler * theNudgeAndDampPageHandlerInstance;
    
    USBDeviceConnected* usbDevice;
    USBHostGC* usbHostGC;
    
    // singleton class -> constructor is private
    NudgeAndDampPageHandler(USBDeviceConnected* newUsbDevice, USBHostGC* newUsbHostGC);    
    ~NudgeAndDampPageHandler();
    
    void SendCommandToGCAndGetResponse(char* command, char* response);
    bool SendCommandToGCWithDACKResponse(char *cmd);

    bool GetNudgeOrDampFactorFromGC(char *cmd, GuiConst_TEXT* easyGUIVariable);
    bool SetNudgeOrDampFactorInGC(char *cmd, GuiConst_TEXT* easyGUIVariable);

    void GetColumnOvenNudgeAndDampFactorsFromGC(void);
    void GetColumnDHNudgeAndDampFactorsFromGC(void);
    void GetInjectorNudgeAndDampFactorsFromGC(void);
    void GetDetectorNudgeAndDampFactorsFromGC(void);
    void GetAuxiliaryNudgeAndDampFactorsFromGC(void);
    void GetFanPowerValuesFromGC(void);

    void SetColumnOvenNudgeAndDampFactorsInGC(void);
    void SetColumnDHNudgeAndDampFactorsInGC(void);
    void SetInjectorNudgeAndDampFactorsInGC(void);
    void SetDetectorNudgeAndDampFactorsInGC(void);
    void SetAuxiliaryNudgeAndDampFactorsInGC(void);
    void SetFanPowerValuesInGC(void);

    enum { COUNT_OF_VARIABLES_FOR_TOUCH_AREAS = 19 };
    NudgeAndDampEasyGUITouchAreaAndVariables variablesForTouchArea[COUNT_OF_VARIABLES_FOR_TOUCH_AREAS];

    int GetIndexOfValueToEditForTouchArea(int touchAreaIndex);
};

#endif // NUDGEANDDAMPPAGEHANDLER_H
