#include "NudgeAndDampPageHandler.h"
#include "EasyGUITouchAreaIndices.h"
#include "GetGCStatusLoop.h"
#include "NumericKeypadPageHandler.h"
#include "USBHostGCUtilities.h"

/*
    Static function to tell the caller whether or not the specified easyGUI page is one of the "Nudge and Damp" pages,
    and therefore one that this class handles. (All the other "xxxPageHandler" classes handle one page only.)
    
    Static so that the caller does not have to get/create the "NudgeAndDampPageHandler" instance
    until it knows that the page needs to be handled by this class.
    
    Args: the number of the page in question
    
    Returns true if it is a nudge and damp page, false otherwise
*/
bool NudgeAndDampPageHandler::PageIsANudgeAndDampPage(int pageNumber)
{
    return ((pageNumber == GuiStruct_ColumnOvenNudgeAndDampPage_0)
         || (pageNumber == GuiStruct_ColumnDHNudgeAndDampPage_0)
         || (pageNumber == GuiStruct_InjectorNudgeAndDampPage_0)
         || (pageNumber == GuiStruct_DetectorNudgeAndDampPage_0)
         || (pageNumber == GuiStruct_AuxiliaryNudgeAndDampPage_0)
         || (pageNumber == GuiStruct_FanPowerPage_0));
}


/*                      
    Note that this class is a singleton - we do not need or want there to be more than one instance of it
    (we do not want multiple values for the same nudge and damp parameters, etc, nor will we show 
    more than one easyGUI page to the user at the same time).
*/
NudgeAndDampPageHandler * NudgeAndDampPageHandler::theNudgeAndDampPageHandlerInstance = NULL;


/*
    Singleton class - return the one and only instance, first creating it if necessary.
*/
NudgeAndDampPageHandler * NudgeAndDampPageHandler::GetInstance(USBDeviceConnected* newUsbDevice, USBHostGC* newUsbHostGC)
{
    if (theNudgeAndDampPageHandlerInstance == NULL) {
        theNudgeAndDampPageHandlerInstance = new NudgeAndDampPageHandler(newUsbDevice, newUsbHostGC);
    }
    
    return theNudgeAndDampPageHandlerInstance;
}

/*
    Overriden version of the above, that does not take any arguments and does not create the instance 
    if it does not already exist.
    
    Provided for callers that do not have the 'usbDevice' and 'usbHostGC' pointers, and just want access 
    to the instance if it exists
*/
NudgeAndDampPageHandler * NudgeAndDampPageHandler::GetInstance(void)
{
    return theNudgeAndDampPageHandlerInstance;
}


// Singleton class - private constructor
NudgeAndDampPageHandler::NudgeAndDampPageHandler(USBDeviceConnected* newUsbDevice, USBHostGC* newUsbHostGC)
{
    usbDevice = newUsbDevice;
    usbHostGC = newUsbHostGC;


    variablesForTouchArea[0].touchAreaIndex     = COLUMN_OVEN_NUDGE_FACTOR_EDIT;
    variablesForTouchArea[0].easyGUIVariablePtr = GuiVar_columnOvenNudgeFactor;
    variablesForTouchArea[0].maxValue           = 2550;
    variablesForTouchArea[0].easyGUICallingPage = GuiStruct_ColumnOvenNudgeAndDampPage_0;
    strcpy(variablesForTouchArea[0].variableTitle, "Nudge");

    variablesForTouchArea[1].touchAreaIndex     = COLUMN_OVEN_DAMP_FACTOR_EDIT;
    variablesForTouchArea[1].easyGUIVariablePtr = GuiVar_columnOvenDampFactor;
    variablesForTouchArea[1].maxValue           = 2550;
    variablesForTouchArea[1].easyGUICallingPage = GuiStruct_ColumnOvenNudgeAndDampPage_0;
    strcpy(variablesForTouchArea[1].variableTitle, "Damp");

    variablesForTouchArea[2].touchAreaIndex     = COLUMN_OVEN_RAMP_NUDGE_FACTOR_EDIT;
    variablesForTouchArea[2].easyGUIVariablePtr = GuiVar_columnOvenRampNudgeFactor;
    variablesForTouchArea[2].maxValue           = 2550;
    variablesForTouchArea[2].easyGUICallingPage = GuiStruct_ColumnOvenNudgeAndDampPage_0;
    strcpy(variablesForTouchArea[2].variableTitle, "Ramp Nudge");

    variablesForTouchArea[3].touchAreaIndex     = COLUMN_OVEN_RAMP_DAMP_FACTOR_EDIT;
    variablesForTouchArea[3].easyGUIVariablePtr = GuiVar_columnOvenRampDampFactor;
    variablesForTouchArea[3].maxValue           = 2550;
    variablesForTouchArea[3].easyGUICallingPage = GuiStruct_ColumnOvenNudgeAndDampPage_0;
    strcpy(variablesForTouchArea[3].variableTitle, "Ramp Damp");

    variablesForTouchArea[4].touchAreaIndex     = COLUMN_OVEN_TEMP_OFFSET_EDIT;
    variablesForTouchArea[4].easyGUIVariablePtr = GuiVar_columnOvenTempOffset;
    variablesForTouchArea[4].maxValue           = 2550;
    variablesForTouchArea[4].easyGUICallingPage = GuiStruct_ColumnOvenNudgeAndDampPage_0;
    strcpy(variablesForTouchArea[4].variableTitle, "Temp. Offset");

    variablesForTouchArea[5].touchAreaIndex     = COLUMN_DH_NUDGE_FACTOR_EDIT;
    variablesForTouchArea[5].easyGUIVariablePtr = GuiVar_columnDHNudgeFactor;
    variablesForTouchArea[5].maxValue           = 2550;
    variablesForTouchArea[5].easyGUICallingPage = GuiStruct_ColumnDHNudgeAndDampPage_0;
    strcpy(variablesForTouchArea[5].variableTitle, "DH Col. Nudge");

    variablesForTouchArea[6].touchAreaIndex     = COLUMN_DH_DAMP_FACTOR_EDIT;
    variablesForTouchArea[6].easyGUIVariablePtr = GuiVar_columnDHDampFactor;
    variablesForTouchArea[6].maxValue           = 2550;
    variablesForTouchArea[6].easyGUICallingPage = GuiStruct_ColumnDHNudgeAndDampPage_0;
    strcpy(variablesForTouchArea[6].variableTitle, "DH Col. Damp");

    variablesForTouchArea[7].touchAreaIndex     = COLUMN_DH_RAMP_NUDGE_FACTOR_EDIT;
    variablesForTouchArea[7].easyGUIVariablePtr = GuiVar_columnDHRampNudgeFactor;
    variablesForTouchArea[7].maxValue           = 2550;
    variablesForTouchArea[7].easyGUICallingPage = GuiStruct_ColumnDHNudgeAndDampPage_0;
    strcpy(variablesForTouchArea[7].variableTitle, "DH Ramp Nudge");

    variablesForTouchArea[8].touchAreaIndex     = COLUMN_DH_RAMP_DAMP_FACTOR_EDIT;
    variablesForTouchArea[8].easyGUIVariablePtr = GuiVar_columnDHRampDampFactor;
    variablesForTouchArea[8].maxValue           = 2550;
    variablesForTouchArea[8].easyGUICallingPage = GuiStruct_ColumnDHNudgeAndDampPage_0;
    strcpy(variablesForTouchArea[8].variableTitle, "DH Ramp Damp");

    variablesForTouchArea[9].touchAreaIndex     = INJECTOR_NUDGE_FACTOR_EDIT;
    variablesForTouchArea[9].easyGUIVariablePtr = GuiVar_injectorNudgeFactor;
    variablesForTouchArea[9].maxValue           = 2550;
    variablesForTouchArea[9].easyGUICallingPage = GuiStruct_InjectorNudgeAndDampPage_0;
    strcpy(variablesForTouchArea[9].variableTitle, "Inj. Nudge");

    variablesForTouchArea[10].touchAreaIndex     = INJECTOR_DAMP_FACTOR_EDIT;
    variablesForTouchArea[10].easyGUIVariablePtr = GuiVar_injectorDampFactor;
    variablesForTouchArea[10].maxValue           = 2550;
    variablesForTouchArea[10].easyGUICallingPage = GuiStruct_InjectorNudgeAndDampPage_0;
    strcpy(variablesForTouchArea[10].variableTitle, "Inj. Damp");

    variablesForTouchArea[11].touchAreaIndex     = DETECTOR_NUDGE_FACTOR_EDIT;
    variablesForTouchArea[11].easyGUIVariablePtr = GuiVar_detectorNudgeFactor;
    variablesForTouchArea[11].maxValue           = 2550;
    variablesForTouchArea[11].easyGUICallingPage = GuiStruct_DetectorNudgeAndDampPage_0;
    strcpy(variablesForTouchArea[11].variableTitle, "Det. Nudge");

    variablesForTouchArea[12].touchAreaIndex     = DETECTOR_DAMP_FACTOR_EDIT;
    variablesForTouchArea[12].easyGUIVariablePtr = GuiVar_detectorDampFactor;
    variablesForTouchArea[12].maxValue           = 2550;
    variablesForTouchArea[12].easyGUICallingPage = GuiStruct_DetectorNudgeAndDampPage_0;
    strcpy(variablesForTouchArea[12].variableTitle, "Det. Damp");

    variablesForTouchArea[13].touchAreaIndex     = AUXILIARY_NUDGE_FACTOR_EDIT;
    variablesForTouchArea[13].easyGUIVariablePtr = GuiVar_auxiliaryNudgeFactor;
    variablesForTouchArea[13].maxValue           = 2550;
    variablesForTouchArea[13].easyGUICallingPage = GuiStruct_AuxiliaryNudgeAndDampPage_0;
    strcpy(variablesForTouchArea[13].variableTitle, "Aux. Nudge");

    variablesForTouchArea[14].touchAreaIndex     = AUXILIARY_DAMP_FACTOR_EDIT;
    variablesForTouchArea[14].easyGUIVariablePtr = GuiVar_auxiliaryDampFactor;
    variablesForTouchArea[14].maxValue           = 2550;
    variablesForTouchArea[14].easyGUICallingPage = GuiStruct_AuxiliaryNudgeAndDampPage_0;
    strcpy(variablesForTouchArea[14].variableTitle, "Aux. Damp");

    variablesForTouchArea[15].touchAreaIndex     = FAN_POWER_NORMAL_EDIT;
    variablesForTouchArea[15].easyGUIVariablePtr = GuiVar_fanPowerNormal;
    variablesForTouchArea[15].maxValue           = 255;
    variablesForTouchArea[15].easyGUICallingPage = GuiStruct_FanPowerPage_0;
    strcpy(variablesForTouchArea[15].variableTitle, "Fan Normal");
    
    variablesForTouchArea[16].touchAreaIndex     = FAN_POWER_COOLING_EDIT;
    variablesForTouchArea[16].easyGUIVariablePtr = GuiVar_fanPowerCooling;
    variablesForTouchArea[16].maxValue           = 255;
    variablesForTouchArea[16].easyGUICallingPage = GuiStruct_FanPowerPage_0;
    strcpy(variablesForTouchArea[16].variableTitle, "Fan Cooling");
    
    variablesForTouchArea[17].touchAreaIndex     = FAN_POWER_DH_CALIB_EDIT;
    variablesForTouchArea[17].easyGUIVariablePtr = GuiVar_fanPowerDHCalibration;
    variablesForTouchArea[17].maxValue           = 255;
    variablesForTouchArea[17].easyGUICallingPage = GuiStruct_FanPowerPage_0;
    strcpy(variablesForTouchArea[17].variableTitle, "Fan DH Calib");
    
    variablesForTouchArea[18].touchAreaIndex     = FAN_POWER_MINIMUM_EDIT;
    variablesForTouchArea[18].easyGUIVariablePtr = GuiVar_fanPowerMinimum;
    variablesForTouchArea[18].maxValue           = 255;
    variablesForTouchArea[18].easyGUICallingPage = GuiStruct_FanPowerPage_0;
    strcpy(variablesForTouchArea[18].variableTitle, "Fan Minimum");
}


// Private destructor also
NudgeAndDampPageHandler::~NudgeAndDampPageHandler()
{
}


/*
    Tells the caller whether or not the specified touch index is on one of the easyGUI Nudge and Damp pages,
    and therefore needs to be handled by this class.
    
    Args: the touch area index in question
    
    Return code: true if the touch area is 'one of ours', false if not
*/
bool NudgeAndDampPageHandler::TouchAreaIsOnNudgeAndDampPage(int touchAreaIndex)
{
    if((touchAreaIndex >= MIN_COLUMN_OVEN_NUDGE_AND_DAMP_TOUCHINDEX) && (touchAreaIndex <= MAX_COLUMN_OVEN_NUDGE_AND_DAMP_TOUCHINDEX)) {
        return true;
    }

    if((touchAreaIndex >= MIN_COLUMN_DH_NUDGE_AND_DAMP_TOUCHINDEX) && (touchAreaIndex <= MAX_COLUMN_DH_NUDGE_AND_DAMP_TOUCHINDEX)) {
        return true;
    }

    if((touchAreaIndex >= MIN_INJECTOR_NUDGE_AND_DAMP_TOUCHINDEX) && (touchAreaIndex <= MAX_INJECTOR_NUDGE_AND_DAMP_TOUCHINDEX)) {
        return true;
    }

    if((touchAreaIndex >= MIN_DETECTOR_NUDGE_AND_DAMP_TOUCHINDEX) && (touchAreaIndex <= MAX_DETECTOR_NUDGE_AND_DAMP_TOUCHINDEX)) {
        return true;
    }

    if((touchAreaIndex >= MIN_AUXILIARY_NUDGE_AND_DAMP_TOUCHINDEX) && (touchAreaIndex <= MAX_AUXILIARY_NUDGE_AND_DAMP_TOUCHINDEX)) {
        return true;
    }

    if((touchAreaIndex >= MIN_FAN_POWER_TOUCHINDEX) && (touchAreaIndex <= MAX_FAN_POWER_TOUCHINDEX)) {
        return true;
    }

    // 'else'
    return false;
}


/*
    If the specified touch area represents a key or field on one of the easyGUI Nudge and Damp pages,
    this function performs whatever action is appropriate for it. Provided so that the caller
    can (in effect) say "if this is one of yours, deal with it", without needing to know 
    anything about what this class does, or how it handles the touch areas on that easyGUI page.

    Args: the touch area index in question
    
    Returns true if it dealt with the touch area (so the caller need not do anything else 
    with the value), or false if it did not deal with the touch area (implying that it 
    was not a touch area on an easyGUI Nudge and Damp page, and so the caller
    must deal with it some other way).
*/
bool NudgeAndDampPageHandler::DealWithTouch(int touchAreaIndex)
{
    // TODO: if 'touchAreaIndex' is one of the areas handled by this class,
    // deal with it

    // Has the user selected a field to edit - 
    // if so, we will have an entry in our 'variablesForTouchArea' array that corresponds with this touch area
    int indexOfValueToEdit = GetIndexOfValueToEditForTouchArea(touchAreaIndex);
    if(indexOfValueToEdit  != -1) {        
        // User has selected a field to edit
        NumericKeypadPageHandler* numericKeypadPageHandler = NumericKeypadPageHandler::GetInstance(usbDevice, usbHostGC);
        if(numericKeypadPageHandler != NULL) {
            numericKeypadPageHandler->StartEditing(variablesForTouchArea[indexOfValueToEdit].easyGUIVariablePtr);
            numericKeypadPageHandler->SetEasyGUIVariableToEdit(variablesForTouchArea[indexOfValueToEdit].easyGUIVariablePtr);
            numericKeypadPageHandler->SetEasyGUICallingPage(variablesForTouchArea[indexOfValueToEdit].easyGUICallingPage);
            numericKeypadPageHandler->SetEditVariableRange(0, variablesForTouchArea[indexOfValueToEdit].maxValue);
            numericKeypadPageHandler->SetEditVariableName(variablesForTouchArea[indexOfValueToEdit].variableTitle);
            numericKeypadPageHandler->DisplayEasyGUIPage();
        }

        return true;
    } 
        
    
    if(touchAreaIndex == COLUMN_OVEN_NUDGE_AND_DAMP_GET) {
        GetColumnOvenNudgeAndDampFactorsFromGC();
        return true;
    }

    if(touchAreaIndex == COLUMN_DH_NUDGE_AND_DAMP_GET) {
        GetColumnDHNudgeAndDampFactorsFromGC();
        return true;
    }

    if(touchAreaIndex == INJECTOR_NUDGE_AND_DAMP_GET) {
        GetInjectorNudgeAndDampFactorsFromGC();
        return true;
    }

    if(touchAreaIndex == DETECTOR_NUDGE_AND_DAMP_GET) {
        GetDetectorNudgeAndDampFactorsFromGC();
        return true;
    }

    if(touchAreaIndex == AUXILIARY_NUDGE_AND_DAMP_GET) {
        GetAuxiliaryNudgeAndDampFactorsFromGC();
        return true;
    }

    if(touchAreaIndex == FAN_POWER_GET) {
        GetFanPowerValuesFromGC();
        return true;
    }

    if(touchAreaIndex == COLUMN_OVEN_NUDGE_AND_DAMP_SET) {
        SetColumnOvenNudgeAndDampFactorsInGC();
        return true;
    }

    if(touchAreaIndex == COLUMN_DH_NUDGE_AND_DAMP_SET) {
        SetColumnDHNudgeAndDampFactorsInGC();
        return true;
    }

    if(touchAreaIndex == INJECTOR_NUDGE_AND_DAMP_SET) {
        SetInjectorNudgeAndDampFactorsInGC();
        return true;
    }

    if(touchAreaIndex == DETECTOR_NUDGE_AND_DAMP_SET) {
        SetDetectorNudgeAndDampFactorsInGC();
        return true;
    }

    if(touchAreaIndex == AUXILIARY_NUDGE_AND_DAMP_SET) {
        SetAuxiliaryNudgeAndDampFactorsInGC();
        return true;
    }

    if(touchAreaIndex == FAN_POWER_SET) {
        SetFanPowerValuesInGC();
        return true;
    }

    // 'else'...
    
    return false;
}


/*
    Caller is telling us it is about to display one of the easyGUI Nudge and Damp pages,
    and that we should do whatever we have to do to get it ready, before the caller displays it.
    
    No arguments, no return code
*/
void NudgeAndDampPageHandler::DisplayingEasyGUIPage(int pageNumber, bool updateEasyGUIVariables)
{
    // TODO: if 'updateEasyGUIVariables' is true, update the easyGUI variables for that page
    
    if(updateEasyGUIVariables) {
        if(pageNumber == GuiStruct_ColumnOvenNudgeAndDampPage_0) {
            GetColumnOvenNudgeAndDampFactorsFromGC();
        }
        else if(pageNumber == GuiStruct_ColumnDHNudgeAndDampPage_0) {
            GetColumnDHNudgeAndDampFactorsFromGC();
        }
        else if(pageNumber == GuiStruct_InjectorNudgeAndDampPage_0) {
            GetInjectorNudgeAndDampFactorsFromGC();
        }
        else if (pageNumber == GuiStruct_DetectorNudgeAndDampPage_0) {
            GetDetectorNudgeAndDampFactorsFromGC();
        }
        else if (pageNumber == GuiStruct_AuxiliaryNudgeAndDampPage_0) {
            GetAuxiliaryNudgeAndDampFactorsFromGC();
        }
        else if (pageNumber == GuiStruct_FanPowerPage_0) {
            GetFanPowerValuesFromGC();
        }
    }
}


/*
    As the name implies, sends a command to the GC and returns the response.
    
    Args: pointer to a buffer containing the command, as a null-terminated string
          pointer to a buffer to contain the response, as a null-terminated string
          
    No return code (it is up to the caller to examine the response to see whether 
    the command succeeded or failed)
*/
void NudgeAndDampPageHandler::SendCommandToGCAndGetResponse(char* command, char* response)
{
#define USE_GC_UTILS // Testing new class
#ifdef USE_GC_UTILS
    USBHostGCUtilities::SendCommandToGCAndGetResponse(usbDevice, usbHostGC, command, response);
#else
    while(usbHostGC->ExecutingSetDeviceReport()) {}

    usbHostGC->SetDeviceReport(usbDevice, command, response);
//#define DEBUG_PRINT_HERE
#ifdef DEBUG_PRINT_HERE
    char dbg[100];
    sprintf(dbg, "NDPH cmd \"%s\", response \"%s\"", command, response);
    SpecialDebugPrint(dbg, 10, 275);
#endif // DEBUG_PRINT_HERE

#endif // USE_GC_UTILS
}


/*
    Sends a command to the GC for which we expect a response of "DACK" if successful,
    "DNAK" or "EPKT" if failure.
    
    Args: a pointer to the command in question, as a null terminated string
    
    Returns true if the GC responded with "DACK", false for anything else
*/
bool NudgeAndDampPageHandler::SendCommandToGCWithDACKResponse(char *cmd)
{
#define USE_GC_UTILS // Testing new class
#ifdef USE_GC_UTILS
    return USBHostGCUtilities::SendCommandToGCWithDACKResponse(usbDevice, usbHostGC, cmd);
#else
    char response[50];
    SendCommandToGCAndGetResponse(cmd, response);
    // We expect a response like this: "DACK" for success, "DNAK" for failure, "EPKT" for error
    
    return (response[1] == 'A');
#endif // USE_GC_UTILS
}


/*
    Gets a nudge or damp factor from the GC, and put its value into an easyGUI variable.
    It is up to the caller to display the updated variable to the user.
    
    We use the same factor for nudge *and* damp, since the GC returns both values
    in the same format.
    
    Args: a string, set to the command to get the value from the GC
          a pointer to the easyGUI variable to update
          
    Returns true if successful, false if error
*/
bool NudgeAndDampPageHandler::GetNudgeOrDampFactorFromGC(char *cmd, GuiConst_TEXT* easyGUIVariable)
{
    char response[50];
    SendCommandToGCAndGetResponse(cmd, response);

    if(response[0] != 'D') {
        return false;
    }

    if(response[1] != cmd[1]) {
        return false;
    }

    if(response[2] != cmd[2]) {
        return false;
    }

    if(response[3] != cmd[3]) {
        return false;
    }

    // OK so far...
    
    response[8] = '\0'; // Ensure null terminator
    
    // Need to remove leading zeroes, if any
    int factor;
    sscanf(&response[4], "%d", &factor);
    
    sprintf(easyGUIVariable, "%d", factor);
    
    return true;
}


/*
    Gets the current column oven nudge and damp factors from the GC,
    and copies them into the relevant easyGUI variables for display to the user.
    It is up to the caller to (re)display the Column Oven Nudge and Damp easyGUI page.
    
    No arguments, no return code
*/
void NudgeAndDampPageHandler::GetColumnOvenNudgeAndDampFactorsFromGC(void)
{
    GetNudgeOrDampFactorFromGC("GCNU", GuiVar_columnOvenNudgeFactor);
    GetNudgeOrDampFactorFromGC("GCDA", GuiVar_columnOvenDampFactor);
    GetNudgeOrDampFactorFromGC("GRNU", GuiVar_columnOvenRampNudgeFactor);
    GetNudgeOrDampFactorFromGC("GRDA", GuiVar_columnOvenRampDampFactor);
    GetNudgeOrDampFactorFromGC("GCTO", GuiVar_columnOvenTempOffset);
}


/*
    Gets the current directly heated column nudge and damp factors from the GC,
    and copies them into the relevant easyGUI variables for display to the user.
    It is up to the caller to (re)display the Column DH Nudge and Damp easyGUI page.
    
    No arguments, no return code
*/
void NudgeAndDampPageHandler::GetColumnDHNudgeAndDampFactorsFromGC(void)
{
    GetNudgeOrDampFactorFromGC("GHNU", GuiVar_columnDHNudgeFactor);
    GetNudgeOrDampFactorFromGC("GHDA", GuiVar_columnDHDampFactor);
    GetNudgeOrDampFactorFromGC("GHRN", GuiVar_columnDHRampNudgeFactor);
    GetNudgeOrDampFactorFromGC("GHRD", GuiVar_columnDHRampDampFactor);
}


/*
    Gets the current injector nudge and damp factors from the GC,
    and copies them into the relevant easyGUI variables for display to the user.
    It is up to the caller to (re)display the Injector Nudge and Damp easyGUI page.
    
    No arguments, no return code
*/
void NudgeAndDampPageHandler::GetInjectorNudgeAndDampFactorsFromGC(void)
{
    GetNudgeOrDampFactorFromGC("GINU", GuiVar_injectorNudgeFactor);
    GetNudgeOrDampFactorFromGC("GIDA", GuiVar_injectorDampFactor);
}


/*
    Gets the current detector nudge and damp factors from the GC,
    and copies them into the relevant easyGUI variables for display to the user.
    It is up to the caller to (re)display the Detector Nudge and Damp easyGUI page.
    
    No arguments, no return code
*/
void NudgeAndDampPageHandler::GetDetectorNudgeAndDampFactorsFromGC(void)
{
    GetNudgeOrDampFactorFromGC("GDNU", GuiVar_detectorNudgeFactor);
    GetNudgeOrDampFactorFromGC("GDDA", GuiVar_detectorDampFactor);
}


/*
    Gets the current auxiliary nudge and damp factors from the GC,
    and copies them into the relevant easyGUI variables for display to the user.
    It is up to the caller to (re)display the Auxiliary Nudge and Damp easyGUI page.
    
    No arguments, no return code
*/
void NudgeAndDampPageHandler::GetAuxiliaryNudgeAndDampFactorsFromGC(void)
{
    GetNudgeOrDampFactorFromGC("GANU", GuiVar_auxiliaryNudgeFactor);
    GetNudgeOrDampFactorFromGC("GADA", GuiVar_auxiliaryDampFactor);
}


/*
    Gets the current values for each fan power setting from the GC,
    and copies them into the relevant easyGUI variables for display to the user.
    It is up to the caller to (re)display the Fan Power easyGUI page.
    
    No arguments, no return code
*/
void NudgeAndDampPageHandler::GetFanPowerValuesFromGC(void)
{
    // The commands to get the fan power values are very similar to those for the nudge and damp factors
    GetNudgeOrDampFactorFromGC("GFNM", GuiVar_fanPowerNormal);
    GetNudgeOrDampFactorFromGC("GFCL", GuiVar_fanPowerCooling);
    GetNudgeOrDampFactorFromGC("GFCA", GuiVar_fanPowerDHCalibration);
    GetNudgeOrDampFactorFromGC("GFMI", GuiVar_fanPowerMinimum);
}


/*
    The commands to set the various nudge and damp factors are very similar.
    In particular the format of their data values is identical. This function sets the value in the GC 
    using the specified command, using the value in the specified easyGUI variable.
    
    Returns true if successful, false if not.
*/
bool NudgeAndDampPageHandler::SetNudgeOrDampFactorInGC(char *cmd, GuiConst_TEXT* easyGUIVariable)
{
    char buffer[50];
    
    // In the command, the nudge/damp value must have 4 digits, padded if necessary with leading zeroes
    int factor;
    sscanf(easyGUIVariable, "%d", &factor);
    sprintf(buffer, "%s%.4d", cmd, factor); 
    
    return SendCommandToGCWithDACKResponse(buffer);
}


/*
    Sets the current column oven nudge and damp factors in the GC,
    using the values in the relevant easyGUI variables.
    
    No arguments, no return code
*/
void NudgeAndDampPageHandler::SetColumnOvenNudgeAndDampFactorsInGC(void)
{
    SetNudgeOrDampFactorInGC("SCNU", GuiVar_columnOvenNudgeFactor);
    SetNudgeOrDampFactorInGC("SCDA", GuiVar_columnOvenDampFactor);
    SetNudgeOrDampFactorInGC("SRNU", GuiVar_columnOvenRampNudgeFactor);
    SetNudgeOrDampFactorInGC("SRDA", GuiVar_columnOvenRampDampFactor);
    SetNudgeOrDampFactorInGC("SCTO", GuiVar_columnOvenTempOffset);
}


/*
    Sets the current directly heated column nudge and damp factors in the GC,
    using the values in the relevant easyGUI variables.
    
    No arguments, no return code
*/
void NudgeAndDampPageHandler::SetColumnDHNudgeAndDampFactorsInGC(void)
{
    SetNudgeOrDampFactorInGC("SHNU", GuiVar_columnDHNudgeFactor);
    SetNudgeOrDampFactorInGC("SHDA", GuiVar_columnDHDampFactor);
    SetNudgeOrDampFactorInGC("SHRN", GuiVar_columnDHRampNudgeFactor);
    SetNudgeOrDampFactorInGC("SHRD", GuiVar_columnDHRampDampFactor);
}


/*
    Sets the current injector nudge and damp factors in the GC,
    using the values in the relevant easyGUI variables.
    
    No arguments, no return code
*/
void NudgeAndDampPageHandler::SetInjectorNudgeAndDampFactorsInGC(void)
{
    SetNudgeOrDampFactorInGC("SINU", GuiVar_injectorNudgeFactor);
    SetNudgeOrDampFactorInGC("SIDA", GuiVar_injectorDampFactor);
}


/*
    Sets the current detector nudge and damp factors in the GC,
    using the values in the relevant easyGUI variables.
    
    No arguments, no return code
*/
void NudgeAndDampPageHandler::SetDetectorNudgeAndDampFactorsInGC(void)
{
    SetNudgeOrDampFactorInGC("SDNU", GuiVar_detectorNudgeFactor);
    SetNudgeOrDampFactorInGC("SDDA", GuiVar_detectorDampFactor);
}


/*
    Sets the current auxiliary nudge and damp factors in the GC,
    using the values in the relevant easyGUI variables.
    
    No arguments, no return code
*/
void NudgeAndDampPageHandler::SetAuxiliaryNudgeAndDampFactorsInGC(void)
{
    SetNudgeOrDampFactorInGC("SANU", GuiVar_auxiliaryNudgeFactor);
    SetNudgeOrDampFactorInGC("SADA", GuiVar_auxiliaryDampFactor);
}


/*
    Sets the current values for each fan power setting in the GC,
    using the values in the relevant easyGUI variables.
    
    No arguments, no return code
*/
void NudgeAndDampPageHandler::SetFanPowerValuesInGC(void)
{
    // The commands to set the fan power values are very similar to those for the nudge and damp factors
    SetNudgeOrDampFactorInGC("SFNM", GuiVar_fanPowerNormal);
    SetNudgeOrDampFactorInGC("SFCL", GuiVar_fanPowerCooling);
    SetNudgeOrDampFactorInGC("SFCA", GuiVar_fanPowerDHCalibration);
    SetNudgeOrDampFactorInGC("SFMI", GuiVar_fanPowerMinimum);
}


/*
    Tells the caller if the specified touch area corresponds to a value the user can edit
    on one of the easyGUI nudge and damp pages, and if so, which value it is.
    
    Args: the touch area index in question
    
    Returns the index of the corresponding entry in our 'variablesForTouchArea' array,
    or -1 if there is no such entry. It is up to the caller to check for -1
                                     **************************************
*/
int NudgeAndDampPageHandler::GetIndexOfValueToEditForTouchArea(int touchAreaIndex)
{
    for (int index = 0; index < COUNT_OF_VARIABLES_FOR_TOUCH_AREAS; ++index) {
        if(variablesForTouchArea[index].touchAreaIndex == touchAreaIndex) {
            return index;
        }
    }
    
    // 'else' no nudge or damp factor corresponds to the specified touch area
    return -1;
}
