#ifndef NETWORKPARAMETERS_H
#define NETWORKPARAMETERS_H

#include "EthernetInterface.h"
#include "USBHostGC.h"

#include "GuiLib.h"


/*
    A struct to relate the index of an easyGUI touch area on the 'EthernetParametersPage',
    with the easyGUI variable and our own internal variable to which it corresponds
*/
typedef struct structNetworkParametersEasyGUITouchAreaAndVariables {
    int touchAreaIndex;
    GuiConst_TEXT * easyGUIVariablePtr;
    unsigned int * internalVariablePtr;
    unsigned int maxValue;
    GuiConst_INTCOLOR * easyGUIBackgroundColorVariablePtr;
    char variableName[41];
} NetworkParametersEasyGUITouchAreaAndVariables;

/*
    This class handles the IP address, etc, to be used by the system. It deals with the user interface
    (i.e. the easyGUI 'EthernetParametersPage'), as well as storing the values in QSPI memory.
    
    Note that this class is a singleton - we do not need or want there to be more than one instance of it
    (we do not want multiple values for the IP address, etc, and nor will we show 
    more than one easyGUI 'EthernetParametersPage' to the user at the same time).
*/
class NetworkParameters {
public:
    /**
    * Static method to create (if necessary) and retrieve the single NetworkParameters instance
    */
    static NetworkParameters * GetInstance(USBDeviceConnected* newUsbDevice, USBHostGC* newUsbHostGC);
    
    void GetPortNumberAsString(char *portBuffer);
    int GetPortNumber(void) { return portNumber; }

    void GetIPAddressAsString(char *ipAddressBuffer);
    void SetIPAddressFromString(char *ipAddressBuffer);

    void GetSubnetMaskAsString(char *subnetMaskBuffer);
    void SetSubnetMaskFromString(char *subnetMaskBuffer);

    void GetGatewayAddressAsString(char *gatewayAddressBuffer);
    void SetGatewayAddressFromString(char *gatewayAddressBuffer);
    
    bool GetUseDHCP(void) { return useDHCP; }
    
    bool TouchAreaIsNetworkParameter(int touchAreaIndex);

    bool DealWithTouch(int touchAreaIndex);
    
    void DisplayingEasyGUIPage(bool updateEasyGUIVariables);
    
    // Public so TouchCallback(main.cpp) can call it when this page is about to be displayed
    void RestoreRealValues(void);
    
    void SaveRealValues(EthernetInterface *eth);
    
private:
    static NetworkParameters * theNetworkParametersInstance;
    
    USBDeviceConnected* usbDevice;
    USBHostGC* usbHostGC;
    
    static const GuiConst_INTCOLOR inactiveFieldBackgroundColour;
    static const GuiConst_INTCOLOR activeFieldBackgroundColour;
    
    static const unsigned int defaultPortNumber;
    static const unsigned int defaultIPAddress[4];
    static const unsigned int defaultSubnetMask[4];
    static const unsigned int defaultGatewayAddress[4];
    

    // singleton class -> constructor is private
    NetworkParameters(USBDeviceConnected* newUsbDevice, USBHostGC* newUsbHostGC);    
    ~NetworkParameters();

    unsigned int portNumber;
    unsigned int ipAddress[4];
    unsigned int subnetMask[4];
    unsigned int gatewayAddress[4];
    
    // The parameter values we are actually using to talk to the world
    unsigned int realPortNumber;
    unsigned int realIPAddress[4];
    unsigned int realSubnetMask[4];
    unsigned int realGatewayAddress[4];
    
    bool useDHCP;
    
    enum { COUNT_OF_VARIABLES_FOR_TOUCH_AREAS = 13 };
    NetworkParametersEasyGUITouchAreaAndVariables variablesForTouchArea[COUNT_OF_VARIABLES_FOR_TOUCH_AREAS];
    
    int indexOfVariablesCurrentlyBeingEdited;
    
    void GetIPAddressFieldAsString(int fieldIndex, char *ipAddressBuffer, bool wantLeadingSpaces = true);
    void GetSubnetMaskFieldAsString(int fieldIndex, char *subnetMaskBuffer, bool wantLeadingSpaces = true);
    void GetGatewayAddressFieldAsString(int fieldIndex, char *gatewayAddressBuffer, bool wantLeadingSpaces = true);

    void SetPortNumberFromString(char *portBuffer);
    void SetIPAddressFieldFromString(int fieldIndex, char *ipAddressBuffer);
    void SetSubnetMaskFieldFromString(int fieldIndex, char *subnetMaskBuffer);
    void SetGatewayAddressFieldFromString(int fieldIndex, char *gatewayAddressBuffer);
    
    bool DealWithApplyButton(int touchAreaIndex);
    bool DealWithCancelButton(int touchAreaIndex);
    bool DealWithUseDHCPButton(int touchAreaIndex);
    
    bool DealWithIPEditButton(int touchAreaIndex);
    bool DealWithMaskEditButton(int touchAreaIndex);
    bool DealWithGatewayEditButton(int touchAreaIndex);
    
    void RestoreInitialValues(void);

    void UpdateEasyGUIPage(void);

    void SaveToQSPISettings(void);
    void ReadFromQSPISettings(void);
        
    int GetIndexOfVariablesToEditForTouchArea(int touchAreaIndex);
    
    void UpdateInternalValuesFromEasyGUIVariables(void);
    void UpdateEasyGUIVariablesFromInternalValues(void);
    
    void SetAllFieldBackgroundColoursToInactive(void);
    void SetAllFieldBackgroundColoursToWhite(void);
    
};

#endif // NETWORKPARAMETERS_H

