#ifndef GETGCSTATUSLOOP_H
#define GETGCSTATUSLOOP_H

#include "USBHostGC.h"
#include "GuiLib.h"

#include "GCComponentStatusEnums.h"
#include "GCComponentStatusColorArea.h"
#include "GCComponentTypeEnums.h"
#include "GCIgnitionStateEnum.h"

#include "GCStateAndFaultCodes.h"
#include "TouchPanelPageSelector.h"
#include "EasyGUITouchAreaIndices.h"
#include "SimplifiedTouchListener.h"
#include "EthernetHandler.h"
#include "ProgressBar.h"
#include "QSPIBitmap.h"
#include "MethodRampData.h"

#include "EthernetInterface.h"

#include "GuiLibGraph.h"


/*
    Together with functions in main.cpp - especially TouchCallback - this class does most of the work
    of interfacing with and controlling the GC. After the appropriate setting up and initialisation,
    the main() function calls this class's member function MainLoopWithEthernet - which never returns.
    
    Note that GetGCStatusLoop is a singleton - we do not need or want there to be more than one instance of it
    (there is only one GC, and only one LPC4088). On creation, the instance is passed pointers to the USBHostGC instance
    and the USB device corresponding to the GC.
*/
class GetGCStatusLoop {
public:
    /**
    * Static method to create (if necessary) and retrieve the single GetGCStatusLoop instance
    */
    static GetGCStatusLoop * GetInstance(USBDeviceConnected* newUsbDevice, USBHostGC* newUsbHostGC);

    /*
    * Static method to return the current instance, if it exists. If not, this function
    * will return NULL - it will *not* create a new instance
    */
    static GetGCStatusLoop * GetInstance(void);

    GuiConst_INT16U GetCurrentPage(void);
    void SetCurrentPage(GuiConst_INT16U newCurrentPage);
    
    void SetHomePageGCComponentStatusColorAreas(HomePageGCComponentStatusColorAreas* newColorAreas);
    
    void SetSingleGCComponentPageStatusColorAreas(SingleGCComponentPageStatusColorAreas* newColorAreas);
    
    void UpdateHomePageGCComponentStatusColorAreas(void);
    
    void UpdateSingleGCComponentPageStatusColorArea(GCComponent component);
    
    void UpdateAllGCComponentStatusColorAreas(void);
    
    void SetupAllEasyGUIVariables(void);
    
    void ExitedGCStandbyMode(void);
    
    void UpdateGCMethodRunningProfiles(void);

    // Need to call this from DisplayEasyGuiStructure function in main.cpp
    void DisplayInjectorTempProfilePageGraph(void);

    // Need to call this from DisplayEasyGuiStructure function in main.cpp
    void DisplayGasFlowProfilePageGraph(void);

    // Need to call this from DisplayEasyGuiStructure function in main.cpp
    void DisplayColumnTempProfilePageGraph(ColumnType columnType);
    
    void ForceUpdateOfGasCalibrationPage(void);
    
    void ForceUpdateOfColumnDHAutoCalibrationPage(void);
    void ForceUpdateOfColumnDHManualCalibrationPage(void);
    void ForceUpdateOfColumnDHSensorCalibrationPage(void);
    void ForceUpdateOfColumnDHPSUDACPage(void);
    
    void ForceUpdateOfGasBackPressureDACPage(void);
    void ForceUpdateOfGasChannelDACAndADCPage(void);
    
    void SetGCIsRunning(void);
    void ClearGCIsRunning(void);
    bool GetGCIsRunningFlag(void);
    
    bool UpdateMethodRunTimeEasyGUIVariables(bool runHasCompleted);
    void UpdateAndDisplayRunningPage1ProgressBar(bool runHasCompleted);
    void SetRunningPage1ProgressBarToZero(void);
    
    void UpdateColumnStatusEasyGUIVariable(void);
    void UpdateInjectorStatusEasyGUIVariable(void);
    
    static void TimerCallback(void const * argument);
    
    void ForceRestart(void);
    
    void RedisplayColumnCalibrationPage(void);
    
    void RedisplayColumnPage(void);
    
    static void DisplayColumnLockAndReleaseButtons(void);
    
    void SetQSPIBitmaps(QSPIBitmaps* ptrToQSPIBitmaps);
    
    void SetupTemperatureWhileRunningEasyGUIVariables(void);
    
    void SetupGCNotReadyStateEasyGUIVariable(void);
    
    void ShowColumnMethodPageScrollButtonsIfNecessary(void);
    void ScrollColumnMethodRampsUpIfPossible(void);
    void ScrollColumnMethodRampsDownIfPossible(void);
    
    void ShowInjectorMethodPageScrollButtonsIfNecessary(void);
    void ScrollInjectorMethodRampsUpIfPossible(void);
    void ScrollInjectorMethodRampsDownIfPossible(void);
    
    void ShowGasMethodPageScrollButtonsIfNecessary(void);
    void ScrollGasMethodRampsUpIfPossible(void);
    void ScrollGasMethodRampsDownIfPossible(void);
    
    static const char *GetDegCUnitsWithSpace(void);
    static const char *GetDegCUnitsWithoutSpace(void);


    /* ** This never returns ** */
    void MainLoopWithEthernet(DMBoard* board); 
    
private:
    static GetGCStatusLoop * theGetGCStatusLoop;

    // singleton class -> constructor is private
    GetGCStatusLoop(USBDeviceConnected* newUsbDevice, USBHostGC* newUsbHostGC);    
    ~GetGCStatusLoop();

    USBDeviceConnected* usbDevice;
    USBHostGC* usbHostGC;
    
    GuiConst_INT16U currentPage; // This is the easyGUI 'Structure' value (see GuiStruct.h)
    
    ProgressBar* runningPage1ProgressBar;
    
    GuiLibGraph* runningColumnPageGraph; // Graph index 0
    GuiLibGraphDataSet* runningColumnPageGraphCompleteProfileDataSet; // Line representing the complete temperature profile from start to finish
    // We do not display the above data set in the graph - we use it to generate the data sets below. 
    // The data set numbers correspond with those set for the graph in easyGUI (as do the colours).
    GuiLibGraphDataSet* runningColumnPageGraphDataSet0; // Line representing the temperature profile from 'now' to finish (green)
    GuiLibGraphDataSet* runningColumnPageGraphDataSet1; // Bars representing the temperature profile from 'now' to finish (green)
    GuiLibGraphDataSet* runningColumnPageGraphDataSet2; // Line representing the temperature profile from start to 'now' (red)
    GuiLibGraphDataSet* runningColumnPageGraphDataSet3; // Bars representing the temperature profile from start to 'now' (red)
    GuiLibGraphDataSet* runningColumnPageGraphDataSet4; // Dot representing the temperature 'now' (red)
    float previousColumnMethodRunTime;
    bool columnMethodFinished;
    
    GuiLibGraph* runningGasPageGraph; // Graph index 1
    GuiLibGraphDataSet* runningGasPageGraphCompleteProfileDataSet; // Line representing the complete flow profile from start to finish
    // We do not display the above data set in the graph - we use it to generate the data sets below. 
    // The data set numbers correspond with those set for the graph in easyGUI.
    GuiLibGraphDataSet* runningGasPageGraphDataSet0; // Line representing the flow profile from 'now' to finish (green)
    GuiLibGraphDataSet* runningGasPageGraphDataSet1; // Bars representing the flow profile from 'now' to finish (green)
    GuiLibGraphDataSet* runningGasPageGraphDataSet2; // Line representing the flow profile from start to 'now' (blue)
    GuiLibGraphDataSet* runningGasPageGraphDataSet3; // Bars representing the flow profile from start to 'now' (blue)
    GuiLibGraphDataSet* runningGasPageGraphDataSet4; // Dot representing the temperature 'now' (blue)
    float previousGasMethodRunTime;
    bool gasMethodFinished;
    
    GuiLibGraph* injectorTempProfilePageGraph; // Graph index 2
    GuiLibGraphDataSet* injectorTempProfilePageGraphCompleteProfileDataSet; // Line representing the complete temperature profile from start to finish
    // We do not display the above data set in the graph - we use it to generate the data sets below. 
    // The data set numbers correspond with those set for the graph in easyGUI (as do the colours).
    GuiLibGraphDataSet* injectorTempProfilePageGraphDataSet0; // Line representing the temperature profile from 'now' to finish (green)
    GuiLibGraphDataSet* injectorTempProfilePageGraphDataSet1; // Bars representing the temperature profile from 'now' to finish (green)
    GuiLibGraphDataSet* injectorTempProfilePageGraphDataSet2; // Line representing the temperature profile from start to 'now' (red)
    GuiLibGraphDataSet* injectorTempProfilePageGraphDataSet3; // Bars representing the temperature profile from start to 'now' (red)
    GuiLibGraphDataSet* injectorTempProfilePageGraphDataSet4; // Dot representing the temperature 'now' (red)
    
    GuiLibGraph* gasFlowProfilePageGraph;  // Graph index 3
    GuiLibGraphDataSet* gasFlowProfilePageGraphCompleteProfileDataSet; // Line representing the complete flow profile from start to finish
    // We do not display the above data set in the graph - we use it to generate the data sets below. 
    // The data set numbers correspond with those set for the graph in easyGUI.
    GuiLibGraphDataSet* gasFlowProfilePageGraphDataSet0; // Line representing the flow profile from 'now' to finish (green)
    GuiLibGraphDataSet* gasFlowProfilePageGraphDataSet1; // Bars representing the flow profile from 'now' to finish (green)
    GuiLibGraphDataSet* gasFlowProfilePageGraphDataSet2; // Line representing the flow profile from start to 'now' (blue)
    GuiLibGraphDataSet* gasFlowProfilePageGraphDataSet3; // Bars representing the flow profile from start to 'now' (blue)
    GuiLibGraphDataSet* gasFlowProfilePageGraphDataSet4; // Dot representing the temperature 'now' (blue)
    
    GuiLibGraph* columnTempProfilePageGraph; // Graph index 4
    GuiLibGraphDataSet* columnTempProfilePageGraphCompleteProfileDataSet; // Line representing the complete temperature profile from start to finish
    // We do not display the above data set in the graph - we use it to generate the data sets below. 
    // The data set numbers correspond with those set for the graph in easyGUI (as do the colours).
    GuiLibGraphDataSet* columnTempProfilePageGraphDataSet0; // Line representing the temperature profile from 'now' to finish (green)
    GuiLibGraphDataSet* columnTempProfilePageGraphDataSet1; // Bars representing the temperature profile from 'now' to finish (green)
    GuiLibGraphDataSet* columnTempProfilePageGraphDataSet2; // Line representing the temperature profile from start to 'now' (red)
    GuiLibGraphDataSet* columnTempProfilePageGraphDataSet3; // Bars representing the temperature profile from start to 'now' (red)
    GuiLibGraphDataSet* columnTempProfilePageGraphDataSet4; // Dot representing the temperature 'now' (red)
    
    GuiLibGraph* runningInjectorPageGraph; // Graph index 6
    GuiLibGraphDataSet* runningInjectorPageGraphCompleteProfileDataSet; // Line representing the complete injector temperature profile from start to finish
    // We do not display the above data set in the graph - we use it to generate the data sets below. 
    // The data set numbers correspond with those set for the graph in easyGUI.
    GuiLibGraphDataSet* runningInjectorPageGraphDataSet0; // Line representing the injector temperature profile from 'now' to finish (green)
    GuiLibGraphDataSet* runningInjectorPageGraphDataSet1; // Bars representing the injector temperature profile from 'now' to finish (green)
    GuiLibGraphDataSet* runningInjectorPageGraphDataSet2; // Line representing the injector temperature profile from start to 'now' (yellow)
    GuiLibGraphDataSet* runningInjectorPageGraphDataSet3; // Bars representing the injector temperature profile from start to 'now' (yellow)
    GuiLibGraphDataSet* runningInjectorPageGraphDataSet4; // Dot representing the temperature 'now' (yellow)
    float previousInjectorMethodRunTime;
    bool injectorMethodFinished;
    
    void RedrawSingleEasyGUIVariableOnComponentPage(GuiConst_INT16S X, GuiConst_INT16S Y, void *varPtr, GuiConst_INT8U alignment, GCComponent gcComponent);
    void RedrawSingleEasyGUIVariableOnHomePage(GuiConst_INT16S X, GuiConst_INT16S Y, void *varPtr, GuiConst_INT8U alignment);
    
    void DrawColumnTemperatureVariableOnHomePage(void);
    void DrawInjectorTemperatureVariableOnHomePage(void);
    void DrawDetectorTemperatureVariableOnHomePage(void);
    void DrawColumnTargetTemperatureVariableOnHomePage(void);
    void DrawInjectorTargetTemperatureVariableOnHomePage(void);
    void DrawDetectorTargetTemperatureVariableOnHomePage(void);
    void DrawGasPressureVariableOnHomePage(void);
    void DrawGasTargetPressureVariableOnHomePage(void);
    void DrawGCRealTimeClockVariableOnHomePage(void);
    
    void DrawDetectorTemperatureVariableOnDetectorPage(void);
    void DrawDetectorTargetTemperatureVariableOnDetectorPage(void);

    void UpdateGasTargetPressuresOnHomePage(char *buffer);

    void DisplayHomePageData(bool mustUpdateDisplay);
    void DisplayColumnPageData(bool mustUpdateDisplay, ColumnType columnType, int pageNumber);
    void DisplayColumnInformationPageData(bool mustUpdateDisplay, ColumnType columnType, int pageNumber);
    void DisplayColumnMethodPageData(bool mustUpdateDisplay);
    void DisplayColumnTempProfilePageData(bool mustUpdateDisplay, ColumnType columnType, int pageNumber);
    void DisplayColumnDHAutoCalibrationPageData(bool mustUpdateDisplay);
    void DisplayColumnDHManualCalibrationPageData(bool mustUpdateDisplay);
    void DisplayColumnDHSensorCalibrationPageData(bool mustUpdateDisplay);
    void DisplayColumnDHPSUDACPageData(bool mustUpdateDisplay);
    void DisplayInjectorPageData(bool mustUpdateDisplay);
    void DisplayInjectorMethodPageData(bool mustUpdateDisplay);
    void DisplayInjectorTempProfilePageData(bool mustUpdateDisplay);
    void DisplayDetectorPageData(bool mustUpdateDisplay, DetectorType detectorType, int pageNumber);
    void DisplayGasInformationPageData(bool mustUpdateDisplay);
    void DisplayGasMethodPageData(bool mustUpdateDisplay);
    void DisplayGasFlowProfilePageData(bool mustUpdateDisplay);
    void DisplayGasCalibrationPageData(bool mustUpdateDisplay);
    void DisplayGasBackPressureDACPageData(bool mustUpdateDisplay);
    void DisplayGasChannelDACAndADCPageData(bool mustUpdateDisplay);

    void DisplaySimplePageData(int pageNumber, bool mustUpdateDisplay);
    void DisplayColumnOvenNudgeAndDampPageData(bool mustUpdateDisplay);
    void DisplayColumnDHNudgeAndDampPageData(bool mustUpdateDisplay);
    void DisplayInjectorNudgeAndDampPageData(bool mustUpdateDisplay);
    void DisplayDetectorNudgeAndDampPageData(bool mustUpdateDisplay);
    void DisplayAuxiliaryNudgeAndDampPageData(bool mustUpdateDisplay);
    void DisplayFanPowerPageData(bool mustUpdateDisplay);
    void DisplayDebugCommandsPageData(bool mustUpdateDisplay);

    void DisplayRunningPageData(bool mustUpdateDisplay, bool runHasCompleted);
    void DisplaySettingsPageData(bool mustUpdateDisplay);
    void DisplayRunningColumnPageData(bool mustUpdateDisplay, bool runHasCompleted);
    void DisplayRunningDetectorPageData(bool mustUpdateDisplay);
    void DisplayRunningGasPageData(bool mustUpdateDisplay, bool runHasCompleted);
    void DisplayRunningInjectorPageData(bool mustUpdateDisplay);
    void DisplayRunningInjectorProfilePageData(bool mustUpdateDisplay, bool runHasCompleted);
    
    void DisplayServicingRequiredPage(void);
    void DisplayServicingPage(void);
    
    void SetupServicingPageEasyGUIVariables(void);
    
    void DisplayCurrentPageData(bool mustUpdateDisplay);
    
    void DisplayEthernetConnectionPage(void);
    
    void DisplayFailedToFindBitmapsPage(void);

    void DisplayDownloadingMethodPage(void);
    void UndisplayDownloadingMethodPage(void);
    
    void CreateGuiLibGraphsAndDataSets(void);

    void SetupColumnAndInjectorAndGasProfileData(void);

    void SetupColumnTempProfilePageGraphDataFromGC(void);
    void SetupInjectorTempProfilePageGraphDataFromGC(void);
    void SetupGasFlowProfilePageGraphDataFromGC(void);
    
    void SetupColumnTempProfilePageXAxisLabel(TimeUnit timeUnit);
    void SetupInjectorTempProfilePageXAxisLabel(TimeUnit timeUnit);
    void SetupGasFlowProfilePageXAxisLabel(TimeUnit timeUnit);

    void SetupRunningColumnPageGraphCompleteProfileDataSetFromGC(void);
    bool SetupRunningColumnPageGraphPartialDataSetsToMatchCurrentRunTime(bool runHasCompleted);
    
    void SetupRunningGasPageGraphCompleteProfileDataSetFromGC(void);
    bool SetupRunningGasPageGraphPartialDataSetsToMatchCurrentRunTime(bool runHasCompleted);
    
    void SetupRunningInjectorPageGraphCompleteProfileDataSetFromGC(void);
    bool SetupRunningInjectorPageGraphPartialDataSetsToMatchCurrentRunTime(bool runHasCompleted);
    
    void GetComponentTemperature(char *cmd, char *temp, bool wantPrefix, bool wantDegSuffix, bool oneTenthDegree = true);
    void GetComponentTemperature(char *cmd, float *temp, bool oneTenthDegree = true);
    
    void GetColumnTemperature(char *temp, bool wantPrefix, bool wantSuffix);
    void GetColumnTemperature(char *temp, bool wantPrefix);
    void GetColumnTemperature(float *temp);
    
    void GetDirectlyHeatedColumnTemperature(char *temp, bool wantPrefix);
    void GetDirectlyHeatedColumnTemperature(float *temp);
    
    void GetColumnTargetTemperature(char *temp, const char *format);
    
    void GetDetectorTemperature(char *temp);
    void GetDetectorTemperature(float *temp);
    void GetDetectorTargetTemperature(char *temp, const char *format);
    
    void GetInjectorTemperature(char *temp, bool wantPrefix);
    void GetInjectorTemperature(float *temp);
    void GetInjectorTargetTemperature(char *temp, const char *format);
    
    void GetPressure(char *cmd, char *press, bool wantPrefix, bool wantUnits);

    void GetGasPressure(char *press, bool wantPrefix, bool wantUnits);
    void GetGasPressure(float *press);
    void GetGasPulsedPressure(char *pulsedPress);
    void GetCurrentGasPressure(char *press, bool wantPrefix, bool wantUnits);
    
    bool GetGasControlMode(char *mode, bool wantFullPrefix, bool wantAnyPrefix);
    void GetCarrierGasType(char *type);
    void GetGasFilterDateChanged(char *date);
    
    void GetColumnMaxTemperature(char *maxTemp, bool wantFullPrefix, bool wantAnyPrefix, bool wantUnits = false);
    void GetColumnMaxTemperature(float *temp);
    void GetColumnType(char *type);
    ColumnType GetColumnType(void);
    
    void GetInjectionMode(char *mode);
    void GetInjectorType(char *mode, bool wantPrefix);
    void GetInjectorSplitTime(char *splitTime);
    void GetInjectionCountSinceLinerChanged(char *injCount);
    void GetInjectionCountSinceSeptaChanged(char *injCount);
    void GetInjectionCountSinceOringChanged(char *injCount);
    
    void GetInjectionCountSinceColumnInstalled(char *injCount);
    void GetColumnLength(char *columnLengthBuffer);
    void GetColumnInnerDiameter(char *columnIDBuffer);
    void GetColumnOuterDiameter(char *columnODBuffer);

    void GetTotalFlow(char *totalFlow);
    void GetTotalFlow(float *totalFlow);
    void GetColumnFlow(char *columnFlow);
    void GetColumnFlow(float *columnFlow);
    
    void GetSplitFlow(float *splitFlow);
    void GetSplitFlow(char *splitFlow);
    void GetSplitRatio(float *splitRatio);
    void GetSplitRatio(char *splitRatio);

    void GetDetectorTypeOld(char *type, bool wantFullPrefix);
    void GetDetectorType(char *type, bool wantFullPrefix, bool wantPrefix);
    void GetDetectorRange(char *range);
    void GetDetectorIgnitionState(char *state);
    IgnitionState GetDetectorIgnitionState(void);
    
    void GetTCDDetectorFilamentPolarity(char *polarity);
    void GetTCDDetectorFilamentTemperature(char *temp);
    void GetTCDDetectorRange(char *range);
    
    void GetECDDetectorCurrent(char *current);
    
    void GetFPDDetectorRange(char *range);
    void GetFPDDetectorSensitivity(char *sensitivity);
    
    void GetFlowRate(char *cmd, char *rate);
    void GetFuelFlowRate(char *rate);
    void GetAirFlowRate(char *rate);
    
    bool ExecuteCommandWithDACKResponse(char *cmd);

    void GetGCSoftwareVersion(char *version);
    void GetActuatorSoftwareVersion(char *version);
    void GetRunTime(char *time);
    void GetRunTime(float *time);
    void GetSerialNumber(char *serialNumber);
    void GetGCRealTimeClockTime(char *clockTime);
    
    void GetInitialHoldTime(char *time);
    
    float GetColumnMethodTimeRemaining(void);

    bool pageJustChanged;
    bool displayingData;
    bool needToUpdateProfileGraphs;
    
    bool IsSendMethodCommand(char *gcCommand);

    bool sendingMethod;
    
    bool gcInStandbyMode;
    
    bool GCIsInStandbyMode(void);
    void DisplayStandbyModePage(void);
    bool ExitGCStandbyMode(void);

    bool ClearGCErrors(void);
    
    bool AbortGCRun(void);

    HomePageGCComponentStatusColorAreas* homePageGCComponentStatusColorAreas;
    SingleGCComponentPageStatusColorAreas* singleGCComponentPageStatusColorAreas;
        
    void SetGCDeviceReport(char *cmd, char *response);
    
    int GetGCStatusOrFaultCode(char *cmd);
    int GetGCStatus(void);
    int GetGCFaultCode(void);
    void GetGCStateAsInfoString(int gcStateCode, int gcFaultCode, char *statusString);
    bool GCHasFaulted(int gcStatus, char* statusString);
    bool GCHasFaulted(void);
    void DisplayGCInFaultStatePage(bool clearDisplay);    
    void DisplayRunCompletePage(void);
    void DisplayRunAbortedPage(void);
    
    bool IsHeatOn(void);
    
    int GetInstrumentStatus(void);
    bool GCIsRunning(void);
    bool realGCIsRunning;
    bool GCIsStabilising(void);
    
    GCComponentStatus GetComponentStatus(GCComponent component);
    void GetComponentStatusString(GCComponent component, char *buff);
    
    bool HomePageGCComponentStatusesHaveChanged(void);

    bool SinglePageGCComponentStatusHasChanged(GCComponent component);
    bool SinglePageGCComponentStatusHasChanged(GCComponent component, GCComponentStatus lastStatusDisplayed);
    
    GCComponentStatus lastColumnStatusDisplayedOnHomePage;
    GCComponentStatus lastInjectorStatusDisplayedOnHomePage;
    GCComponentStatus lastDetectorStatusDisplayedOnHomePage;
    GCComponentStatus lastGasStatusDisplayedOnHomePage;

    GCComponentStatus lastColumnStatusDisplayedOnColumnPage;
    GCComponentStatus lastInjectorStatusDisplayedOnInjectorPage;
    GCComponentStatus lastDetectorStatusDisplayedOnDetectorPage;
    GCComponentStatus lastGasStatusDisplayedOnGasInformationPage;
    
    bool SuccessfulRunCommand(char* gcCommand, char* gcResponse);
    bool SuccessfulAbortCommand(char* gcCommand, char* gcResponse);

    void ShortEthernetThreadWait(Thread* ethernetThread, int debugValue);
    void ShortTouchThreadWait(SimplifiedTouchListener* stl, int debugValue);
    void ShortThreadWait(SimplifiedTouchListener* stl, Thread* ethernetThread, int debugValue);
    
    void PollGC(SimplifiedTouchListener* stl, Thread* ethernetThread);

    void HandleEthernetMessage(Thread* ethernetThread, int debugValue);
    bool handlingEthernetMessage;
    
    void HandleTouchEvent(short touchX, short touchY, short touchZ, int debugValue);
    bool handlingTouchEvent;
    
    bool PageIncludesComponentStatus(int pageNumber);
    
    void HandleUserInterfaceAndEthernet(SimplifiedTouchListener* stl, Thread* ethernetThread);
    
    GCStateAndFaultCodes gcStateAndFaultCodes;
    
    bool gotAtLeastOneTimeout;
    
    bool runWasAborted;
    
    int  ethernetPort;
    char ethernetIP[20];
    char ethernetMask[20];
    char ethernetGateway[20];
    bool useDHCPForEthernet;

    bool restartRequired;
    
    GCStateSimplified lastSimplifiedGCState;
    
    Timer gcResponseTimer;
    Timer touchTimer;
    
    QSPIBitmaps* qspiBitmaps;

    ColumnType runningColumnType;

    // Same for all three easyGUI pages displaying method data - column, method and gas
    void ShowMethodPageScrollButtons(bool needToScrollRampData);
    void ShowMethodPageScrollButtonsIfNecessary(MethodRampData *methodRampData);

    ColumnMethodRampData* columnMethodRampData;
    void UpdateColumnMethodPageData(void);
    unsigned int columnMethodPageScrollIndex; // It would never be valid for this to be less than zero
    unsigned int previousColumnMethodPageScrollIndex;
    
    InjectorMethodRampData* injectorMethodRampData;
    void UpdateInjectorMethodPageData(void);
    unsigned int injectorMethodPageScrollIndex; // It would never be valid for this to be less than zero
    unsigned int previousInjectorMethodPageScrollIndex;
    
    GasMethodRampData* gasMethodRampData;
    void UpdateGasMethodPageData(void);
    unsigned int gasMethodPageScrollIndex; // It would never be valid for this to be less than zero
    unsigned int previousGasMethodPageScrollIndex;
    
    // RTosTimer stuff
    RtosTimer* rtosTimer;
    int lastTouchEventTickCount;
    short lastTouchEventX;
    short lastTouchEventY;
    static osThreadId timerCallbackThreadToSignal;
    static const int waitTimeMs;
    static const int shortWaitTimeMs;
    static int timerTickCount;
    static const uint32_t timerIntervalMilliSec;
    static const uint32_t minTimerTicksBetweenTouchEvents;
    
    static const float graphBarIntervalMinutes;
    static const float graphBarIntervalSeconds;
    
    static const float methodTimeUnitsThreshold;
    
    static const int   defaultEthernetPort;
    static const char* defaultEthernetIP;
    static const char* defaultEthernetMask;
    static const char* defaultEthernetGateway;
    
    static const char* startDHColumnCalibration;
    static const char* exitDHColumnCalibration;

    static const char degSymbol;
    static const char stringFormatdegCUnits[6];
        
    static void DisplayText(char *text, short X, short Y, GuiConst_INT8U alignment, GuiConst_INT16U fontNo, GuiConst_INTCOLOR foreColor);
};

#endif // GETGCSTATUSLOOP_H
