#ifndef SERVICEINTERVAL_H
#define SERVICEINTERVAL_H

#include "mbed.h"
#include "DMBoard.h"

#include "USBHostGC.h"

/*
    These classes do what the name implies - handle service intervals. These are measured
    either in instrument cycles (i.e. runs), or time - in the latter case, the interval
    is always twelve months.
    
    Each ServiceInterval object deals with one (and only one) service interval, allowing the caller
    to specify its description (a text string).
    
    Note that the caller must 'poll' the ServiceInterval object to find out whether or not 
    it has expired - it does not raise an interrupt, or call a callback function,
    or do anything unsolicited.
*/

//#define SERVICE_INTERVALS_ACTIVE // #define this to activate the ServiceInterval code in the whole application

class ServiceInterval {
public:
    ServiceInterval(char *intervalDescription, int touchAreaIndex);
    ~ServiceInterval();

    int GetDescriptionLength(void);
    void GetDescription(char *buffer);
    
    virtual void Start(void) = 0;
    
    virtual void InstrumentHasCycled(void) = 0;
    
    virtual bool HasExpired(void) = 0;
    
    bool IntervalHasStarted(void) { return intervalHasStarted; }
    
    virtual void SaveToQSPISettings(void) = 0;
    virtual void ReadFromQSPISettings(void) = 0;
    
    static void SetupAllServiceIntervals(void);
    static void StartAllServiceIntervals(void);
    static int GetServiceIntervalCount(void) { return SERVICE_INTERVAL_COUNT; }
    static ServiceInterval* GetServiceInterval(int serviceIntervalIndex);
    static void TellAllServiceIntervalsInstrumentHasCycled(void);
    static bool AtLeastOneServiceIntervalHasExpired(void);
    static ServiceInterval* GetNextExpiredServiceInterval(ServiceInterval* thisExpiredInterval);
    static void SaveAllServiceIntervalsToQSPISettings(void);
    static void ReadAllServiceIntervalsFromQSPISettings(void);
    static bool IsServicedTouchArea(int touchAreaIndex);
    static bool DealWithServicedTouchArea(int touchAreaIndex);
    

private:
    char *description;
    
    enum { SERVICE_INTERVAL_COUNT = 6 };
    static ServiceInterval* serviceIntervalArray[SERVICE_INTERVAL_COUNT];
    
    int servicedTouchArea; // The easyGUI touch area that specifies "this component has now been serviced"
                                    

protected:
    bool intervalHasStarted;
};

class CyclesServiceInterval : public ServiceInterval {
public:
    CyclesServiceInterval(char *intervalDescription, int touchAreaIndex);

    void SetDurationInNumberOfInstrumentCycles(int newDurationInNumberOfInstrumentCycles);

    virtual void InstrumentHasCycled(void);
    
    virtual void Start(void);
    
    virtual bool HasExpired(void) { return intervalHasExpired; }
    
    virtual void SaveToQSPISettings(void);
    virtual void ReadFromQSPISettings(void);

private:
    int durationInNumberOfInstrumentCycles;
    int countOfRemainingCycles;

    bool intervalHasExpired;
};

class TwelveMonthServiceInterval : public ServiceInterval {
public:
    TwelveMonthServiceInterval(char *intervalDescription, int touchAreaIndex);

    virtual void Start(void);
    
    virtual void InstrumentHasCycled(void);
    
    virtual bool HasExpired(void);
    
    virtual void SaveToQSPISettings(void);
    virtual void ReadFromQSPISettings(void);

private:
    struct tm startTime;
    
    static void GetCurrentLocalTime(struct tm *currentLocalTime);
};

#endif  // SERVICEINTERVAL_H

