#ifndef GCSTATUSANDFAULTCODES_H
#define GCSTATUSANDFAULTCODES_H

#include "GuiLib.h"

/*
    Classes to deal with the GC state and fault codes, making them simpler to handle for the rest of this application.
    
    The initial state and fault codes below are taken from the Nemesis 2 GC Control Protocol,
    version 1.0, dated 01/10/15. Note that the fault codes include those marked 'not used'.
    
    The codes marked 'Version 102' are taken from the 500 Series GC Comunications Protocol,
    Version 1.0, dated 11/02/16. Until further notice, they should be used with all versions
    from 1.02 onwards (i.e. those that return "DWHO0102" or later in response to "QWHO").
    
*/

// To use the 'Version 102' codes, uncomment this line:
#define USE_VERSION_102
// Comment out this line to reinstate the original codes.

enum GC_STATE {
    GC_STATE_IDLE = 0,
    GC_STATE_NOT_READY = 1,
    GC_STATE_EQUILIBRATING = 2,
    
    // Start of 'GC is running' status codes
    GC_STATE_RUNNING_INITIAL_TIME = 4,
    GC_STATE_RAMPING_SEGMENT_1 = 5,
    GC_STATE_AT_UPPER_TEMP_SEGMENT_1 = 6,
    GC_STATE_RAMPING_SEGMENT_2 = 7,
    GC_STATE_AT_UPPER_TEMP_SEGMENT_2 = 8,
    GC_STATE_RAMPING_SEGMENT_3 = 9,
    GC_STATE_AT_UPPER_TEMP_SEGMENT_3 = 10,
    GC_STATE_RAMPING_SEGMENT_4 = 11,
    GC_STATE_AT_UPPER_TEMP_SEGMENT_4 = 12,
    GC_STATE_RAMPING_SEGMENT_5 = 13,
    GC_STATE_AT_UPPER_TEMP_SEGMENT_5 = 14,    
    
    GC_STATE_HOLDING = 15,
    GC_STATE_ABORTING = 16,
    GC_STATE_COOLING = 17,
    // End of 'GC is running' status codes

    // Record the above
    GC_STATE_RUNNING_MINIMUM = GC_STATE_RUNNING_INITIAL_TIME,
    GC_STATE_RUNNING_MAXIMUM = GC_STATE_COOLING,
    
    GC_STATE_READY_TO_PRE_RUN = 31,
    GC_STATE_PRE_RUNNING = 32,
    GC_STATE_READY_TO_RUN = 33,        

    GC_STATE_FAULTED = 99,    
};

enum GC_STATE_102 {
    GC_STATE_102_METHOD_IDLE = 0,
    GC_STATE_102_METHOD_EQUILIBRATING = 10,
    GC_STATE_102_METHOD_STABILISING = 20,
    GC_STATE_102_METHOD_WAIT_READY_IO = 30,
    GC_STATE_102_METHOD_READY_TO_RUN = 40,
    GC_STATE_102_METHOD_INIT_TIME = 50,
    
    GC_STATE_102_METHOD_RAMPING_1  = 60,
    GC_STATE_102_METHOD_RAMPING_2  = 61,
    GC_STATE_102_METHOD_RAMPING_3  = 62,
    GC_STATE_102_METHOD_RAMPING_4  = 63,
    GC_STATE_102_METHOD_RAMPING_5  = 64,
    GC_STATE_102_METHOD_RAMPING_6  = 65,
    GC_STATE_102_METHOD_RAMPING_7  = 66,
    GC_STATE_102_METHOD_RAMPING_8  = 67,
    GC_STATE_102_METHOD_RAMPING_9  = 68,
    GC_STATE_102_METHOD_RAMPING_10 = 69,
    
    GC_STATE_102_METHOD_HOLDING_1  = 70,
    GC_STATE_102_METHOD_HOLDING_2  = 71,
    GC_STATE_102_METHOD_HOLDING_3  = 72,
    GC_STATE_102_METHOD_HOLDING_4  = 73,
    GC_STATE_102_METHOD_HOLDING_5  = 74,
    GC_STATE_102_METHOD_HOLDING_6  = 75,
    GC_STATE_102_METHOD_HOLDING_7  = 76,
    GC_STATE_102_METHOD_HOLDING_8  = 77,
    GC_STATE_102_METHOD_HOLDING_9  = 78,
    GC_STATE_102_METHOD_HOLDING_10 = 79,
    
    GC_STATE_102_METHOD_COOLING = 80,
    GC_STATE_102_METHOD_FAULTED = 99,
    
    GC_STATE_102_METHOD_RUNNING_MINIMUM = GC_STATE_102_METHOD_INIT_TIME,
    GC_STATE_102_METHOD_RUNNING_MAXIMUM = GC_STATE_102_METHOD_HOLDING_10
};

/*
    A simplified version of the above, appropriate for both pre-102 and post-102 versions.

    The static function GCStateOrFaultCode::GetSimplifiedGCState converts the above codes
    to the simplified values below. Calling code can then ignore the USE_VERSION_102 
    #define'd symbol.
*/
typedef enum enumGCStateSimplified {
    GC_IDLE = 0,
    GC_EQUILIBRATING = 1,
    GC_STABILISING = 2,
    GC_WAIT_READY_IO = 3,
    GC_READY_TO_RUN = 4,
    GC_RUNNING = 5,
    GC_COOLING = 6,
    GC_FAULTED = 7,
    GC_STATE_SIMPLIFIED_COUNT = 8
} GCStateSimplified;


enum GC_FAULT {
    GC_FAULT_NO_ERROR = 0,
    GC_FAULT_EEPROM_FAILURE = 1,
    GC_FAULT_DOOR_OPEN_ERROR = 2,
    GC_FAULT_IRSENSOR_OPEN_ERROR = 3,
    GC_FAULT_IRSENSOR_SHORT_ERROR = 4,
    GC_FAULT_INJECTOR_PRT_OPEN_ERROR = 5,
    GC_FAULT_INJECTOR_PRT_SHORT_ERROR = 6,
    GC_FAULT_DETECTOR_PRT_OPEN_ERROR = 7,
    GC_FAULT_DETECTOR_PRT_SHORT_ERROR = 8,
    
    GC_FAULT_NOT_USED_INJECTOR_COUPLING_PRT_OPEN_ERROR = 9,
    GC_FAULT_NOT_USED_INJECTOR_COUPLING_PRT_SHORT_ERROR = 10,
    GC_FAULT_NOT_USED_DETECTOR_COUPLING_PRT_OPEN_ERROR = 11,
    GC_FAULT_NOT_USED_DETECTOR_COUPLING_PRT_SHORT_ERROR = 12,
    
    GC_FAULT_NOT_USED_13 = 13,
    GC_FAULT_NOT_USED_14 = 14,
    
    GC_FAULT_VOLTAGE_REF_ERROR = 15,
    GC_FAULT_VOLTAGE_SENSOR_ERROR = 16,
    
    GC_FAULT_COLUMN_OVER_TEMP_ERROR = 17,
    GC_FAULT_INJECTOR_OVER_TEMP_ERROR = 18,
    GC_FAULT_DETECTOR_OVER_TEMP_ERROR = 19,
    
    GC_FAULT_NOT_USED_INJECTOR_COUPLING_OVER_TEMP_ERROR = 20,
    GC_FAULT_NOT_USED_DETECTOR_COUPLING_OVER_TEMP_ERROR = 21,
    
    GC_FAULT_NOT_USED_22 = 22,
    
    GC_FAULT_NOT_USED_COLUMN_NOT_INIT = 23,
    GC_FAULT_NOT_USED_COLUMN_INSERT_FAILED = 24,
    
    GC_FAULT_I2C_NAK_ERROR = 25,
    
    GC_FAULT_TWI_TIMEOUT_ERROR = 26,
    GC_FAULT_TWI_BUS_ERROR = 27,
    GC_FAULT_TWI_SLAW_NACK = 28,
    GC_FAULT_TWI_SLADATA_NACK = 29,
    GC_FAULT_TWI_SLAR_NACK = 30,
    GC_FAULT_TWI_ARBITRATION_LOST = 31,
    GC_FAULT_TWI_UNKNOWN_ERROR = 32,
    
    GC_FAULT_TWI2_TIMEOUT_ERROR = 33,
    GC_FAULT_TWI2_BUS_ERROR = 34,
    GC_FAULT_TWI2_UNKNOWN_ERROR1 = 35,
    GC_FAULT_TWI2_UNKNOWN_ERROR2 = 36,
};

enum GC_FAULT_102 {
    GC_FAULT_102_NO_ERROR = 0,
    GC_FAULT_102_EEPROM_FAILURE = 1,
    GC_FAULT_102_DOOR_OPEN_ERROR = 2,
    
    GC_FAULT_102_COL_PRT_OPEN_ERROR = 3,
    GC_FAULT_102_COL_PRT_SHORT_ERROR = 4,
    GC_FAULT_102_INJ_PRT_OPEN_ERROR = 5,
    GC_FAULT_102_INJ_PRT_SHORT_ERROR = 6,
    GC_FAULT_102_DET_PRT_OPEN_ERROR = 7,
    GC_FAULT_102_DET_PRT_SHORT_ERROR = 8,
    GC_FAULT_102_AUX_PRT_OPEN_ERROR = 9,
    GC_FAULT_102_AUX_PRT_SHORT_ERROR = 10,
    GC_FAULT_102_TEMP4_PRT_OPEN_ERROR = 11,
    GC_FAULT_102_TEMP4_PRT_SHORT_ERROR = 12,
        
    GC_FAULT_102_VOLTAGE_REF_ERROR = 13,
    GC_FAULT_102_VSENSOR_ERROR = 14,
    
    GC_FAULT_102_COL_OVER_TEMP_ERROR = 15,
    GC_FAULT_102_INJ_OVER_TEMP_ERROR = 16,
    GC_FAULT_102_DET_OVER_TEMP_ERROR = 17,
    GC_FAULT_102_AUX_OVER_TEMP_ERROR = 18,
    GC_FAULT_102_TEMP4_OVER_TEMP_ERROR = 19,
    
    GC_FAULT_102_TC_K_SHORT_ERROR = 20,
   
    GC_FAULT_102_EPPC_NOTFITTED_ERROR = 25,
};


// This class simply contains an integer state or fault code, and an associated descriptive string
class GCStateOrFaultCode
{
public:
    GCStateOrFaultCode(int theCodeNumber, char* theCodeString);
    
    int GetCodeNumber(void);
    
    void GetCodeString(char *buff);
    
    static GCStateSimplified GetSimplifiedGCState(int rawGCState);
    
    static void DrawTextOnRunButton(char* text, GuiConst_INTCOLOR foreColor, GuiConst_INTCOLOR backColor, GuiConst_INT16U fontNo);

    static void DrawSimplifiedStateMessageOnHomePageRunButton(GCStateSimplified simplifiedState);
    
private:
    int codeNumber;
    char codeString[100];
};


// This class contains arrays of GC state and fault codes, each with its associated descriptive string.
// It allows the caller to obtain the descriptive string for any valid state or fault code.
class GCStateAndFaultCodes
{
public:
    GCStateAndFaultCodes();
    ~GCStateAndFaultCodes();
    
    bool GetStateCodeString(int stateCode, char *buff);

    bool GetSimplifiedStateCodeString(GCStateSimplified simplifiedState, char *buff);

    bool GetFaultCodeString(int faultCode, char *buff);
    
private:
#ifdef USE_VERSION_102
    enum StateCodeCount { STATE_CODE_COUNT = 28 };
#else
    enum StateCodeCount { STATE_CODE_COUNT = 21 };
#endif
    GCStateOrFaultCode *stateCodeArray[STATE_CODE_COUNT];
    
    GCStateOrFaultCode *simplifiedStateCodeArray[GC_STATE_SIMPLIFIED_COUNT];

#ifdef USE_VERSION_102
    enum FaultCodeCount { FAULT_CODE_COUNT = 22 };
#else
    enum FaultCodeCount { FAULT_CODE_COUNT = 37 };
#endif
    GCStateOrFaultCode *faultCodeArray[FAULT_CODE_COUNT];
};

#endif // GCSTATUSANDFAULTCODES_H
