#include "GCStateAndFaultCodes.h"

#include <string.h>


/*
    Passed three 8-bit colour components - red, green and blue.
    Returns the corresponding 16-bit colour value (5 bits for red, 6 bits for green, 5 bits for blue).
    
    Defined in main.cpp
*/
extern GuiConst_INTCOLOR SixteenBitColorValue(GuiConst_INT8U red, GuiConst_INT8U green, GuiConst_INT8U blue);


/*
  Classes that match state or fault code integer values to the corresponding strings
*/

GCStateOrFaultCode::GCStateOrFaultCode(int theCodeNumber, char* theCodeString)
{
    codeNumber = theCodeNumber;
    
    strcpy(codeString, theCodeString);
}

int GCStateOrFaultCode::GetCodeNumber(void)
{
    return codeNumber;
}

void GCStateOrFaultCode::GetCodeString(char *buff)
{
    strcpy(buff, codeString);
}   

// Static member function - provided so that we do not need to have
// '#ifdef USE_VERSION_102' code everywhere
GCStateSimplified GCStateOrFaultCode::GetSimplifiedGCState(int rawGCState)
{
#ifdef USE_VERSION_102 // #defined (or not) in GCStateAndFaultCodes.h
    if ((rawGCState >= GC_STATE_102_METHOD_RUNNING_MINIMUM) 
     && (rawGCState <= GC_STATE_102_METHOD_RUNNING_MAXIMUM)) {
         return GC_RUNNING;
    }

    // 'else'...
    switch(rawGCState) {
        case GC_STATE_102_METHOD_IDLE:
            return GC_IDLE;
            
        case GC_STATE_102_METHOD_EQUILIBRATING:
            return GC_EQUILIBRATING;
            
        case GC_STATE_102_METHOD_STABILISING:
            return GC_STABILISING;
            
        case GC_STATE_102_METHOD_WAIT_READY_IO:
            return GC_WAIT_READY_IO;
            
        case GC_STATE_102_METHOD_READY_TO_RUN:
            return GC_READY_TO_RUN;
            
        case GC_STATE_102_METHOD_COOLING:
            return GC_COOLING;
            
        default: // Assume faulted
            break;
    }
#else
    if ((rawGCState >= GC_STATE_RUNNING_MINIMUM) 
     && (rawGCState <= GC_STATE_RUNNING_MAXIMUM)) {
         return GC_RUNNING;
    }


    // 'else'...
    switch(rawGCState) {
        case GC_STATE_IDLE:
            return GC_IDLE;
            
        case GC_STATE_EQUILIBRATING:
            return GC_EQUILIBRATING;
            
        case GC_STATE_READY_TO_RUN:
            return GC_READY_TO_RUN;
            
        case GC_STATE_READY_TO_PRE_RUN:
        case GC_STATE_PRE_RUNNING:
            return GC_STABILISING;
            
        default: // Assume faulted
            break;
    }
#endif
    return GC_FAULTED;    
}

/*
    Static member function to draw a message over the Run button location on the Home page,
    showing the GC state.
    
    Args: the GC state, as a value in the "GCStateSimplified" enumeration
    
    No return code
*/
void GCStateOrFaultCode::DrawSimplifiedStateMessageOnHomePageRunButton(GCStateSimplified simplifiedState)
{
    switch (simplifiedState) {
        case GC_STABILISING:
            DrawTextOnRunButton("Stabilising", SixteenBitColorValue(0, 0xFF, 0), SixteenBitColorValue(0, 0, 0), GuiFont_Helv36Bold);
            break;
        case GC_EQUILIBRATING:
            DrawTextOnRunButton("Equilibrating", SixteenBitColorValue(255, 255, 50), SixteenBitColorValue(0, 0, 0), GuiFont_Helv36Bold);
            break;
        case GC_COOLING:
            DrawTextOnRunButton("Cooling", SixteenBitColorValue(0, 243, 255), SixteenBitColorValue(0, 0, 0), GuiFont_Helv36Bold);
            break;
        case GC_READY_TO_RUN:
//            DrawTextOnRunButton("Ready", SixteenBitColorValue(0, 0xFF, 0), SixteenBitColorValue(0, 0, 0), GuiFont_Helv36Bold);
// Test - don't display "Ready" message on top of green, enabled, Run button
            break;
        case GC_IDLE:
            DrawTextOnRunButton("Idle", SixteenBitColorValue(192, 0, 0), SixteenBitColorValue(0, 0, 0), GuiFont_Helv36Bold);
            break;
        case GC_WAIT_READY_IO:
            DrawTextOnRunButton("Waiting IO", SixteenBitColorValue(192, 192, 0), SixteenBitColorValue(0, 0, 0), GuiFont_Helv36Bold);
            break;
        case GC_FAULTED:
            DrawTextOnRunButton("Faulted", SixteenBitColorValue(255, 0, 0), SixteenBitColorValue(0, 0, 0), GuiFont_Helv36Bold);
            break;
        default:
            // Some other state - do nothing
            break;
    }
}

/*
    Draws the specified text on top of the Run button in the centre of the Home page,
    in the specified colour.
    
    It is up to the caller to display the Home page before calling this function.
    
    Args: pointer to the (null-terminated) text
          the foregound (i.e. text) colour
          the background colour
          the font ID (from the GuiFont_xxx list at the bottom of GuiFont.h)
                     (caller may want to e.g. use a larger font for shorter strings)
    
    No return code.
*/
void GCStateOrFaultCode::DrawTextOnRunButton(char* text, GuiConst_INTCOLOR foreColor, GuiConst_INTCOLOR backColor, GuiConst_INT16U fontNo)
{
    const GuiConst_INT16S X = 400; // Approx centre of Run button
    const GuiConst_INT16S Y = 220;
    
    GuiLib_DrawStr(
        X,                      //GuiConst_INT16S X,
        Y,                      //GuiConst_INT16S Y,
        fontNo,                 //GuiConst_INT16U FontNo,
        text,                   //GuiConst_TEXT PrefixLocate *String,
        GuiLib_ALIGN_CENTER,    //GuiConst_INT8U Alignment, 
        GuiLib_PS_ON,           //GuiConst_INT8U PsWriting,
        GuiLib_TRANSPARENT_ON,  //GuiConst_INT8U Transparent,
        GuiLib_UNDERLINE_OFF,   //GuiConst_INT8U Underlining,
        0,                      //GuiConst_INT16S BackBoxSizeX,
        0,                      //GuiConst_INT16S BackBoxSizeY1,
        0,                      //GuiConst_INT16S BackBoxSizeY2,
        GuiLib_BBP_NONE,        //GuiConst_INT8U BackBorderPixels,
        foreColor,              //GuiConst_INTCOLOR ForeColor,
        backColor               //GuiConst_INTCOLOR BackColor
    ); 
}


/*
    Set up the arrays of state and fault codes.
    
    The state and fault codes below are taken from the Nemesis 2 GC Control Protocol,
    version 1.0, dated 01/10/15. Note that the fault codes include those marked 'not used'.
*/
GCStateAndFaultCodes::GCStateAndFaultCodes()
{
    // First - the state codes

#ifdef USE_VERSION_102
    stateCodeArray[0]  = new GCStateOrFaultCode(GC_STATE_102_METHOD_IDLE,  "Idle");
    stateCodeArray[1]  = new GCStateOrFaultCode(GC_STATE_102_METHOD_EQUILIBRATING,  "Equilibrating");
    stateCodeArray[2]  = new GCStateOrFaultCode(GC_STATE_102_METHOD_STABILISING,  "Stabilising");
    stateCodeArray[3]  = new GCStateOrFaultCode(GC_STATE_102_METHOD_WAIT_READY_IO,  "Wait Ready IO");
    stateCodeArray[4]  = new GCStateOrFaultCode(GC_STATE_102_METHOD_READY_TO_RUN,  "Ready to Run");
    stateCodeArray[5]  = new GCStateOrFaultCode(GC_STATE_102_METHOD_INIT_TIME,  "Initial time");
    
    stateCodeArray[6]  = new GCStateOrFaultCode(GC_STATE_102_METHOD_RAMPING_1,  "Ramping segment 1");
    stateCodeArray[7]  = new GCStateOrFaultCode(GC_STATE_102_METHOD_RAMPING_2,  "Ramping segment 2");
    stateCodeArray[8]  = new GCStateOrFaultCode(GC_STATE_102_METHOD_RAMPING_3,  "Ramping segment 3");
    stateCodeArray[9]  = new GCStateOrFaultCode(GC_STATE_102_METHOD_RAMPING_4,  "Ramping segment 4");
    stateCodeArray[10] = new GCStateOrFaultCode(GC_STATE_102_METHOD_RAMPING_5,  "Ramping segment 5");
    stateCodeArray[11] = new GCStateOrFaultCode(GC_STATE_102_METHOD_RAMPING_6,  "Ramping segment 6");
    stateCodeArray[12] = new GCStateOrFaultCode(GC_STATE_102_METHOD_RAMPING_7,  "Ramping segment 7");
    stateCodeArray[13] = new GCStateOrFaultCode(GC_STATE_102_METHOD_RAMPING_8,  "Ramping segment 8");
    stateCodeArray[14] = new GCStateOrFaultCode(GC_STATE_102_METHOD_RAMPING_9,  "Ramping segment 9");
    stateCodeArray[15] = new GCStateOrFaultCode(GC_STATE_102_METHOD_RAMPING_10, "Ramping segment 10");
    stateCodeArray[16] = new GCStateOrFaultCode(GC_STATE_102_METHOD_HOLDING_1,  "Holding segment 1");
    stateCodeArray[17] = new GCStateOrFaultCode(GC_STATE_102_METHOD_HOLDING_2,  "Holding segment 2");
    stateCodeArray[18] = new GCStateOrFaultCode(GC_STATE_102_METHOD_HOLDING_3,  "Holding segment 3");
    stateCodeArray[19] = new GCStateOrFaultCode(GC_STATE_102_METHOD_HOLDING_4,  "Holding segment 4");
    stateCodeArray[20] = new GCStateOrFaultCode(GC_STATE_102_METHOD_HOLDING_5,  "Holding segment 5");
    stateCodeArray[21] = new GCStateOrFaultCode(GC_STATE_102_METHOD_HOLDING_6,  "Holding segment 6");
    stateCodeArray[22] = new GCStateOrFaultCode(GC_STATE_102_METHOD_HOLDING_7,  "Holding segment 7");
    stateCodeArray[23] = new GCStateOrFaultCode(GC_STATE_102_METHOD_HOLDING_8,  "Holding segment 8");
    stateCodeArray[24] = new GCStateOrFaultCode(GC_STATE_102_METHOD_HOLDING_9,  "Holding segment 9");
    stateCodeArray[25] = new GCStateOrFaultCode(GC_STATE_102_METHOD_HOLDING_10, "Holding segment 10");
    
    stateCodeArray[26] = new GCStateOrFaultCode(GC_STATE_102_METHOD_COOLING,  "Cooling");
    stateCodeArray[27] = new GCStateOrFaultCode(GC_STATE_102_METHOD_FAULTED,  "Faulted");
#else    
    stateCodeArray[0]  = new GCStateOrFaultCode(GC_STATE_IDLE,  "Idle");
    stateCodeArray[1]  = new GCStateOrFaultCode(GC_STATE_NOT_READY,  "Not Ready");
    stateCodeArray[2]  = new GCStateOrFaultCode(GC_STATE_EQUILIBRATING,  "Equilibrating");
    
    stateCodeArray[3]  = new GCStateOrFaultCode(GC_STATE_RUNNING_INITIAL_TIME,  "Running initial time");
    stateCodeArray[4]  = new GCStateOrFaultCode(GC_STATE_RAMPING_SEGMENT_1,  "Ramping segment 1");
    stateCodeArray[5]  = new GCStateOrFaultCode(GC_STATE_AT_UPPER_TEMP_SEGMENT_1,  "At upper temp segment 1");
    stateCodeArray[6]  = new GCStateOrFaultCode(GC_STATE_RAMPING_SEGMENT_2,  "Ramping segment 2");
    stateCodeArray[7]  = new GCStateOrFaultCode(GC_STATE_AT_UPPER_TEMP_SEGMENT_2,  "At upper temp segment 2");
    stateCodeArray[8]  = new GCStateOrFaultCode(GC_STATE_RAMPING_SEGMENT_3,  "Ramping segment 3");
    stateCodeArray[9]  = new GCStateOrFaultCode(GC_STATE_AT_UPPER_TEMP_SEGMENT_3, "At upper temp segment 3");
    stateCodeArray[10] = new GCStateOrFaultCode(GC_STATE_RAMPING_SEGMENT_4, "Ramping segment 4");
    stateCodeArray[11] = new GCStateOrFaultCode(GC_STATE_AT_UPPER_TEMP_SEGMENT_4, "At upper temp segment 4");
    stateCodeArray[12] = new GCStateOrFaultCode(GC_STATE_RAMPING_SEGMENT_5, "Ramping segment 5");
    stateCodeArray[13] = new GCStateOrFaultCode(GC_STATE_AT_UPPER_TEMP_SEGMENT_5, "At upper temp segment 5");

    stateCodeArray[14] = new GCStateOrFaultCode(GC_STATE_HOLDING, "Holding");
    stateCodeArray[15] = new GCStateOrFaultCode(GC_STATE_ABORTING, "Aborting");
    stateCodeArray[16] = new GCStateOrFaultCode(GC_STATE_COOLING, "Cooling");

    stateCodeArray[17] = new GCStateOrFaultCode(GC_STATE_READY_TO_PRE_RUN, "Ready to pre-run");
    stateCodeArray[18] = new GCStateOrFaultCode(GC_STATE_PRE_RUNNING, "Pre-running");
    stateCodeArray[19] = new GCStateOrFaultCode(GC_STATE_READY_TO_RUN, "Ready to run");

    stateCodeArray[20] = new GCStateOrFaultCode(GC_STATE_FAULTED, "Faulted");
#endif    
    
    // The simplified state codes
    simplifiedStateCodeArray[0] = new GCStateOrFaultCode(GC_IDLE, "Idle");
    simplifiedStateCodeArray[1] = new GCStateOrFaultCode(GC_EQUILIBRATING, "Equilibrating");
    simplifiedStateCodeArray[2] = new GCStateOrFaultCode(GC_STABILISING, "Stabilising");
    simplifiedStateCodeArray[3] = new GCStateOrFaultCode(GC_WAIT_READY_IO, "Waiting for IO ready");
    simplifiedStateCodeArray[4] = new GCStateOrFaultCode(GC_READY_TO_RUN, "Ready to run");
    simplifiedStateCodeArray[5] = new GCStateOrFaultCode(GC_RUNNING, "Running");
    simplifiedStateCodeArray[6] = new GCStateOrFaultCode(GC_COOLING, "Cooling");
    simplifiedStateCodeArray[7] = new GCStateOrFaultCode(GC_FAULTED, "Faulted");

    // Now the fault codes - these only apply if the state is 99, "Faulted"
    
#ifdef USE_VERSION_102
    faultCodeArray[0]  = new GCStateOrFaultCode(GC_FAULT_102_NO_ERROR,  "No error");
    faultCodeArray[1]  = new GCStateOrFaultCode(GC_FAULT_102_EEPROM_FAILURE,  "EEPROM failure");
    faultCodeArray[2]  = new GCStateOrFaultCode(GC_FAULT_102_DOOR_OPEN_ERROR,  "Door open error");
    
    faultCodeArray[3]  = new GCStateOrFaultCode(GC_FAULT_102_COL_PRT_OPEN_ERROR,  "Column PRT open error");
    faultCodeArray[4]  = new GCStateOrFaultCode(GC_FAULT_102_COL_PRT_SHORT_ERROR,  "Column PRT short error");
    faultCodeArray[5]  = new GCStateOrFaultCode(GC_FAULT_102_INJ_PRT_OPEN_ERROR,  "Injector PRT open error");
    faultCodeArray[6]  = new GCStateOrFaultCode(GC_FAULT_102_INJ_PRT_SHORT_ERROR,  "Injector PRT short error");
    faultCodeArray[7]  = new GCStateOrFaultCode(GC_FAULT_102_DET_PRT_OPEN_ERROR,  "Detector PRT open error");
    faultCodeArray[8]  = new GCStateOrFaultCode(GC_FAULT_102_DET_PRT_SHORT_ERROR,  "Detector PRT short error");
    faultCodeArray[9]  = new GCStateOrFaultCode(GC_FAULT_102_AUX_PRT_OPEN_ERROR,  "Auxiliary PRT open error");
    faultCodeArray[10] = new GCStateOrFaultCode(GC_FAULT_102_AUX_PRT_SHORT_ERROR,  "Auxiliary PRT short error");
    faultCodeArray[11] = new GCStateOrFaultCode(GC_FAULT_102_TEMP4_PRT_OPEN_ERROR,  "Temp4 PRT open error");
    faultCodeArray[12] = new GCStateOrFaultCode(GC_FAULT_102_TEMP4_PRT_SHORT_ERROR,  "Temp4 PRT short error");
        
    faultCodeArray[13] = new GCStateOrFaultCode(GC_FAULT_102_VOLTAGE_REF_ERROR,  "Voltage Ref error");
    faultCodeArray[14] = new GCStateOrFaultCode(GC_FAULT_102_VSENSOR_ERROR,  "Voltage sensor error");
    
    faultCodeArray[15] = new GCStateOrFaultCode(GC_FAULT_102_COL_OVER_TEMP_ERROR,  "Column over temperature error");
    faultCodeArray[16] = new GCStateOrFaultCode(GC_FAULT_102_INJ_OVER_TEMP_ERROR,  "Injector over temperature error");
    faultCodeArray[17] = new GCStateOrFaultCode(GC_FAULT_102_DET_OVER_TEMP_ERROR,  "Detector over temperature error");
    faultCodeArray[18] = new GCStateOrFaultCode(GC_FAULT_102_AUX_OVER_TEMP_ERROR,  "Auxiliary over temperature error");
    faultCodeArray[19] = new GCStateOrFaultCode(GC_FAULT_102_TEMP4_OVER_TEMP_ERROR,  "Temp4 over temperature error");
    
    faultCodeArray[20] = new GCStateOrFaultCode(GC_FAULT_102_TC_K_SHORT_ERROR,  "TC K short error");
   
    faultCodeArray[21] = new GCStateOrFaultCode(GC_FAULT_102_EPPC_NOTFITTED_ERROR,  "EPPC not fitted error");
#else
    faultCodeArray[0]  = new GCStateOrFaultCode(GC_FAULT_NO_ERROR,  "No error");
    faultCodeArray[1]  = new GCStateOrFaultCode(GC_FAULT_EEPROM_FAILURE,  "EEPROM failure");
    faultCodeArray[2]  = new GCStateOrFaultCode(GC_FAULT_DOOR_OPEN_ERROR,  "Door open");
    faultCodeArray[3]  = new GCStateOrFaultCode(GC_FAULT_IRSENSOR_OPEN_ERROR,  "IRsensor open");
    faultCodeArray[4]  = new GCStateOrFaultCode(GC_FAULT_IRSENSOR_SHORT_ERROR,  "IRsensor short");
    faultCodeArray[5]  = new GCStateOrFaultCode(GC_FAULT_INJECTOR_PRT_OPEN_ERROR,  "Injector PRT open");
    faultCodeArray[6]  = new GCStateOrFaultCode(GC_FAULT_INJECTOR_PRT_SHORT_ERROR,  "Injector PRT short");
    faultCodeArray[7]  = new GCStateOrFaultCode(GC_FAULT_DETECTOR_PRT_OPEN_ERROR,  "Detector PRT open");
    faultCodeArray[8]  = new GCStateOrFaultCode(GC_FAULT_DETECTOR_PRT_SHORT_ERROR,  "Detector PRT short");

    faultCodeArray[9]  = new GCStateOrFaultCode(GC_FAULT_NOT_USED_INJECTOR_COUPLING_PRT_OPEN_ERROR,  "(Not used): Injector coupling PRT open");
    faultCodeArray[10] = new GCStateOrFaultCode(GC_FAULT_NOT_USED_INJECTOR_COUPLING_PRT_SHORT_ERROR, "(Not used): Injector coupling PRT short");
    faultCodeArray[11] = new GCStateOrFaultCode(GC_FAULT_NOT_USED_DETECTOR_COUPLING_PRT_OPEN_ERROR, "(Not used): Detector coupling PRT open");
    faultCodeArray[12] = new GCStateOrFaultCode(GC_FAULT_NOT_USED_DETECTOR_COUPLING_PRT_SHORT_ERROR, "(Not used): Detector coupling PRT short");

    faultCodeArray[13] = new GCStateOrFaultCode(GC_FAULT_NOT_USED_13, "(Fault code 13, not used)");
    faultCodeArray[14] = new GCStateOrFaultCode(GC_FAULT_NOT_USED_14, "(Fault code 14, not used)");

    faultCodeArray[15] = new GCStateOrFaultCode(GC_FAULT_VOLTAGE_REF_ERROR, "Voltage ref error");
    faultCodeArray[16] = new GCStateOrFaultCode(GC_FAULT_VOLTAGE_SENSOR_ERROR, "Voltage sensor error");

    faultCodeArray[17] = new GCStateOrFaultCode(GC_FAULT_COLUMN_OVER_TEMP_ERROR, "Column over temp error");
    faultCodeArray[18] = new GCStateOrFaultCode(GC_FAULT_INJECTOR_OVER_TEMP_ERROR, "Injector over temp error");
    faultCodeArray[19] = new GCStateOrFaultCode(GC_FAULT_DETECTOR_OVER_TEMP_ERROR, "Detector over temp error");

    faultCodeArray[20] = new GCStateOrFaultCode(GC_FAULT_NOT_USED_INJECTOR_COUPLING_OVER_TEMP_ERROR, "(Not used): Injector coupling over temp error");
    faultCodeArray[21] = new GCStateOrFaultCode(GC_FAULT_NOT_USED_DETECTOR_COUPLING_OVER_TEMP_ERROR, "(Not used): Detector coupling over temp error");

    faultCodeArray[22] = new GCStateOrFaultCode(GC_FAULT_NOT_USED_22, "(Fault code 22, not used)");
    
    faultCodeArray[23] = new GCStateOrFaultCode(GC_FAULT_NOT_USED_COLUMN_NOT_INIT, "(Not used): Column not init");
    faultCodeArray[24] = new GCStateOrFaultCode(GC_FAULT_NOT_USED_COLUMN_INSERT_FAILED, "(Not used): Column insert failed");

    faultCodeArray[25] = new GCStateOrFaultCode(GC_FAULT_I2C_NAK_ERROR, "I2C NAK error");
    
    faultCodeArray[26] = new GCStateOrFaultCode(GC_FAULT_TWI_TIMEOUT_ERROR, "TWI timeout error");
    faultCodeArray[27] = new GCStateOrFaultCode(GC_FAULT_TWI_BUS_ERROR, "TWI bus error");
    faultCodeArray[28] = new GCStateOrFaultCode(GC_FAULT_TWI_SLAW_NACK, "TWI SLAW NACK");
    faultCodeArray[29] = new GCStateOrFaultCode(GC_FAULT_TWI_SLADATA_NACK, "TWI SLADATA NACK");
    faultCodeArray[30] = new GCStateOrFaultCode(GC_FAULT_TWI_SLAR_NACK, "TWI SLAR NACK");
    faultCodeArray[31] = new GCStateOrFaultCode(GC_FAULT_TWI_ARBITRATION_LOST, "TWI arbitration lost");
    faultCodeArray[32] = new GCStateOrFaultCode(GC_FAULT_TWI_UNKNOWN_ERROR, "TWI unknown error");
    
    faultCodeArray[33] = new GCStateOrFaultCode(GC_FAULT_TWI2_TIMEOUT_ERROR, "TWI2 timeout error");
    faultCodeArray[34] = new GCStateOrFaultCode(GC_FAULT_TWI2_BUS_ERROR, "TWI2 bus error");
    faultCodeArray[35] = new GCStateOrFaultCode(GC_FAULT_TWI2_UNKNOWN_ERROR1, "TWI2 unknown error 1");
    faultCodeArray[36] = new GCStateOrFaultCode(GC_FAULT_TWI2_UNKNOWN_ERROR2, "TWI2 unknown error 2");
#endif
}

GCStateAndFaultCodes::~GCStateAndFaultCodes()
{
    int i;
    
    for (i = 0; i < STATE_CODE_COUNT; ++i) {
        delete stateCodeArray[i];
    }
    
    for (i = 0; i < FAULT_CODE_COUNT; ++i) {
        delete faultCodeArray[i];
    }
}

/*
    Given an integer state code, returns the associated descriptive string.
    
    Args:   state code
            pointer to the buffer to contain the associated string (if found)
            
    Returns true if the state code is valid, false if not
*/
bool GCStateAndFaultCodes::GetStateCodeString(int stateCode, char *buff)
{
    for (int i = 0; i < STATE_CODE_COUNT; ++i) {
        if(stateCodeArray[i]->GetCodeNumber() == stateCode) {
            stateCodeArray[i]->GetCodeString(buff);
            return true;
        }
    }
    
    // 'else' - state code not found
    buff[0] = '\0';
    return false;
}

/*
    Given a simplified state value, returns the associated descriptive string.
    
    Args:   simplified state code 
            pointer to the buffer to contain the associated string (if found)
            
    Returns true if the state code is valid, false if not
*/
bool GCStateAndFaultCodes::GetSimplifiedStateCodeString(GCStateSimplified simplifiedStateCode, char *buff)
{
    for (int i = 0; i < GC_STATE_SIMPLIFIED_COUNT; ++i) {
        if(simplifiedStateCodeArray[i]->GetCodeNumber() == simplifiedStateCode) {
            simplifiedStateCodeArray[i]->GetCodeString(buff);
            return true;
        }
    }
    
    // 'else' - simplified state code not found
    buff[0] = '\0';
    return false;
}



/*
    Given an integer fault code, returns the associated descriptive string.
    
    Args:   state code
            pointer to the buffer to contain the associated string (if found)
            
    Returns true if the fault code is valid, false if not
*/
bool GCStateAndFaultCodes::GetFaultCodeString(int faultCode, char *buff)
{
    for (int i = 0; i < FAULT_CODE_COUNT; ++i) {
        if(faultCodeArray[i]->GetCodeNumber() == faultCode) {
            faultCodeArray[i]->GetCodeString(buff);
            return true;
        }
    }
    
    // 'else' - fault code not found
    buff[0] = '\0';
    return false;
}


    

