#include "GCHeatControl.h"

/*
    A class to 'encapsulate' the heat control on the GC
    
    Requires a pointer to the USBHostGC instance that corresponds to the GC,
             a pointer to the USBDeviceConnected instance that (also) corresponds to the GC

*/

// Defined in main.cpp
extern void EasyGUIDebugPrint(char *stuffToPrint, short X, short Y);

    
/*
    Constructor - passed the USBHostGC and USBDeviceConnected instances for the GC
*/    
GCHeatControl::GCHeatControl(USBDeviceConnected* newUsbDevice, USBHostGC* newUsbHostGC)
{
    usbDevice = newUsbDevice;
    usbHostGC = newUsbHostGC;

#ifdef DEBUG_TEST
    heatIsOn = false;
#endif
}

/*
    Tells the caller if the heat is currently on.
    
    Returns true if it is on, false if not.
*/
bool GCHeatControl::IsHeatOn(void)
{
#ifdef DEBUG_TEST
    return heatIsOn;
#else
    while(usbHostGC->ExecutingSetDeviceReport()) {}

    char response[50];
    usbHostGC->SetDeviceReport(usbDevice, "QHTS", response);
    // We expect a response like this: "QHTS0000" for off, "QHTS0001" for on
    
    char dbg[100];
    sprintf(dbg, "QHTS returned %s", response);
    EasyGUIDebugPrint(dbg, 100, 100);
    
    return (response[7] != '0');
#endif
}
        
/*
    Turns the heat on. Returns true if this was successful, false if not.
    
    Note that it will return false if the heat is already on.
*/      
bool GCHeatControl::TurnHeatOn(void)
{
#ifdef DEBUG_TEST
    if(heatIsOn) {
        // Heat is already on
        return false;
    }
    
    heatIsOn = true;
    
    return true;
#else    
    while(usbHostGC->ExecutingSetDeviceReport()) {}

    if(IsHeatOn()) {
        // Heat is already on
        return false;
    }
    
    char response[50];
    usbHostGC->SetDeviceReport(usbDevice, "CHON", response);
    // We expect a response like this: "DACK" for success, "DNAK" for failure

    char dbg[100];
    sprintf(dbg, "CHON returned %s", response);
    EasyGUIDebugPrint(dbg, 100, 150);
    
    return (response[1] == 'A');
#endif
}
    
/*
    Turns the heat off. Returns true if this was successful, false if not.
    
    Note that it will return false if the heat is already off.
*/      
bool GCHeatControl::TurnHeatOff(void)
{
#ifdef DEBUG_TEST
    if(!heatIsOn) {
        // Heat is already off
        return false;
    }
    
    heatIsOn = false;
    
    return true;
#else    
    while(usbHostGC->ExecutingSetDeviceReport()) {}

    if(!IsHeatOn()) {
        // Heat is already off
        return false;
    }
    
    char response[50];
    usbHostGC->SetDeviceReport(usbDevice, "CHOF", response);
    // We expect a response like this: "DACK" for success, "DNAK" for failure
    
    char dbg[100];
    sprintf(dbg, "CHOF returned %s", response);
    EasyGUIDebugPrint(dbg, 100, 200);
    
    return (response[1] == 'A');
#endif
}



