#ifndef ETHERNETHANDLER_H
#define ETHERNETHANDLER_H

#include "mbed.h"
#include "DMBoard.h"
#include "EthernetInterface.h"

#include "ThreadSignalEnums.h"

#include "USBHostGC.h"

/*
    This class handles Ethernet traffic between the LPC4088 and the PC. 
    
    The idea is that it 'listens' for an Ethernet command on its own thread, then either:
    
        1 - signals the main thread "I have a command ready for you to send to the GC" (because we do not want to have two threads
        sending commands to the GC over the USB link at the same time), then waits for the main thread to signal
        "I have the response ready for you to send back", then sends the response back over the Ethernet link.
    
    or:
        
        2 - does its own comms to the GC, using the same USBHostGC instance as the main thread uses, so that
        we do not send a command to the GC at the same time as the main (or any other) thread is doing the same.
        
    #define the symbol 'DO_USB_IN_THIS_THREAD' in the cpp file to select option 2 - comment it out to select option 1.

    This way, the UI, handled by the main thread, should not lock up while we deal with Ethernet transactions
    (this is the idea, anyway).
    
    Requires the ID of the main thread. 
    
    If it is to do its own USB comms to the GC, requires the appropriate USBHostGC pointers.
    
    Note also that all this class' members are static.
*/

class EthernetHandler {
public:
    static void HandlerFunction(void const *argument);
    
    static void SetMainThreadId(osThreadId threadId);
    
    static void SetEthernetServer(TCPSocketServer* server);
    
    static void SetUsbGC(USBDeviceConnected* newUsbDevice, USBHostGC* newUsbHostGC);
    
    static bool GCCommandReceived(void);
    
    static void GetGCCommand(char *commandBuffer, int* responseThreadSignalCode);
    
    static void SetGCResponse(char *response);
    
private:
    static osThreadId mainThreadId;
    static bool mainThreadIdSet;
    
    static TCPSocketServer* ethernetServer;
    
    static USBDeviceConnected* usbDevice;
    static USBHostGC* usbHostGC;
    
    static bool receivedGCCommand;
    
    static int responseReadyThreadSignalCode;
    static const int maxResponseReadyThreadSignalCode;
    
    static void SetupNextResponseReadyThreadSignalCode(void);

    static bool HandleEthernetMessage(TCPSocketConnection* client);
    
    static void SetGCDeviceReport(char *cmd, char *response);
    
    static char gcCommand[20];
    static char gcResponse[20];

    static Timer timer;
};

#endif // ETHERNETHANDLER_H