#ifndef TOUCHPANELPAGESELECTOR_H
#define TOUCHPANELPAGESELECTOR_H

#include "mbed.h"
#include "DMBoard.h"

#include "GuiLib.h"
#include "GuiDisplay.h"

#include "USBHostGC.h"

#include "GCComponentTypeEnums.h"
#include "GCIgnitionStateEnum.h"


/*
    A number of touch areas in our easyGUI user interface are used to select pages - i.e. if you press touch area index 'n',
    you expect easyGUI page (or "structure") index 'p' to appear.
    
    This class encapsulates the relationship between 'n' and 'p'.
*/

class TouchPanelPageSelector
{
public:
    TouchPanelPageSelector();
    
    TouchPanelPageSelector(int index, int page);
        
    int GetIndex(void) { return panelIndex; }
    
    // Parameters are not used in the base class, but may be required in derived classes
    virtual int GetPageNumber(USBDeviceConnected* usbDevice, USBHostGC* usbHostGC) { return pageNumber; }
    
    bool CanChangePage(void) { return pageChangeEnabled; }
    
    static void SetPageChangeEnabled(bool enabled) { pageChangeEnabled = enabled; }
    
private:
    int panelIndex;
    
    int pageNumber;
    
    static bool pageChangeEnabled;

protected:
    int GetBasePageNumber(void) { return pageNumber; }
};


/*
    TouchPanelPageSelector always returns the same page for the same index.
    For the detector page(s), this is not what we want - 
    we display a different detector page depending on the detector type.
    This class allows for this, and its 'GetPageNumber' function 
    returns the appropriate page number for the type
*/
class TouchPanelDetectorPageSelector : public TouchPanelPageSelector
{
public: 
    TouchPanelDetectorPageSelector();
    
    TouchPanelDetectorPageSelector(int index);
    
    virtual int GetPageNumber(USBDeviceConnected* usbDevice, USBHostGC* usbHostGC); 
    
private:
    DetectorType GetDetectorType(USBDeviceConnected* usbDevice, USBHostGC* usbHostGC); 
};


/*
    TouchPanelPageSelector always returns the same page for the same index.
    When the user aborts a run, however, this is not (necessarily) what we want - 
    we usually display the Home page, but if any components now require servicing
    (since even an aborted run counts as a 'cycle'), we want to display 
    the "Servicing Required" page instead. 
    
    This class allows for this, and its 'GetPageNumber' function 
    returns the appropriate page number.
*/
class TouchPanelAbortRunPageSelector : public TouchPanelPageSelector
{
public: 
    TouchPanelAbortRunPageSelector();
    
    TouchPanelAbortRunPageSelector(int index);
    
    virtual int GetPageNumber(USBDeviceConnected* usbDevice, USBHostGC* usbHostGC); 
};


/*
    TouchPanelPageSelector always returns the same page for the same index.
    For the 'Running Settings' page, this is not what we want - we want the 'Return' button
    on this page to return the user to the page from which he invoked the Running Settings page.
    A similar situation exists for the 'Abort Run' page - if the user presses 'No' on that page,
    he will expect to return to the page from which he invoked the 'Abort Run' page. 
    
    This class is intended to deal with situations like these. It includes a static variable that contains 
    the number of the page to be returned to. If 'thisIsTheReturnInstance' is true, the 'GetPageNumber' function 
    returns that page number, otherwise it sets that variable to the page to be returned to, 
    and returns the number of the page to be displayed now.
*/
class TouchPanelReturnPageSelector : public TouchPanelPageSelector
{
public: 
    TouchPanelReturnPageSelector();
    
    TouchPanelReturnPageSelector(int index, int pageToReturnTo, int pageToDisplayNow, bool thisWillBeTheReturnInstance);
    
    virtual int GetPageNumber(USBDeviceConnected* usbDevice, USBHostGC* usbHostGC); 

private:
    bool thisIsTheReturnInstance;
    int thisReturnPage;
    
    static int theReturnPage;
};


/*
    A class to encapsulate an array of (pointers to) page selectors.
*/
class TouchPanelPageSelectors
{
public:
    TouchPanelPageSelectors();

    ~TouchPanelPageSelectors();
    
    TouchPanelPageSelector* GetTouchPanelPageSelector(int touchAreaIndex);
    
    int GetTouchPanelPageSelectorIndex(TouchPanelPageSelector *tppsptr);
    
private:
    enum SelectorCount { SELECTOR_COUNT = 87 };
    
    TouchPanelPageSelector *tppsArray[SELECTOR_COUNT];
};

#endif //TOUCHPANELPAGESELECTOR_H
