
#include "SettingsHandler.h"

/*
    As explained in the header file, these are really C-style functions. They are all static,
    and they simply perform self-contained operations using the values passed to them.
    Having them in a class like this, however, seems more C++ like, easier to understand,
    and more consistent with the remainder of the code in this application.

    This class has no member variables and cannot be instantiated. It is simply 
    a container for the functions below.
*/

// Use QSPI file system - uncomment '#define DM_BOARD_USE_QSPI' and '#define DM_BOARD_USE_QSPI_FS' in dm_board_config.h, and:
#include "QSPIFileSystem.h"
static QSPIFileSystem shqspifs("qspi");
// Can now use QSPI memory as file device '/qspi/'

extern void EasyGUIDebugPrint(char *stuffToPrint, short X, short Y); // In main.cpp


void SettingsHandler::FormatQSPIIfRequired(void)
{
    if (!shqspifs.isformatted()) {
        shqspifs.format();
    }    
}

/*
    Gets a setting value as a string, by reading it 
    from the corresponding file in QSPI memory.
    
    Args: pointer to the setting name (as a string)
          pointer to a buffer to contain the value read (as a string)
          the length of the buffer
          
    Returns the amount of data read (this will be zero if the setting was not found)
*/
int SettingsHandler::GetSettingValueFromQSPI(char *settingName, char *settingValueBuff, int valueBuffLen)
{
    FILE * pFile;
    size_t result;
    
    char filename[100];
    sprintf(filename, "/qspi/%s.txt", settingName);
    
    FormatQSPIIfRequired();

//    pFile = fopen ( filename , "rb" );
// No - text mode
    pFile = fopen ( filename , "r" );
    if (pFile == NULL) {
        return 0;
    }

    // copy the file into the buffer:
    result = fread (settingValueBuff, 1, valueBuffLen, pFile);

    // terminate
    fclose (pFile);
    
    // Make sure we return a null-terminated string
    if(result < valueBuffLen) {
        settingValueBuff[result] = '\0';
    }
    
    return result;    
}

/*
    Settings values are stored in QSPI memory as strings. This function
    gets the setting specified, and returns its value as an integer.
    
    Args: the setting name
          a default value in case the setting is not found

    Return value: either the setting value, converted to an integer, or the default value
*/
int SettingsHandler::GetIntegerValueFromQSPISettings(char *settingName, int defaultValue)
{
    int retVal;
    
    char buff[100];
    
    if(GetSettingValueFromQSPI(settingName, buff, 100) > 0) {
        // We read a value for the specified setting from QSPI settings
        sscanf(buff, "%d", &retVal);
    } else {
        // Value not found in QSPI settings
        retVal = defaultValue;
    }
    
    return retVal;
}

/*
    Puts (i.e. writes) a setting value as a string,
    to the corresponding file in QSPI memory.
    
    Args: pointer to the setting name (as a string)
          pointer to a buffer containing the value to be written (as a string)
          the length of the buffer
          
    Returns the amount of data written
*/
int SettingsHandler::PutSettingValueToQSPI(char *settingName, char *settingValueBuff, int valueBuffLen)
{
    FILE * pFile;
    size_t result;

    char filename[100];
    sprintf(filename, "/qspi/%s.txt", settingName);

    FormatQSPIIfRequired();

//    pFile = fopen (filename, "wb");
// No - text mode
    pFile = fopen (filename, "w");
    if (pFile == NULL) {
        return 0;
    }
    
    result = fwrite (settingValueBuff, 1, valueBuffLen, pFile);
    
    fclose (pFile);
    
    return result;
}

/*
    Settings values are stored in QSPI memory as strings. This function
    takes an integer as the new value for the specified setting,
    and converts it to a string before storing it.
    
    Args: the setting name
          the new value for the setting

    No return value.
*/
void SettingsHandler::PutIntegerValueToQSPISettings(char *settingName, int value)
{
    char buff[100];
    
    sprintf(buff, "%d", value);
    
    PutSettingValueToQSPI(settingName, buff, strlen(buff));
}

/*
    Displays the directory of files in QSPI memory.
    
    Used for debugging/testing - not appropriate or required in the 'real' system.
    
    Args: the X and Y coordinates at which to display the directory on the screen
*/
void SettingsHandler::DisplayQSPIDirectory(GuiConst_INT16S X, GuiConst_INT16S Y)
{
    DIR *dp;
    dp = opendir("/qspi/");

    if(dp != NULL) {
        struct dirent *dirp;
        
        EasyGUIDebugPrint("SH::Start of QSPI directory", X, Y);
        Y += 30;
    
        while((dirp = readdir(dp)) != NULL) {
            EasyGUIDebugPrint(dirp->d_name, X, Y);
            Y += 30;
        }
        closedir(dp);
    
        EasyGUIDebugPrint("SH::End of QSPI directory", X, Y);
    } else {
        EasyGUIDebugPrint("SH::Failed to open QSPI directory", X, Y);
    }
}
