#ifndef METHODRAMPDATA_H
#define METHODRAMPDATA_H

#include "GuiLib.h"
#include "GuiDisplay.h"

#include "USBHostGC.h"

/*
    These classes deal with the method ramps - column, injector or gas.
    They obtain all the ramp data at the same time, and can then return information about the ramps to the caller.
    
    Provided to simplify the ramp handling, and particularly to avoid adding yet more code to "GetGCStatusLoop".
    
    The base class does most of the work, since the ramp handling is basically the same for the column, injector and gas - but is abstract. 
    Derived classes must define the functions to get the ramp data from the GC, since the commands to do so are different for each type.
*/
class MethodRampData
{
public:

    // Public constructor.
    //
    // There seems no particular reason to make this class a singleton.
    // There is only one method loaded into the GC at any time,
    // but so what if we have more than one copy of the ramp data?
    MethodRampData(USBDeviceConnected* newUsbDevice, USBHostGC* newUsbHostGC);    
    
    void GetRampDataFromGC(void);
    
    bool GetRampRate(unsigned int rampIndex, float *rampRate);
    bool GetRampUpperLimit(unsigned int rampIndex, float *rampUpperLimit);
    bool GetRampUpperTime(unsigned int rampIndex, float *rampUpperTime);
    
    unsigned int GetRampCount(void) { return rampCount; }
    bool GotRampData(void) { return gotRampData; }
    
    void RampsHaveBeenChangedInGC(void) { gotRampData = false; }
    
    void UpdateEasyGUIMethodPageVariables(unsigned int firstRampIndex);
    
    bool NeedToUpdateEasyGUIMethodPageRampVariables(void) { return needToUpdateEasyGUIMethodPageRampVariables; }
    
    unsigned int GetScrollRange(void);

protected:
    // Different GC commands for each ramp type
    virtual float GetRampRateFromGC(unsigned int rampIndex) = 0;
    virtual float GetRampUpperLimitFromGC(unsigned int rampIndex) = 0;
    virtual float GetRampUpperTimeFromGC(unsigned int rampIndex) = 0;
    
    // Different easyGUI variables for each ramp type
    enum { EASYGUI_RAMP_LINES = 5 };
    GuiConst_TEXT* easyGUIRampNumber[EASYGUI_RAMP_LINES];
    GuiConst_TEXT* easyGUIRampRateVariable[EASYGUI_RAMP_LINES];
    GuiConst_TEXT* easyGUIRampUpperLimitVariable[EASYGUI_RAMP_LINES];
    GuiConst_TEXT* easyGUIRampUpperTimeVariable[EASYGUI_RAMP_LINES];
    
    // Different units for the method ramp values in each ramp type - so apply a different format string 
    // when sprintf'ing the values to the corresponding easyGUI variables
    char easyGUIRampRateSprintfFormat[10];
    char easyGUIRampUpperLimitSprintfFormat[10];
    char easyGUIRampUpperTimeSprintfFormat[10];
    
    // Different variables for the ramp row colours (to show scrolling)
    GuiConst_INTCOLOR* easyGUIMethodRampOddRowsColour;
    GuiConst_INTCOLOR* easyGUIMethodRampEvenRowsColour;

    int GetIntRampValueFromGC(char *firstThreeCharsOfCommand, unsigned int rampIndex);
    
private:

    typedef struct structRampData
    {
        float rampRate;
        float rampUpperLimit;
        float rampUpperTime;
        
    } RampData;
    
    enum { MAX_RAMP_COUNT = 10 };
    RampData rampData[MAX_RAMP_COUNT];
    
    USBDeviceConnected* usbDevice;
    USBHostGC* usbHostGC;
    
    unsigned int rampCount;
    bool gotRampData;
    
    bool needToUpdateEasyGUIMethodPageRampVariables;
    
    static const GuiConst_INTCOLOR oddRampEasyGUIColour;
    static const GuiConst_INTCOLOR evenRampEasyGUIColour;
};


/*
    Derived class for the column method - with its own commands to get the data from the GC,
    and its own easyGUI variables to display the data
*/
class ColumnMethodRampData : public MethodRampData
{
public:
    ColumnMethodRampData(USBDeviceConnected* newUsbDevice, USBHostGC* newUsbHostGC);    

private:
    virtual float GetRampRateFromGC(unsigned int rampIndex);
    virtual float GetRampUpperLimitFromGC(unsigned int rampIndex);
    virtual float GetRampUpperTimeFromGC(unsigned int rampIndex);
};


/*
    Derived class for the injector method - with its own commands to get the data from the GC,
    and its own easyGUI variables to display the data
*/
class InjectorMethodRampData : public MethodRampData
{
public:
    InjectorMethodRampData(USBDeviceConnected* newUsbDevice, USBHostGC* newUsbHostGC);    

private:
    virtual float GetRampRateFromGC(unsigned int rampIndex);
    virtual float GetRampUpperLimitFromGC(unsigned int rampIndex);
    virtual float GetRampUpperTimeFromGC(unsigned int rampIndex);
};


/*
    Derived class for the gas method - with its own commands to get the data from the GC,
    and its own easyGUI variables to display the data
*/
class GasMethodRampData : public MethodRampData
{
public:
    GasMethodRampData(USBDeviceConnected* newUsbDevice, USBHostGC* newUsbHostGC);    

private:
    virtual float GetRampRateFromGC(unsigned int rampIndex);
    virtual float GetRampUpperLimitFromGC(unsigned int rampIndex);
    virtual float GetRampUpperTimeFromGC(unsigned int rampIndex);
};

#endif // METHODRAMPDATA_H

