#include "mbed.h"
#include "uLCD_4DGL.h"
#include "AnalogIn.h"

#include "Speaker.h"
#include "PinDetect.h"

#include "SDFileSystem.h"
#include <string>

#include "MMA8452.h"

uLCD_4DGL uLCD(p9, p10, p11);          // used in all parts

Speaker mySpeaker(p21);                // used in parts 4 and 6

SDFileSystem sd(p5, p6, p7, p8, "sd"); // used in part 5

Serial pc(USBTX,USBRX);                // used in part 6

DigitalOut myled1(LED1);
DigitalOut myled2(LED2);
DigitalOut myled3(LED3);
DigitalOut myled4(LED4);

PinDetect pb1(p16);
PinDetect pb2(p17);
PinDetect pb3(p18);
PinDetect selector(p14);

int currentPart = 1; // global placeholder

void part1() // What's up mbed!
{
    // begin transition code
    uLCD.cls();
    uLCD.printf("\nLab 2 - part 1:\n  What's up mbed!\n");
    wait(2);
    uLCD.cls();
    // end transition code
    
    uLCD.printf("\nWhat's up mbed!\n");
}

void part2() // Bouncing Ball
{
    // begin transition code
    uLCD.cls();
    uLCD.printf("\nLab 2 - part 2:\n  Bouncing Balls\n");
    wait(2);
    uLCD.cls();
    // end transition code
    
    
    uLCD.display_control(PORTRAIT);
    uLCD.cls();
    uLCD.printf("Bouncing Ball");
    uLCD.baudrate(BAUD_3000000); //jack up baud rate to max for fast display
    wait(1.0);
    
    //Set up initial conditions
    float fx=50.0,fy=21.0,vx=0.4,vy=0.3;
    int x=50,y=21,radius=4;
    uLCD.background_color(BLACK);
    uLCD.cls();
    
    //draw borders
    uLCD.line(0, 0, 127, 0, WHITE);
    uLCD.line(127, 0, 127, 127, WHITE);
    uLCD.line(127, 127, 0, 127, WHITE);
    uLCD.line(0, 127, 0, 0, WHITE);
    
    for (int i=0; i<1500; i++)
    {
        //draw ball
        uLCD.circle(x, y, radius, RED);
        //bounce off edge walls and slow down a bit
        if ((x<=radius+1) || (x>=126-radius)) vx = -.95*vx;
        if ((y<=radius+1) || (y>=126-radius)) vy = -.95*vy;
        //erase old ball location
        uLCD.filled_circle(x, y, radius, BLACK);
        //calculate new ball position
        fx=fx+vx;
        fy=fy+vy;
        x=(int)fx;
        y=(int)fy;
    } //end for loop
}

void part3() // Digital Thermometer
{
    // begin transition code
    uLCD.cls();
    uLCD.printf("\nLab 2 - part 3:\n  Digital\n  Thermometer\n");
    wait(2);
    uLCD.cls();
    // end transition code
    
    
    class TMP36
    {
        public:
            TMP36(PinName pin);
            TMP36();
            float read();
        private:
            AnalogIn _pin;
    };

    TMP36::TMP36(PinName pin) : _pin(pin) {} //This is an initializer list … pass pin to AnalogIn constructor
    float TMP36::read()
    {
        //convert sensor reading to C
        return ((_pin.read() * 3.3) - 0.500) * 100.0;
        // _pin.read() returns normalized voltage value as float
    }

    //instantiate new class to set p15 to analog input
    //to read and convert TMP36 sensor's voltage output
    TMP36 myTMP36(p15);

    float tempC, tempF;
    while(1)
    {
        tempC = myTMP36.read();
        // convert to F
        tempF = (9.0 * tempC) / 5.0 + 32.0;
        // print current temp
        uLCD.cls();
        uLCD.printf("\n%5.2f C\n%5.2f F\n\n", tempC, tempF);
        wait(0.5);
    }
}

void part4() // Jazzy Tunes
{
    // begin transition code
    uLCD.cls();
    uLCD.printf("\nLab 2 - part 4:\n  Jazzy Tunes\n");
    wait(2);
    uLCD.cls();
    // end transition code
    
    
    void pb1_hit_callback (void)
    {
        myled1 = !myled1;
        mySpeaker.PlayNote(200.0,0.25,0.1);
    }
    //------------------------------------------------------------------------------
    // Callback routine is interrupt activated by a debounced pb2 hit
    // That is … this code runs with interrupt is generated by second button press
    void pb2_hit_callback (void)
    {
        myled2 = !myled2;
        mySpeaker.PlayNote(400.0,0.25,0.1);
    }
    //------------------------------------------------------------------------------
    // Callback routine is interrupt activated by a debounced pb3 hit
    // That is … this code runs with interrupt is generated by third button press
    void pb3_hit_callback (void)
    {
        myled3 = !myled3;
        mySpeaker.PlayNote(800.0,0.25,0.1);
    }
    //------------------------------------------------------------------------------

    //setup push buttons
    pb1.mode(PullUp);
    pb2.mode(PullUp);
    pb3.mode(PullUp);
    // Delay for initial pullup to take effect
    wait(.01);
    // Setup Interrupt callback functions for a pb hit
    pb1.attach_deasserted(&pb1_hit_callback);
    pb2.attach_deasserted(&pb2_hit_callback);
    pb3.attach_deasserted(&pb3_hit_callback);
    // Start sampling pb inputs using interrupts
    pb1.setSampleFrequency();
    pb2.setSampleFrequency();
    pb3.setSampleFrequency();
    // pushbuttons now setup and running
    while(1)
    {
        myled4 = !myled4;
        wait(0.5);
    }
}

void part5_1() // Hello Micro SD Card - Writing
{
    // begin transition code
    uLCD.cls();
    uLCD.printf("\nLab 2 - part 5.1:\n  Hello Micro SD\n  Card - Writing\n");
    wait(2);
    uLCD.cls();
    // end transition code
    
    
    uLCD.printf("Hello Micro SD\nCard!\n");
    mkdir("/sd/mydir", 0777);
    FILE *fp = fopen("/sd/mydir/sdtest.txt", "w");
    if(fp == NULL) 
    {
        uLCD.printf("Error on open \n");
    }
    fprintf(fp, "I love ECE2036");
    fclose(fp);
}

void part5_2() // Hello Micro SD Card - Reading
{
    // begin transition code
    uLCD.cls();
    uLCD.printf("\nLab 2 - part 5.2:\n  Hello Micro SD\n  Card - Reading\n");
    wait(2);
    uLCD.cls();
    // end transition code
    
    
    string inputString;
    uLCD.printf("Content of sdtest.txt is: \n");
    FILE *fp = fopen("/sd/mydir/sdtest.txt", "r");
    if(fp == NULL) 
    {
        uLCD.printf("Error on open \n");
    }
    else
    {
        while (fscanf(fp,"%s", inputString)!= EOF) //reads in a string delineated by white space
        {
            uLCD.printf("%s ", inputString.c_str());
        }
    }
    fclose(fp);
}

void part6() // Triple Axis Accelerometer
{
    // begin transition code
    uLCD.cls();
    uLCD.printf("\nLab 2 - part 6:\n  Triple Axis\n  Accelerometer\n");
    wait(2);
    uLCD.cls();
    // end transition code
    
    
    // you can play around with the parameters to see the response
    int radius = 10;
    int offsetx = 63;
    int offsety = 63;
    double factor = 50;
    double music_factor = 200;
    bool MusicOn = false;
   
    //set the push buttons that control sounds
    pb1.mode(PullUp);
    pb2.mode(PullUp);
    //I will not use interupts like in jazzy tunes
 
   
    double x = 0, y = 0, z = 0;
 
    MMA8452 acc(p28, p27, 40000);  //instantiate an acc object! 
   
    //set parameters -- use these and don't change
    acc.setBitDepth(MMA8452::BIT_DEPTH_12);
    acc.setDynamicRange(MMA8452::DYNAMIC_RANGE_4G);
    acc.setDataRate(MMA8452::RATE_100);
   
   
    while(1) 
    {
        uLCD.circle(-1*y*factor+offsety, -1*x*factor+offsetx, radius, BLACK);
       
        if(!acc.isXYZReady()) 
        {
            wait(0.01);
        }
        else
        { 
            acc.readXYZGravity(&x,&y,&z); //notice this is passed by reference use pointers
      
            uLCD.circle(-1*y*factor+offsety, -1*x*factor+offsetx, radius, WHITE);
      
            if (MusicOn) mySpeaker.PlayNote(440.0+x*music_factor,0.25+0.2*y,0.05);
      
            if (pb1 == true) MusicOn = true;
           
            if (pb2 == false) MusicOn = false;
                 
            // You can uncomment this line to see the values coming off the MMA8452           
            uLCD.cls();
            uLCD.printf("\n(%.2f,%.2f,%.2f)  \n", x,y,z);
      
        } //end else
      
    } //end infinite while loop
}

void main_callback()
{
    switch (currentPart)
    {
        case 1: part1();
                currentPart++;
                break;
        case 2: part2();
                currentPart++;
                break;
        case 3: part3();
                currentPart++;
                break;
        case 4: part4();
                currentPart++;
                break;
        case 5: part5_1();
                part5_2();
                currentPart++;
                break;
        case 6: part6();
                currentPart = 1;
                break;
        
    }
}

int main() {
    selector.mode(PullUp);
    wait(.01);
    selector.attach_deasserted(&main_callback);
    selector.setSampleFrequency();
    while(1) 
    {
        continue;
    }
}
