/*************************************************************************
 * @file	jmSwitch.c
 * @brief	Switch Debounce Module
 * 				 
 * @version	1.0
 * @date	Feb 12, 2011
*/

#include "jmRingBuffer.h"
#include "jmMessages.h"
#include "jmSwitch.h"
#include "LPC17xx.h"
#include "main.h"
#include "jmCommands.h"
#include "jmMessages.h"
#include "jmRingBuffer.h"
#include "stdio.h"

#define swClose 0
#define swOpen 1
#define swNotInit 2


// Static structure creation
struct StructSwitch sSwitch[switchQty];

/***********************************************************************
 * @brief		Limit Switch Module Reset
 * @param[in]	none
 * @return		none
 **********************************************************************/
void SwitchModuleReset(void){
	int i;
	for(i=0;i<switchQty;i++){
		sSwitch[i].state=swNotInit;
	}
}

/***********************************************************************
 * @brief	Initialize an input Switch process
 * Format: command name (arg info)min..max values 
 * Command Line Format: swInit (Switch id)0..7 (Pin number)0..432 (Debounce)1..255
 * @param[in]	Extracted from command line (Switch id)0..7 (Pin number)0..432 (Debounce)1..255
 * @return		Send Message GPPS0 id pin debounce state
 **********************************************************************/
void cli_SwitchInit(void){
 //get switch id, DIP number, debounce time and update structure
   unsigned int swNum, pin, debounce, state, port,bitValue;
   if(ExtractUInteger(pLine,&swNum,0,7)){
      if(ExtractUInteger(pLine,&pin,0,432)){
         if(ExtractUInteger(pLine,&debounce,1,255)){
		    // Get port, bit and bit value
		    port = pin/100;
		    bitValue = 1<<(pin%100);
		
            sSwitch[swNum].pin = (uint8_t)pin;
            sSwitch[swNum].port = jmGPIO[port];
            sSwitch[swNum].bitValue = bitValue;
            sSwitch[swNum].debounce = (uint8_t)debounce;
			// read Switch present state
			state = sSwitch[swNum].port->FIOPIN &= sSwitch[swNum].bitValue;
			if(state>0)sSwitch[swNum].state = swOpen;
			else sSwitch[swNum].state = swClose;
			rGPPS0(swNum);	// Message GPPS0 id pin debounce state
            if(Feedback)printf("SW %d Pin %d Debounce %d Value %d\n",swNum,pin,debounce,state);

            return;
         }
      }
   }
   if(Help)printf("swInit (Switch id)0..7 (Pin number)0..432 (Debounce)1..255\n");
   
   // Ignore pending command line
   NextCommand(nl,pLine);
}

/***********************************************************************
 * @brief		Switch Edge Detection
 * @param[in]	none
 * @return		Send Message GPPS0 id pin debounce state
 **********************************************************************/
void SwitchEdgeDetect(void){
   unsigned int presentState,i;
   
   for(i=0;i<switchQty;i++){           // for each switch SM           
      if(sSwitch[i].state < swNotInit){  // Switch initialized ?
         if(sSwitch[i].eggTimer==0){   // debouncing ?                  
		      
            presentState = sSwitch[i].port->FIOPIN &= sSwitch[i].bitValue;
            
            if(presentState > 0)presentState = swOpen;	  // present switch state
			else 	presentState = swClose;
            
            if(sSwitch[i].state > presentState ){           // High to Low edge ?
		         sSwitch[i].eggTimer = sSwitch[i].debounce; // load debounce time
		        // memorize presentState
		        sSwitch[i].state = presentState;
				rGPPS0(i);						            // GPPS0 message
		    }
            else{   		 
              if(presentState > sSwitch[i].state ){          // Low to High edge ?
		           sSwitch[i].eggTimer = sSwitch[i].debounce;// load debounce time
		            // memorize presentState
		            sSwitch[i].state = presentState;
					rGPPS0(i);						         // GPPS0 message
		      }
            }            
	      }
      }
   }               
}

/***********************************************************************
 * @brief   Command line interface to read Limit Switch Status
 * Format: command name (arg info)min..max values 
 * Command Line Format: swRead (Switch id number)0..7
 * @param[in]	Extracted from command line (switch id number)0..7
 * @return		Message GPPS0 id pin debounce state
 **********************************************************************/
void cli_SwitchRead(void){
   unsigned int swNumber;
   if(ExtractUInteger(pLine,&swNumber,0,7)){   
	  // One Message GPPS0 id pin debounce state
      rGPPS0(swNumber);	 		
      return;
   } 
   if(Help)printf("swRead (Switch id number)0..7\n");
   // Ignore pending command line
   NextCommand(nl,pLine);
}  

/** Get Module Process Properties Command Line Interface
 * @brief	  Command Line Interface to Get All Public Process Properties
 * @param[in] none
 * @returns   8 Messages: GPPS0 id pin debounce state
 */
void cli_GPPS0(void){
	int i;
	for(i=0;i<switchQty;i++){	
		rGPPS0(i); // Message GPPS0 id pin debounce state
	}  
}

/** Public Properties Message
 * @brief	Send Process Properties to update GUI
 * @param[in] id Process identification
 * @returns   Message GPPS0 id pin debounce state
 */
void rGPPS0(unsigned int id )
{
    printf("GPPS0 %d %d %d %d\n",id,sSwitch[id].pin,sSwitch[id].debounce,sSwitch[id].state); 
}
