/*************************************************************************
 * @file     jmRingBuffer.c
 * @brief    Command Line Rx Ring Buffer
 *                  
 * @date    December 27,2010 
*/


#include "jmRingBuffer.h"
#include "stdio.h"

struct RingBuffer  Line, *pLine; 

/** @brief Command line ring buffer initialization.
 * @param none
 * @returns none 
 */
void InitCommandLineRingBuffer(void){
 pLine = &Line;
 FlushRingBuffer(pLine);
}

/** @brief Move ring head pointer foward.
 * @param *p pointer to ring buffer
 * @returns none 
 */
void NextHead(struct RingBuffer *p)              
{    p->head++;            
    if(p->head >= DimRingBuffer) 
       p->head=0;          
}

/** @brief Move ring tail pointer foward.
 * @param pointer to ring buffer
 * @returns none 
 */
void NextTail(struct RingBuffer *p)              
{    p->tail++;              
    if(p->tail >= DimRingBuffer)   
       p->tail=0;            
}
      
/** @brief Check if buffer full.
 * @param *p pointer to ring buffer
 * @returns true if full, false otherwise 
 */ 
bool Full(struct RingBuffer *p)
{  if(p->qty >= DimRingBuffer)         
       return true;
   else
      return false;
}
  
/** @brief Insert a char in buffer.
 * @param c unsigned char to be inserted
 * @param *p pointer to ring buffer
 * @returns none 
 */
void Insert(unsigned char c, struct RingBuffer *p)
{  if(Full(p))              
      NextHead(p);            
   else
      p->qty++;                  
 
   p->Buffer[p->tail]=c;          
   NextTail(p);                   
}

/** @brief Check if ring buffer not empty.
 * @param *p pointer to ring buffer
 * @returns true if full, false otherwise
 */ 
bool NotEmpty(struct RingBuffer *p)
{  if(p->qty == 0)                
      return false ;                          
   else 
      return true;                          
}

/** @brief Extract a char from ring buffer.
 * @param *p pointer to ring buffer
 * @returns unsigned char
 */
unsigned char Extract(struct RingBuffer *p)      
{  unsigned char c;
   c = p->Buffer[p->head];     
   if(NotEmpty(p))            
   {  NextHead(p);            
      p->qty--;                
   }
   return c;
}

/** @brief Flush ring buffer.
 * @param *p pointer to ring buffer
 * @returns none
 */
void FlushRingBuffer(struct RingBuffer *p)    
{  int i;
   p->head = 0;
   p->tail = 0;
   p->qty = 0;
   for(i=0;i<DimRingBuffer;i++)p->Buffer[i]=0;
}

/** @brief Delete last char from ring buffer.
 * @param *p pointer to ring buffer
 * @returns none
 */
void DelChar(struct RingBuffer *p)
{ if(p->qty != 0){
     if(p->tail==0)p->tail=DimRingBuffer;
     else p->tail--;  
     p->qty--;    
   }
}

/** @brief Remove a command line from ring buffer. 
 * @param c end command identifier unsigned char
 * @param *p pointer to ring buffer
 * @returns none
 */
void NextCommand(unsigned char c, struct RingBuffer *p){
   // remove all char till end identifier is found
   while(NotEmpty(p) && p->Buffer[p->head] != c)
   {  NextHead(p);            
      p->qty--;                 
   }

   // remove end identifier
   if(NotEmpty(p)&& p->Buffer[p->head] == c)               
   {  NextHead(p);             
      p->qty--;                 
   }
}

/** @brief View ring buffer content.
 * Print ring buffer content
 * @param *p pointer to ring buffer
 * @returns none
 */
void ViewRingBuffer(struct RingBuffer *p){
   int i,j;

   printf("\nRingBuffer Qty: %d \nContent: ",p->qty);

   for(j=0,i=p->head;j<p->qty;j++){
      printf("%c",p->Buffer[i]); 
      if(i++>DimRingBuffer)i=0; 
   }
   printf("\n");
}

/** @brief Extract a word from ring buffer.
 * The extracted word is put in the array pointed by word
 * @param p pointer to a ring buffer 
 * @param word pointer to array of char
 * @returns true if a word is extracted otherwise returns false
 */ 
bool ExtractWord(struct RingBuffer *p, char * word){   
   unsigned char c;
   int i,j;
   j=0;

   if(NotEmpty(p)){
        for(i=0;i<WordMaxSize-1;i++){
          // extract a char from Rx ring buffer
          c=Extract(p);
    
          // remove leading blanks
          if(c==' ' && j==0)continue;
    
          // end of word or end of command line
          if(c==' ' || c==nl)break;
    
          // build the Word
          word[j++]=c;
       }
       // 0 string termination
       word[j]=0;
       if(j>0)return true;
    }
    return false;
}

/** @brief Extract an unsigned int from ring buffer.
 * Convert a word from buffer into an integer between min and max values
 * Value converted should be between min and max values.
 * Value should be decimal or hexadecimal (beginning by 0x or 0X)
 * @param p pointer to ring buffer
 * @param result pointer to unsigned int
 * @param min minimum limit
 * @param max maximum limit
 * @returns true if value is converted beetween limits, including limits.
 */
bool ExtractUInteger(struct RingBuffer *p, unsigned int *result, unsigned int min, unsigned int max){
   unsigned int i ;
   char word[WordMaxSize-1];

   if(ExtractWord(p,word)){       // Extract string value
      if(word[0]=='0' && (word[1]=='x' || word[1]=='X')) {
         sscanf(word,"%x",&i);  // convert hexadecimal input
      }
      else
         sscanf(word,"%d",&i);  // convert decimal input

      if(i>=min && i<=max){
         *result = i;
         return true;
      }
   }
   *result = 0;
   return false;
}
