/*************************************************************************
 * @file    jmLPC17xx_gpio.c
 * @brief    Command Line Interface Module for LPC_GPIO
 *                  
 * @version    1.1
 * @date    Jan 24, 2011
 */

#include "jmLPC17xx_gpio.h"
#include "LPC17xx.h"
#include "main.h"
#include "jmCommands.h"
#include "jmMessages.h"
#include "jmRingBuffer.h"
#include "stdio.h"


#define DIR 0
#define PIN 1
#define MASK 2
#define CLR 3
#define SET 4

#define Clear 0
#define Set 1
#define Toggle 2

/***********************************************************************
 * @brief        Send port registers content
 * @param[in]    portNum    (Port Number value)0..4
 * @return        none
 **********************************************************************/
void PortInfo(unsigned int portNum){
   if(portNum <5){
      printf("Port %1d\n",portNum);
      printf("FIODIR   \t0x%08X\n",jmGPIO[portNum]->FIODIR);
      printf("FIOMASK  \t0x%08X\n",jmGPIO[portNum]->FIOMASK);
      printf("FIOPIN   \t0x%08X\n",jmGPIO[portNum]->FIOPIN);
      printf("FIOSET   \t0x%08X\n",jmGPIO[portNum]->FIOSET);
      printf("FIOCLR   \t0x%08X\n",jmGPIO[portNum]->FIOCLR);
      printf("\n");
   }
}

/***********************************************************************
 * @brief        Send all ports registers content
 * @param[in]    none
 * @return        none
 **********************************************************************/
void PortsInfo(void){
   unsigned int portNum;
   for(portNum=0;portNum<5;portNum++) PortInfo(portNum);
}

/***********************************************************************
 * @brief    iport Command Line Interface.
 * Send port register content
 * Command Line Format: iport (portNumber)0..4 
 * @example iport 1     Will printout Port1 registers content
 * @param[in]    Extracted from command line ring buffer (PortNumber)0..4
 * @return        none
 **********************************************************************/
void cli_PortInfo(void){
   unsigned int portNum;

   if(ExtractUInteger(pLine,&portNum,0,4))
      PortInfo(portNum);
   else{
      if(Help)printf("iport (portNumber)0..4\n");
      // Ignore pending command line
      NextCommand(nl,pLine);
   }
}


/***********************************************************************
 * @brief        Command line interface to read and send Port.Bit per bit position
 * Command Line Format: bitRead  (Port Number)0..4 (Bit position)0..31
 * ex: bitRead  1 18  read and send P1.18 input pin value
 * pin must be previously set to input in GPIODIR and associated GPIOMASK bit= 0
 * @param[in]    Extracted from command line (Port Number)0..4 (Bit position)0..31
 * @return        none
 **********************************************************************/
void cli_BitRead(void){
  unsigned int portNumber, bitPosition, bitValue;
   // Extract pin number from Command Line
   if(ExtractUInteger(pLine,&portNumber,0,4)){
      // Extract bit position from Command Line
      if(ExtractUInteger(pLine,&bitPosition,0,31)){
         // Extract state from Command Line
         bitValue = 1<<bitPosition;
         if(jmGPIO[portNumber]->FIOPIN & bitValue)bitValue =1;
         printf("P%d.%d %d\n",portNumber,bitPosition,bitValue);
          return;
      }
   } 
   if(Help)printf("bitRead (Port Number)0..4 (Bit position)0..31\n");
   // Ignore pending command line
   NextCommand(nl,pLine);
}

/** Read GPIO Registers 
 * @brief        Command Line Interface to Read and Send GPIO register value
 * @param[in]    Extracted from command line GPPG0 (Port)0..4 (Register DIR/PIN/MASK)0..2
 * @returns    Message: GPPG0 (Port)0..4 (Register DIR/PIN/MASK)0..2 (value)0..0xFFFFFFFF 
 */
void cli_GPPG0(void){
   uint32_t value=0;
   unsigned int port, reg;
   if(ExtractUInteger(pLine,&port,0,4)){    // extract port id
         if(ExtractUInteger(pLine,&reg,0,4)){  // extract register id
         switch(reg){
             case DIR: value = jmGPIO[port]->FIODIR; break;
            case PIN: value = jmGPIO[port]->FIOPIN; break;
            case MASK: value = jmGPIO[port]->FIOMASK; break;
            case CLR: value = jmGPIO[port]->FIOCLR; break;
            case SET: value = jmGPIO[port]->FIOSET; break;
        }
        printf("GPPG0 %d %d 0x%08X\n",port,reg,value);
          return;
      }
    }
  if(Help)printf("GPPG0 (Port)0..4 (Register DIR/PIN/MASK/CLR/SET)0..4\n");  
  // Ignore pending command line
  NextCommand(nl,pLine);
}

/***********************************************************************
 * @brief        Command line interface to access GPIO register bits
 * Use with care. Designed to be used with gui which locks non MBED DIP pins
 * Command Line Format:   gpioBits (Port id)0..4 (Register DIR/PIN/MASK/CLR/SET)0..4 (value)0..0xFFFFFFFF (clear/set/toggle)0..2\n
 * @param[in]    Extracted from command line   (Port)0..4 (Register DIR/PIN/MASK)0..2 (value)0..0xFFFFFFFF 
 * @return        Message: GPPG0 (Port)0..4 (Register DIR/PIN/MASK)0..2 (value)0..0xFFFFFFFF 
 **********************************************************************/
void cli_gpioBits(void){
  unsigned int port, reg, value, action;
   // Extract port id from Command Line
   if(ExtractUInteger(pLine,&port,0,4)){
      // Extract register id from Command Line
      if(ExtractUInteger(pLine,&reg,0,4)){
          // Extract value from Command Line
          if(ExtractUInteger(pLine,&value,0,0xFFFFFFFF)){
             // Extract state from Command Line
             if(ExtractUInteger(pLine,&action,0,2)){
                switch(reg){
                   case DIR:
                       switch(action){
                           case Clear: jmGPIO[port]->FIODIR &= ~value;
                              break;
                           case Set: jmGPIO[port]->FIODIR |= value;
                              break;
                           case Toggle: jmGPIO[port]->FIODIR ^= value;
                              break;
                        }
                        break;
                   case PIN:
                       switch(action){
                           case Clear: jmGPIO[port]->FIOPIN &= ~value;
                              break;
                           case Set: jmGPIO[port]->FIOPIN |= value;
                              break;
                           case Toggle: jmGPIO[port]->FIOPIN ^= value;
                              break;
                        }
                        break;
                   case MASK:
                       switch(action){
                           case Clear: jmGPIO[port]->FIOMASK &= ~value;
                              break;
                           case Set: jmGPIO[port]->FIOMASK |= value;
                              break;
                           case Toggle: jmGPIO[port]->FIOMASK ^= value;
                              break;
                        }
                        break;
                   case CLR:
                       switch(action){
                           case Clear: jmGPIO[port]->FIOCLR &= ~value;
                              break;
                           case Set: jmGPIO[port]->FIOCLR |= value;
                              break;
                           case Toggle: jmGPIO[port]->FIOCLR ^= value;
                              break;
                        }
                        break;
                   case SET:
                       switch(action){
                           case Clear: jmGPIO[port]->FIOSET &= ~value;
                              break;
                           case Set: jmGPIO[port]->FIOSET |= value;
                              break;
                           case Toggle: jmGPIO[port]->FIOSET ^= value;
                              break;
                        }
                        break;

                }
                // gui feedback  
                switch(reg)
                {    
                    case DIR: printf("GPPG0 %d %d 0x%08X\n",port,reg,jmGPIO[port]->FIODIR);
                        break;
                    case PIN: printf("GPPG0 %d %d 0x%08X\n",port,reg,jmGPIO[port]->FIOPIN);
                        break;
                    case MASK: printf("GPPG0 %d %d 0x%08X\n",port,reg,jmGPIO[port]->FIOMASK);
                        break;
                    case CLR: printf("GPPG0 %d %d 0x%08X\n",port,reg,jmGPIO[port]->FIOCLR);
                        break;
                    case SET: printf("GPPG0 %d %d 0x%08X\n",port,reg,jmGPIO[port]->FIOSET);
                        break;
                }
                if(Feedback)printf("GPIO %d %d 0x%08X %d\n",port,reg,value,action);
                return;
             }
         }
      }
   } 
   if(Help)printf("gpioBits (Port id)0..4 (Register DIR/PIN/MASK/CLR/SET)0..4 (value)0..0xFFFFFFFF (clear/set/toggle)0..2\n");
   // Ignore pending command line
   NextCommand(nl,pLine);
}

/** @brief   Reset/Set/Toggle an output pin
 * Also sets pin as outputs.
 * @param[in]    pin      pin id (value)0..432
 * @param[in]    action  0..2 Clear/Set/Toggle
 * @returns     Return Message: GPPB0 pin state
 */
void gpio(unsigned int pin, unsigned int action){
   // Get port and bit for that pin
   uint32_t bitValue;
   uint8_t port;

   // Get port, bit and bit value
   port = pin/100;
   bitValue = 1<<(pin%100);
   
   // set mbed pin direction as output
   jmGPIO[port]->FIODIR |= bitValue;
   // make sure FIOMASK bit is enable
   jmGPIO[port]->FIOMASK &= ~bitValue;

   switch(action)
   {
    case Clear:     jmGPIO[port]->FIOPIN &= ~bitValue; // reset pin low 
                break;
    case Set:    jmGPIO[port]->FIOPIN |= bitValue;  // set pin High
                break;
    case Toggle:jmGPIO[port]->FIOPIN ^= bitValue; // toggle pin 
                break;
   }
   if( (jmGPIO[port]->FIOPIN & bitValue) == 0)bitValue = 0;
   else bitValue = 1;

   // Report bit modification
   printf("GPPB0 %d %d\n",pin, bitValue);
}

/** @brief   Command line to Reset/Set/Toggle an output pin
 * Also sets pin as outputs.  Not all pins are available
 * Use with care. Designed to be used with gui which locks non MBED DIP pins
 * Command Line Format:   gpioBit (Pin id)0..432 (action clear/set/toggle)0..2
 * @param[in]    From command line: (Pin id)0..432 (action clear/set/toggle)0..2
 * @returns     Return Message: GPPB0 pin state
 */
void cli_gpioBit(void){
  unsigned int pin, action;
   // Extract pin number from Command Line
   if(ExtractUInteger(pLine,&pin,0,432)){
      // Extract action from Command Line
      if(ExtractUInteger(pLine,&action,0,2)){
         gpio(pin,action);
         return;
      }
   } 
   if(Help)printf("Use with care ! gpioBit (pin Number)0..432 (action Clear/Set/Toggle)0..2\n");
   // Ignore pending command line
   NextCommand(nl,pLine);
}
