/*************************************************************************
 * @file	jmStepperAxis.c
 * @brief	Command Line Interface StepperAxis Module
 * 				 
 * @version	1.0
 * @date	Feb 12, 2011
 */

#include "jmMessages.h"
#include "jmStepperAxis.h"
#include "LPC17xx.h"
#include "main.h"
#include "jmCommands.h"
#include "jmMessages.h"
#include "jmRingBuffer.h"
#include "stdio.h"
#include "jmSwitch.h"
#include "jmStepper.h"

#ifndef nonStop
   #define nonStop 65535
#endif

#define NotInit 0
#define Initialized 1
#define CW 1
#define CCW 0
#define Open 1
#define Close 0
#define bothLimitSwitchActivated 4
#define LimitCWReached 3
#define LimitCCWReached 2
#define LogicOK 1
#define true 1
#define false 0

// static structures creation
struct StructStepperAxis sStepperAxis[stepperAxisQty];

/** Module Data Structure Initialization
 * @brief   All State Machines put to sleep.
 * @param[in]	none
 * @returns none
 */
void StepperAxisInit(void){
   int i;
   for(i=0;i<stepperAxisQty;i++){
     sStepperAxis[i].active = false;
   }
}

/**  @brief stepperAxis Command Line Interface.
 * Set up a stepper axis from one stepper and two limit switch modules
 * Command Line Format:  stepperAxis id stepperID limitCWid limitCCWid
 * @param[in]	Extracted from command line  id stepperID limitCWid limitCCWid
 * @returns    Message:  GPPSTA id stepperID limitCWid limitCCWid status
 */
void cli_StepperAxis(void){
unsigned int id, stepperID, limitCWid, limitCCWid;

   if(ExtractUInteger(pLine,&id,0,stepperAxisQty-1)){               // extract stepper axis id
   	 if(ExtractUInteger(pLine,&stepperID,0,stepperQty-1)){			// extract stepper id
	 	 if(ExtractUInteger(pLine,&limitCWid,0,switchQty-1)){		// extract Limit Switch CW id
		      if(ExtractUInteger(pLine,&limitCCWid,0,switchQty-1)){ // extract Limit Switch CCW id

				  sStepperAxis[id].active=true;
				  sStepperAxis[id].stepperID=stepperID;
				  sStepperAxis[id].limitCW=limitCWid;
				  sStepperAxis[id].limitCCW=limitCCWid;

				  // chip report to GUI
				  rGPPSTA(id); 
				  if(Feedback)printf("StepperAxis %d StepperID %d limitCCWid %d limitCWid %d\n",id,stepperID,limitCCWid,limitCWid);              
		          return;
			   }// CCW
			}// CW
		}// stepper
    }// axis
    if(Help)printf("stepperAxis (Stepper id)0..3 (limit switch CCW id)0..7 (limit switch CW id)0..7\n"); 
    // Ignore pending command line
    NextCommand(nl,pLine);
}

/** @brief   
 * @param[in] none
 * @returns 
 */
void StepperAxisSM(void)
{
   int i;
   for(i=0;i<stepperAxisQty;i++)
   {
     if(sStepperAxis[i].active==true)
	 {
		 uint8_t oldStatus = sStepperAxis[i].status;	 
		 
		 // Limits Reached
		 // two limits switches activated ? 
		 if(sSwitch[sStepperAxis[i].limitCW].state == Close && sSwitch[sStepperAxis[i].limitCCW].state==Close){
		 	sStepper[sStepperAxis[i].stepperID].nSteps=0;         // stop stepper
			sStepperAxis[i].status = bothLimitSwitchActivated;
			if(oldStatus !=  bothLimitSwitchActivated)rGPPSTA(i); // report change 			   
			continue;  
	     }
		 // Limit Switch CW activated and step direction CW
		 if(sSwitch[sStepperAxis[i].limitCW].state == Close && sStepper[sStepperAxis[i].stepperID].cw ==CW){
			sStepper[sStepperAxis[i].stepperID].nSteps=0; // stop stepper
			sStepperAxis[i].status = LimitCWReached;  
			if(oldStatus !=  LimitCWReached)rGPPSTA(i); // report change  			     
			continue;
		 }

		 // Limit Switch CCW activated and step direction CCW
		 if(sSwitch[sStepperAxis[i].limitCCW].state == Close && sStepper[sStepperAxis[i].stepperID].cw ==CCW){
			sStepper[sStepperAxis[i].stepperID].nSteps=0; // stop stepper
			sStepperAxis[i].status = LimitCCWReached;    
			if(oldStatus !=  LimitCCWReached)rGPPSTA(i); // report change 			 
			continue;
		 }	
		 
		 // safe operating area
		 // no limit switch activated
		 if(sSwitch[sStepperAxis[i].limitCW].state == Open && sSwitch[sStepperAxis[i].limitCCW].state == Open){
			sStepperAxis[i].status = LogicOK;  
			if(oldStatus !=  LogicOK)rGPPSTA(i); // report change   
			continue;
		 }

		 // Limit Switch CW activated and step direction CCW
		 if(sSwitch[sStepperAxis[i].limitCW].state == Close && sStepper[sStepperAxis[i].stepperID].cw ==CCW){
			sStepperAxis[i].status = LogicOK;   
			if(oldStatus !=  LogicOK)rGPPSTA(i); // report change     
			continue;
		 }

		 // Limit Switch CCW activated and step direction CW
		 if(sSwitch[sStepperAxis[i].limitCCW].state == Close && sStepper[sStepperAxis[i].stepperID].cw ==CW){
			sStepperAxis[i].status = LogicOK; 
			if(oldStatus !=  LogicOK)rGPPSTA(i); // report change  
			continue;
		 }	 
	 }
   }
}

/** Module Get Module Process Properties Command Line Interface
 * @brief		Command Line Interface to Get Module Public Process Properties
 * @param[in]	id Extracted from command line id Process identification
 * @returns    none
 */
void cli_GPPSTA(void)
{	unsigned int id;
   if(ExtractUInteger(pLine,&id,0,stepperAxisQty-1)){  // extract id 
       rGPPSTA(id);   
	   return;
	}

    if(Help)printf("GPPSTA (StepperAxis id)0..%d\n",stepperAxisQty-1);  
    // Ignore pending command line
    NextCommand(nl,pLine);
}

/** Public Properties Message
 * @brief	Send Process Properties to update GUI
 * @param[in] id Process identification
 * @returns   Message: GPPSTA id stepperID limitCWid limitCCWid status
 */
void rGPPSTA(unsigned int id ){		 
	int status;
	if(sStepperAxis[id].active==false)status=0;
	else status = sStepperAxis[id].status;
	printf("GPPSTA %d %d %d %d %d\n",id,sStepperAxis[id].stepperID,sStepperAxis[id].limitCCW,sStepperAxis[id].limitCW ,status);
}
