/**********************************************************************************
 * This program monitors the power on state, kill state and battery voltage on
 * Woodstream mouse trap using the Multitech mDot and UDK2 development system.
 * The power on state is monitored on the PA_4(UDK2 Pin D10)and the kill state is
 * monitored on the PA_5 (UDK2 Pin D13) Digital Input Pins.  The battery voltage
 * is monitored on the PB_1 (UDK2 Pin A0) Analog Input.  The status of these pins
 * are transferred from the mDot LoRa to the Conduit LoRa gateway.  The status is
 * also shown on the USB Debug Terminal Window.
 *********************************************************************************/

#include "mbed.h"
#include "mDot.h"
#include "MTSLog.h"
#include <string>
#include <vector>
#include <algorithm>
#include "PinDetect.h"

mDot* dot;
Ticker ledTick;
Ticker batteryvoltageMonitorTick;
Ticker periodicSendTick;
     
AnalogIn batt_voltage(PB_1);
DigitalIn mt_pwron(PA_4);
DigitalIn mt_caught(PA_5);
DigitalOut transmitLED(LED1);

// Configuration variables
static std::string config_network_name = "JFETENGINEERING";
static std::string config_network_pass = "Deboraheng";
static uint8_t config_frequency_sub_band = 7;


//Global Variables
//static uint16_t setPoint = 21;       //21 C is standard room temp
static volatile bool timeToReadBatteryVoltage = true;
static volatile bool dataChanged = true;
//static volatile bool thermostatActivated = false;
//static uint16_t sentSetPoint;

//Function prototypes
void ledTock();
void periodicSendTock();
void batteryvoltageMonitorTock();
//void up_button_callback();
//void down_button_callback();
void printError(mDot* dot, int32_t returnCode);
void printVersion();
bool setFrequencySubBand(uint8_t subBand);
bool setNetworkName(const std::string name);
bool setNetworkPassphrase(const std::string passphrase);
bool setPower(uint8_t power);
bool setAck(uint8_t retries);
bool joinNetwork();
bool send(const std::string text);


int main()
{
    int32_t ret;
    //Start LED startup sequence
    ledTick.attach(&ledTock, 0.1);

    printf("\r\n\r\n");
    printf("=====================================\r\n");
    printf("WoodStream LoRa Mousetrap Demo \r\n");
    printf("=====================================\r\n");
    printVersion();

    // get the mDot handle
    dot = mDot::getInstance();
    
    dot->setLogLevel(mts::MTSLog::INFO_LEVEL);

    // reset to default config so we know what state we're in
    dot->resetNetworkSession();
    dot->resetConfig();

    // set up the mDot with our network information
    setNetworkName(config_network_name);
    setNetworkPassphrase(config_network_pass);
    setFrequencySubBand(config_frequency_sub_band);
    setPower(20);    // Reduce latency for 868 units
    setAck(0);      // Disable ack for less latency

logInfo("Setting TX Spreading factor");
if ((ret = dot->setTxDataRate(mDot::SF_8)) != mDot::MDOT_OK) {
    logError("Failed To Set Tx Datarate %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
}

    while (!joinNetwork()) { wait(2); dot->resetNetworkSession(); }

    // Stop LED startup sequence & configure them for operation
    ledTick.detach();
    transmitLED = 1;
//    buttonPressLED = 1;

    // Configure timers
    periodicSendTick.attach(&periodicSendTock, 5*60);
    batteryvoltageMonitorTick.attach(&batteryvoltageMonitorTock, 0.15);

    // Setup Interrupt callback function
//    up_button.attach_deasserted(&up_button_callback);
//    down_button.attach_deasserted(&down_button_callback);

    // Start sampling buttons using interrupts
//    up_button.setSampleFrequency();
//    down_button.setSampleFrequency();

int old_mt_pwron = -1;
int old_mt_caught = -1;

while (1) {
    if (timeToReadBatteryVoltage) 
        
    if (mt_pwron != old_mt_pwron) {
        old_mt_pwron = mt_pwron;
        dataChanged = true;
    if (mt_caught != old_mt_caught) {
        old_mt_caught = mt_caught;
        dataChanged = true;
        }
       // printf("\n\r mt_pwron: %d, mt_caught: %d, batt_voltage: %f", old_mt_pwron, old_mt_caught, (batt_voltage*3.3));
       
       timeToReadBatteryVoltage = false;
}

        if (dataChanged) {
            char latestData[100];
            transmitLED = 1;
            //sprintf(latestData, "temp: %d,set: %d", temperature, sentSetPoint);
            sprintf(latestData, "power on: %d, kill status: %d, battery voltage: %f", old_mt_pwron, old_mt_caught, (batt_voltage*3.3));
            printf("%s\r\n", latestData);
            
            if (send(latestData)) {
                dataChanged = false;
            }
            transmitLED = 0;
        }
    }
}


void ledTock() {
    transmitLED = !transmitLED;
}

void periodicSendTock() {
    dataChanged = true;
}

void batteryvoltageMonitorTock() {
    timeToReadBatteryVoltage = true;

//    if (sentSetPoint != setPoint) {
        dataChanged = true;
    }

void printVersion()
{
    printf("%s\r\n\r\n", dot->getId().c_str());
}

bool setFrequencySubBand(uint8_t subBand)
{
    int32_t returnCode;
    printf("Setting frequency sub band to '%d'...\r\n", subBand);
    if ((returnCode = dot->setFrequencySubBand(subBand)) != mDot::MDOT_OK) {
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool setNetworkName(const std::string name)
{
    int32_t returnCode;
    printf("Setting network name to '%s'...\r\n", name.c_str());
    if ((returnCode = dot->setNetworkName(name)) != mDot::MDOT_OK)
    {
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool setNetworkPassphrase(const std::string passphrase)
{
    int32_t returnCode;
    printf("Setting passphrase to '%s'...\r\n", passphrase.c_str());
    if ((returnCode = dot->setNetworkPassphrase(passphrase)) != mDot::MDOT_OK)
    {
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool setPower(uint8_t power)
{
    int32_t returnCode;
    printf("Setting tx power to '%d'...\r\n", power);
    if ((returnCode = dot->setTxPower(power)) != mDot::MDOT_OK) {
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool joinNetwork()
{
    int32_t returnCode;
    printf("\r\nJoining network...\r\n");
    if ((returnCode = dot->joinNetworkOnce()) != mDot::MDOT_OK) {
        printError(dot, returnCode);
        return false;
    }
    printf("Network Joined!\r\n");
    return true;
}

bool setAck(uint8_t retries)
{
    int32_t returnCode;
    printf("Setting ack to '%d'...\r\n", retries);
    if ((returnCode = dot->setAck(retries)) != mDot::MDOT_OK)
    {
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool send(const std::string text)
{
    int32_t returnCode;
    uint32_t timeTillSend = dot->getNextTxMs();
    if (timeTillSend != 0) {
        printf("waiting %lu ms to send\r\n", timeTillSend);
        return false;
    }

    printf("Sending data...  ");
    std::vector<uint8_t> data(text.begin(), text.end());
    if ((returnCode = dot->send(data, 1)) != mDot::MDOT_OK)
    {
        printError(dot, returnCode);
        return false;
    }
    printf("Data sent!\r\n");
    return true;
}

void printError(mDot* dot, int32_t returnCode)
{
    std::string error = mDot::getReturnCodeString(returnCode) + " - " + dot->getLastError();
    printf("%s\r\n", error.c_str());
 }
