/* mbed Zumo motor control Library
 * Copyright (c) 2010-2013 jksoft
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef ZUMOCONTROL_H
#define ZUMOCONTROL_H

#include "mbed.h"

#define PERIOD            100.0f     // Hz
#define MINPULSWIDTH     1000.0f    // us

/** Zumo motor control class
 *
 * Example:
 * @code
 * // Drive the Zumo forward, turn left, back, turn right, at half speed for half a second

   #include "mbed.h"
   #include "ZumoControl.h"

   ZumoControl zumo_ctrl;

   int main() {

     wait(0.5);

     zumo_ctrl.forward(0.5);
     wait (0.5);
     zumo_ctrl.left(0.5);
     wait (0.5);
     zumo_ctrl.backward(0.5);
     wait (0.5);
     zumo_ctrl.right(0.5);
     wait (0.5);

     zumo_ctrl.stop();

 }
 * @endcode
 */
class ZumoControl {

    // Public functions
public:

    /** Create the Zumo object connected to the default pins
     *
     * @param rc GPIO pin used for Right motor control. Default is PTC9
     * @param rp GPIO pin used for Right motor PWM. Default is PTD5
     * @param lc GPIO pin used for Left motor control. Default is PTA13
     * @param lp GPIO pin used for Left motor PWM. Default is PTD0
     */
    ZumoControl();


    /** Create the Zumo object connected to specific pins
     *
     */
    ZumoControl(PinName rc,PinName rp,PinName lc,PinName lp);

    /** Directly control the speed and direction of the left motor
     *
     * @param speed A normalised number -1.0 - 1.0 represents the full range.
     */
    void left_motor (float speed);

    /** Directly control the speed and direction of the right motor
     *
     * @param speed A normalised number -1.0 - 1.0 represents the full range.
     */
    void right_motor (float speed);

    /** Drive both motors forward as the same speed
     *
     * @param speed A normalised number 0 - 1.0 represents the full range.
     */
    void forward (float speed);

    /** Drive both motors backward as the same speed
     *
     * @param speed A normalised number 0 - 1.0 represents the full range.
     */
    void backward (float speed);

    /** Drive left motor backwards and right motor forwards at the same speed to turn on the spot
     *
     * @param speed A normalised number 0 - 1.0 represents the full range.
     */
    void left (float speed);

    /** Drive left motor forward and right motor backwards at the same speed to turn on the spot
     * @param speed A normalised number 0 - 1.0 represents the full range.
     */
    void right (float speed);

    /** Stop both motors
     *
     */
    void stop (void);


private :
    
    void cycle(void);

    DigitalOut _rc;
    DigitalOut _rp;
    DigitalOut _lc;
    DigitalOut _lp;
    
    int pwm_set_r,pwm_set_l;
    int pwm_count,pwm_count_max;
    int value_r,value_l;
    
    Ticker _pwm;
};

#endif