/* Copyright (c) 2011, mbed
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *  
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *  
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

// Simple decode class for Wii Remote data

#include "Wiimote.h"

// Wii Report Formats: 
//  * http://wiibrew.org/wiki/Wiimote#0x37:_Core_Buttons_and_Accelerometer_with_10_IR_bytes_and_6_Extension_Bytes
//  * http://wiki.wiimoteproject.com/Reports#0x37_BTN_.2B_XLR_.2B_IR_.2B_6_EXT

// Input Report 0x37:
// 
// 0 [ ?    |    X(1:0)   | PLUS  | UP | DOWN | RIGHT | LEFT ]
// 1 [ HOME | Z(1) | Y(1) | MINUS |  A |   B  |  ONE  |  TWO ]
// 2 [                        X(9:2)                         ]
// 3 [                        Y(9:2)                         ]
// 4 [                        Z(9:2)                         ]
//

#include <stdio.h>
#include <math.h>

#define WII_ZERO 0x206
#define WII_1G   0x6A

void Wiimote::decode(char* data) {

    // read buttons
    left  = (data[0] >> 0) & 0x1;
    right = (data[0] >> 1) & 0x1;
    down  = (data[0] >> 2) & 0x1;
    up    = (data[0] >> 3) & 0x1;
    plus  = (data[0] >> 4) & 0x1;
    two   = (data[1] >> 0) & 0x1;
    one   = (data[1] >> 1) & 0x1;
    b     = (data[1] >> 2) & 0x1;
    a     = (data[1] >> 3) & 0x1;
    minus = (data[1] >> 4) & 0x1;
    home  = (data[1] >> 7) & 0x1;
    
    // read accelerometers
    rawx = (data[2] << 2) | (data[0] & 0x60) >> 5;
    rawy = (data[3] << 2) | (data[1] & 0x20) >> 4;
    rawz = (data[4] << 2) | (data[1] & 0x40) >> 5;

    // calculate accelerometer gravity
    x = (float)((int)rawx - WII_ZERO) / (float)WII_1G;
    y = (float)((int)rawy - WII_ZERO) / (float)WII_1G;
    z = (float)((int)rawz - WII_ZERO) / (float)WII_1G;

    // calculate wheel angle    
    wheel = atan2(-y, -x) * (180.0 / 3.141592);
} 

void Wiimote::dump() {
    printf("%d%d%d%d%d%d%d%d%d%d%d %.3f %.3f %.3f %.2f\n", left, right, down, up, plus, two, one, b, a, minus, home, x, y, z, wheel);
}


// Accelerometer data
//      y-
//     +---+               +--+
//     | + |               +  B
//     | A |               A  |
//     |   |               |  |
// x+  |   |  x-       z+  |  |  z-
//     | 1 |               1  |
//     | 2 |               2  |
//     +---+               +--+
//      y+
//
// x+ 0x19B
// x0 0x205
// x- 0x26E
//
// y+ 0x19C   0x6A
// y0 0x206
// y- 0x26E   0x68
//
// z+ 0x19C 
// z0 0x208
// z- 0x26E
