#include "mbed.h"
#include <btstack/hci_cmds.h>
#include <btstack/run_loop.h>
#include <btstack/sdp_util.h>
#include "hci.h"
#include "l2cap.h"
#include "btstack_memory.h"
#include "remote_device_db.h"
#include "rfcomm.h"
#include "sdp.h"
#include "config.h"
#include "debug.h"
#include "bd_addr.h"  // class bd_addr
#include "att.h"

#include "TB6612.h"
#include "HighSpeedAnalogIn.h"

Serial pc(USBTX, USBRX);
DigitalOut led[] = { (LED1), (LED2), (LED3), (LED4) };
DigitalIn sw[] = { (p29),(p30) };
HighSpeedAnalogIn ain(p15, p16, p17, p18);
TB6612 right(p21,p12,p11);
TB6612 left(p22,p14,p13);

static att_connection_t att_connection;
static uint16_t         att_response_handle = 0;
static uint16_t         att_response_size   = 0;
static uint8_t          att_response_buffer[28];

uint8_t    connection_status=0;

static void att_try_respond(void){
    if (!att_response_size) return;
    if (!att_response_handle) return;
    if (!hci_can_send_packet_now(HCI_ACL_DATA_PACKET)) return;
    
    // update state before sending packet
    uint16_t size = att_response_size;
    att_response_size = 0;
    l2cap_send_connectionless(att_response_handle, L2CAP_CID_ATTRIBUTE_PROTOCOL, att_response_buffer, size);
}


static void att_packet_handler(uint8_t packet_type, uint16_t handle, uint8_t *packet, uint16_t size){
    if (packet_type != ATT_DATA_PACKET) return;
    
    att_response_handle = handle;
    att_response_size = att_handle_request(&att_connection, packet, size, att_response_buffer);
    att_try_respond();
}

// enable LE, setup ADV data
static void packet_handler (void * connection, uint8_t packet_type, uint16_t channel, uint8_t *packet, uint16_t size){
    static bd_addr_t addr;
//    uint8_t adv_data[] = { 02, 01, 05,   03, 02, 0xf0, 0xff }; 
    const uint8_t adv_data[31]="\x02\x01\x05" "\x06\x09mbed";
    switch (packet_type) {
            
        case HCI_EVENT_PACKET:
            switch (packet[0]) {
                
                case BTSTACK_EVENT_STATE:
                    // bt stack activated, get started - set local name
                    if (packet[2] == HCI_STATE_WORKING) {
                       log_info("Working!\n");
                        hci_send_cmd(&hci_read_local_supported_features);
                    }
                    break;
                
                case DAEMON_EVENT_HCI_PACKET_SENT:
                    att_try_respond();
                    break;
                    
                case HCI_EVENT_LE_META:
                    switch (packet[2]) {
                        case HCI_SUBEVENT_LE_CONNECTION_COMPLETE:
                            // reset connection MTU
                            att_connection.mtu = 23;
                            break;
                        default:
                            break;
                    }
                    break;

                case BTSTACK_EVENT_NR_CONNECTIONS_CHANGED:
                    if (packet[2]) {
                        connection_status=1;
                        log_info("CONNECTED\n");
                        led[0] = 1;
                    } else {
                        connection_status=0;
                        log_info("NOT CONNECTED\n");
                        led[0] = 0;
                        right = 0;
                        left = 0;
                    }
                    break;
                    
                case HCI_EVENT_DISCONNECTION_COMPLETE:
                    att_response_handle =0;
                    att_response_size = 0;
                    
                    // restart advertising
                    hci_send_cmd(&hci_le_set_advertise_enable, 1);
                    break;
                    
                case HCI_EVENT_COMMAND_COMPLETE:
                    if (COMMAND_COMPLETE_EVENT(packet, hci_read_bd_addr)){
                        bt_flip_addr(addr, &packet[6]);
                        log_info("BD ADDR: %s\n", bd_addr_to_str(addr));
                        break;
                    }
                    if (COMMAND_COMPLETE_EVENT(packet, hci_read_local_supported_features)){
                        log_info("Local supported features: %04lX%04lX\n", READ_BT_32(packet, 10), READ_BT_32(packet, 6));
                        hci_send_cmd(&hci_set_event_mask, 0xffffffff, 0x20001fff);
                        break;
                    }
                    if (COMMAND_COMPLETE_EVENT(packet, hci_set_event_mask)){
                        hci_send_cmd(&hci_write_le_host_supported, 1, 1);
                        break;
                    }
                    if (COMMAND_COMPLETE_EVENT(packet, hci_write_le_host_supported)){
                        hci_send_cmd(&hci_le_set_event_mask, 0xffffffff, 0xffffffff);
                        break;
                    }
                    if (COMMAND_COMPLETE_EVENT(packet, hci_le_set_event_mask)){
                        hci_send_cmd(&hci_le_read_buffer_size);
                        break;
                    }
                    if (COMMAND_COMPLETE_EVENT(packet, hci_le_read_buffer_size)){
                        log_info("LE buffer size: %u, count %u\n", READ_BT_16(packet,6), packet[8]);
                       hci_send_cmd(&hci_le_read_supported_states);
                       break;
                    }
                    if (COMMAND_COMPLETE_EVENT(packet, hci_le_read_supported_states)){
                       hci_send_cmd(&hci_le_set_advertising_parameters,  0x0400, 0x0800, 0, 0, 0, &addr, 0x07, 0);
                       break;
                    }
                    if (COMMAND_COMPLETE_EVENT(packet, hci_le_set_advertising_parameters)){
                       hci_send_cmd(&hci_le_set_advertising_data, sizeof(adv_data), adv_data);
                       break;
                    }
                    if (COMMAND_COMPLETE_EVENT(packet, hci_le_set_advertising_data)){
                       hci_send_cmd(&hci_le_set_scan_response_data, 10, adv_data);
                       break;
                    }
                    if (COMMAND_COMPLETE_EVENT(packet, hci_le_set_scan_response_data)){
                       hci_send_cmd(&hci_le_set_advertise_enable, 1);
                       break;
                    }
                    if (COMMAND_COMPLETE_EVENT(packet, hci_le_set_advertise_enable)){
                        hci_discoverable_control(1);
                        log_info("startup_state=1\n");
                        //startup_state=1;
                        led[1] = 0;
                        break;
                    }
                    
            }
    }
}

// test profile
#include "profile.h"

static char    strbuf[80];
static uint8_t    ledvalue;

// read requests
static uint16_t att_read_callback(uint16_t handle, uint16_t offset, uint8_t * buffer, uint16_t buffer_size){
    uint16_t    ret=0,val;

    if(buffer){
     //   log_info("READ Callback, handle %04x\n", handle);
    }
    switch(handle){
       case 0x000b:
            ret=strlen((char*)strbuf);
            if(buffer && ret<=buffer_size){
                log_info("Read text: %s\n", strbuf);
                memcpy(buffer,strbuf,ret);
            }
            break;
        case 0x000d:
            if(buffer && buffer_size>=8){
                unsigned short dat[4];
                dat[0] = ain.read_u16(p15);
                dat[1] = ain.read_u16(p16);
                dat[2] = ain.read_u16(p17);
                dat[3] = ain.read_u16(p18);
                memcpy( buffer , dat , 8 );
            }
            ret = 8;
            break;
        case 0x000f:
            if(buffer && buffer_size>=1){
             //   log_info("Read value: %u\n", val);
                buffer[0]=(!sw[0])|((!sw[1])<<1);
            }
            ret=1;
            break;
    }
    return ret;
}

// write requests
static void att_write_callback(uint16_t handle, uint16_t transaction_mode, uint16_t offset, uint8_t *buffer, uint16_t buffer_size, signature_t * signature){
 //   log_info("WRITE Callback, handle %04x\n", handle);
    
    switch(handle){
        case 0x000b:
            signed char r = buffer[0];
            signed char l = buffer[1];
            
            right = r;
            left = l;

            break;
        case 0x000d:
          //  log_info("New value: %u\n", buffer[0]);
            ledvalue=buffer[0];
            if (buffer[0]){
                led[1] = 1;
            } else {
                led[1] = 0;
            }
            break;
    }
}

// main
int main(void)
{
    sw[0].mode(PullUp);
    sw[1].mode(PullUp);
    
    pc.baud(921600);
    log_info("%s\n", __FILE__);
    
    led[0] = 0;
    
    /// GET STARTED with BTstack ///
    btstack_memory_init();
    run_loop_init(RUN_LOOP_EMBEDDED);
    
    // init HCI
    hci_transport_t* transport = hci_transport_usb_instance();
    remote_device_db_t * remote_db = (remote_device_db_t *) &remote_device_db_memory;
    hci_init(transport, NULL, NULL, remote_db);
    
    // init L2CAP
    l2cap_init();
    l2cap_register_fixed_channel(att_packet_handler, L2CAP_CID_ATTRIBUTE_PROTOCOL);
    l2cap_register_packet_handler(packet_handler);
    
    // set up ATT
    att_set_db(profile_data);
    att_set_write_callback(att_write_callback);
    att_set_read_callback(att_read_callback);
    att_dump_attributes();
    att_connection.mtu = 27;

     // turn on!
    hci_power_control(HCI_POWER_ON);

    // go!
    run_loop_execute();    

    return 0;
}
