/**
 * RemoteIR library - Test program.
 *
 * Copyright (C) 2010 Shinichiro Nakamura (CuBeatSystems)
 * http://shinta.main.jp/
 */

#include <mbed.h>

#include "ReceiverIR.h"
#include "TransmitterIR.h"

#define TEST_LOOP_BACK  0

ReceiverIR ir_rx(PA_2);
TransmitterIR ir_tx(PF_0);
Serial pc(USBTX,USBRX);

/**
 * Receive.
 *
 * @param format Pointer to a format.
 * @param buf Pointer to a buffer.
 * @param bufsiz Size of the buffer.
 *
 * @return Bit length of the received data.
 */
int receive(RemoteIR::Format *format, uint8_t *buf, int bufsiz, int timeout = 100) {
    int cnt = 0;
    while (ir_rx.getState() != ReceiverIR::Received) {
        cnt++;
        if (timeout < cnt) {
            return -1;
        }
    }
    return ir_rx.getData(format, buf, bufsiz * 8);
}

/**
 * Transmit.
 *
 * @param format Format.
 * @param buf Pointer to a buffer.
 * @param bitlength Bit length of the data.
 *
 * @return Bit length of the received data.
 */
int transmit(RemoteIR::Format format, uint8_t *buf, int bitlength, int timeout = 100) {
    int cnt = 0;
    while (ir_tx.getState() != TransmitterIR::Idle) {
        cnt++;
        if (timeout < cnt) {
            return -1;
        }
    }
    return ir_tx.setData(format, buf, bitlength);
}

/**
 * Display a current status.
 */
void display_status(char *status, int bitlength) {
    pc.printf("%-5.5s:%02d\r\n", status, bitlength);
}

/**
 * Display a format of a data.
 */
void display_format(RemoteIR::Format format) {

    switch (format) {
        case RemoteIR::UNKNOWN:
            pc.printf("????????\r\n");
            break;
        case RemoteIR::NEC:
            pc.printf("NEC     \r\n");
            break;
        case RemoteIR::NEC_REPEAT:
            pc.printf("NEC  (R)\r\n");
            break;
        case RemoteIR::AEHA:
            pc.printf("AEHA    \r\n");
            break;
        case RemoteIR::AEHA_REPEAT:
            pc.printf("AEHA (R)\r\n");
            break;
        case RemoteIR::SONY:
            pc.printf("SONY    \r\n");
            break;
    }
}

/**
 * Display a data.
 *
 * @param buf Pointer to a buffer.
 * @param bitlength Bit length of a data.
 */
void display_data(uint8_t *buf, int bitlength) {
    const int n = bitlength / 8 + (((bitlength % 8) != 0) ? 1 : 0);
    for (int i = 0; i < n; i++) {
        pc.printf("%02X", buf[i]);
    }
    for (int i = 0; i < 8 - n; i++) {
        pc.printf("--");
    }
}

/**
 * Entry point.
 */
int main(void) {


    /*
     * Splash.
     */
    pc.printf("RemoteIR        \r\n");
    pc.printf("Program example.\r\n");
    wait(3);

    /*
     * Initialize.
     */

    pc.printf("Press a button  \r\n");
    pc.printf("on a controller.\r\n");

    /*
     * Execute.
     */
    while (1) {
        uint8_t buf1[32];
        uint8_t buf2[32];
        int bitlength1;
        int bitlength2;
        RemoteIR::Format format;

        memset(buf1, 0x00, sizeof(buf1));
        memset(buf2, 0x00, sizeof(buf2));

        {
            bitlength1 = receive(&format, buf1, sizeof(buf1));
            if (bitlength1 < 0) {
                continue;
            }
            display_status("RECV", bitlength1);
            display_data(buf1, bitlength1);
            display_format(format);
        }

#if TEST_LOOP_BACK
        wait_ms(100);

        {
            bitlength1 = transmit(format, buf1, bitlength1);
            if (bitlength1 < 0) {
                continue;
            }
            display_status("TRAN", bitlength1);
            display_data(buf1, bitlength1);
            display_format(format);
        }

        wait_ms(100);

        {
            bitlength2 = receive(&format, buf2, sizeof(buf2));
            if (bitlength2 < 0) {
                continue;
            }
            display_status("RECV", bitlength2);
            display_data(buf2, bitlength2);
            display_format(format);
        }

        wait_ms(100);

        {
            for (int i = 0; i < sizeof(buf1); i++) {
                if (buf1[i] != buf2[i]) {
                    display_status("CPERR", bitlength2);
                    wait(1);
                    continue;
                }
            }
        }
#endif
    }
}