// Sabertooth 2x25 V2 motor driver
// Simplified Serial Mode
#include "mbed.h"
#include "QEI.h"
#include <ros.h>
#include <std_msgs/Int32.h>

#define M_2PI (2*3.14)

//#define SERIAL_COM_MODE  // define this if you use serial communication, not rosserial
#define ENCODER_ON  // define this if you use encoders
#define FEEDBACK_ON  // define FEEDBACK_ON and ENCODER_ON if you use feedback

//// parameters from ROS parameter server
//int CONTROL_MODE = 0;  // 0: direct command, 1: velocity feedback

// serial
Serial out_saber(p9, p10);
#if defined(SERIAL_COM_MODE)
Serial pc(USBTX, USBRX);
#endif  // SERIAL_COM_MODE

// control period (ms)
const int CONTROL_PERIOD = 10;

// event queue
EventQueue queue;

// motor -----------------------------------------------------------------------
// gear ratio
const int GEAR_RATIO[2] = { 1, 1 };  // between encoder and driven object

// motor command
// 0: shutdown both motors
// for motor 0 (1 ~ 127)
//   1: reverse at max speed
//  64: stop
// 127: forward at max speed
//
// for motor 1 (128 ~ 255)
// 128: reverse at max speed
// 192: stop
// 255: forward at max speed
const int MAX_MOTOR_COMMAND[2] = { 127, 255 };
const int NEUTRAL_MOTOR_COMMAND[2] = { 64, 192 };
const int MIN_MOTOR_COMMAND[2] = { 1, 128 };
int motor_command[2] = { 64, 192 };

// send motor command
void send_motor_command(int command[]) {
    out_saber.putc(command[0]);
    wait_us(5);
    out_saber.putc(command[1]);
    wait_us(5);
}

// init motor drivers
void init_motor_drivers() {
    out_saber.putc(0);  // shutdown both motors
    motor_command[0] = NEUTRAL_MOTOR_COMMAND[0];
    motor_command[1] = NEUTRAL_MOTOR_COMMAND[1];
    send_motor_command(motor_command);  // stop both motors
}

// limit motor command
void limit_motor_command(int command[]) {
    // motor 0
    if (command[0] > 127) command[0] = MAX_MOTOR_COMMAND[0];
    if (command[0] <   1) command[0] = MIN_MOTOR_COMMAND[0];
    // motor 1
    if (command[1] > 255) command[1] = MAX_MOTOR_COMMAND[1];
    if (command[1] < 128) command[1] = MIN_MOTOR_COMMAND[1];
}

#if !defined(SERIAL_COM_MODE)
// current motor command publisher
std_msgs::Int32 motor_command_ch0_current_msgs;
ros::Publisher motor_command_ch0_current_pub("motor_command_ch0_current", &motor_command_ch0_current_msgs);
std_msgs::Int32 motor_command_ch1_current_msgs;
ros::Publisher motor_command_ch1_current_pub("motor_command_ch1_current", &motor_command_ch1_current_msgs);

#if !defined(FEEDBACK_ON)
// motor command subscriber
void message_motor_command_ch0(const std_msgs::Int32& _motor_command) {
    motor_command[0] = _motor_command.data;
}
ros::Subscriber<std_msgs::Int32> motor_command_ch0_sub("motor_command_ch0", &message_motor_command_ch0);
void message_motor_command_ch1(const std_msgs::Int32& _motor_command) {
    motor_command[1] = _motor_command.data;
}
ros::Subscriber<std_msgs::Int32> motor_command_ch1_sub("motor_command_ch1", &message_motor_command_ch1);
#endif  // FEEDBACK_ON
#endif  // SERIAL_COM_MODE
// -----------------------------------------------------------------------------

// encoder ---------------------------------------------------------------------
#if defined(ENCODER_ON)
const int RES_ENC[2] = { 2000, 2000 };  // resolution of encoder
const int MULTIPLY_ENC[2] = { 4, 4 };   // multiplication of encoder pulse count

QEI enc0(p5, p6, NC, RES_ENC[0], QEI::X4_ENCODING);
QEI enc1(p7, p8, NC, RES_ENC[0], QEI::X4_ENCODING);

int enc_pulse_count_rate[2] = { 0, 0 };
int enc_pulse_count[2] = { 0, 0 };

#if !defined(SERIAL_COM_MODE)
// encoder pulse count rate publisher
std_msgs::Int32 enc_pulse_count_rate_ch0_current_msgs;
ros::Publisher enc_pulse_count_rate_ch0_current_pub("enc_pulse_count_rate_ch0_current", &enc_pulse_count_rate_ch0_current_msgs);
std_msgs::Int32 enc_pulse_count_rate_ch1_current_msgs;
ros::Publisher enc_pulse_count_rate_ch1_current_pub("enc_pulse_count_rate_ch1_current", &enc_pulse_count_rate_ch1_current_msgs);

// encoder pulse count publisher
//std_msgs::Int32 enc_pulse_count_ch0_current_msgs;
//ros::Publisher enc_pulse_count_ch0_current_pub("enc_pulse_count_ch0_current", &enc_pulse_count_ch0_current_msgs);
//std_msgs::Int32 enc_pulse_count_ch1_current_msgs;
//ros::Publisher enc_pulse_count_ch1_current_pub("enc_pulse_count_ch1_current", &enc_pulse_count_ch1_current_msgs);
#endif  // SERIAL_COM_MODE
#endif  // ENCODER_ON
// -----------------------------------------------------------------------------

// feedback_control ------------------------------------------------------------
#if defined(ENCODER_ON) && defined(FEEDBACK_ON)
int desired_enc_pulse_count_rate[2] = { 0, 0 };  // desired encoder pulse count rate

// current desired encoder pulse count rate publisher
std_msgs::Int32 desired_enc_pulse_count_rate_ch0_current_msgs;
ros::Publisher desired_enc_pulse_count_rate_ch0_current_pub("desired_enc_pulse_count_rate_ch0_current", &desired_enc_pulse_count_rate_ch0_current_msgs);
std_msgs::Int32 desired_enc_pulse_count_rate_ch1_current_msgs;
ros::Publisher desired_enc_pulse_count_rate_ch1_current_pub("desired_enc_pulse_count_rate_ch1_current", &desired_enc_pulse_count_rate_ch1_current_msgs);

// desired encoder pulse count rate subscriber
void message_desired_enc_pulse_count_rate_ch0(const std_msgs::Int32& _desired_enc_pulse_count_rate) {
    desired_enc_pulse_count_rate[0] = _desired_enc_pulse_count_rate.data;
}
ros::Subscriber<std_msgs::Int32> desired_enc_pulse_count_rate_ch0_sub("desired_enc_pulse_count_rate_ch0", &message_desired_enc_pulse_count_rate_ch0);
void message_desired_enc_pulse_count_rate_ch1(const std_msgs::Int32& _desired_enc_pulse_count_rate) {
    desired_enc_pulse_count_rate[1] = _desired_enc_pulse_count_rate.data;
}
ros::Subscriber<std_msgs::Int32> desired_enc_pulse_count_rate_ch1_sub("desired_enc_pulse_count_rate_ch1", &message_desired_enc_pulse_count_rate_ch1);

// PI gains obtained from ROS parameter server
float Kp_v;
float Ki_v;    

void feedback_control() {
    // encoder pulse count rate error
    int enc_pulse_count_rate_error[2];
    
    // accumulated error
    static int accumulated_enc_pulse_count_rate_error[2] = { 0, 0 };
    int temp_accumulated_enc_pulse_count_rate_error[2];
    
    for (int i = 0; i < 2; i++) {
        enc_pulse_count_rate_error[i] = desired_enc_pulse_count_rate[i] - enc_pulse_count_rate[i];
        
        temp_accumulated_enc_pulse_count_rate_error[i] = accumulated_enc_pulse_count_rate_error[i];
        accumulated_enc_pulse_count_rate_error[i] += enc_pulse_count_rate_error[i];

        motor_command[i] = (int)(Kp_v * enc_pulse_count_rate_error[i] + Ki_v * accumulated_enc_pulse_count_rate_error[i]);
        motor_command[i] += NEUTRAL_MOTOR_COMMAND[i];
        
        if (motor_command[i] > MAX_MOTOR_COMMAND[i]) {
            motor_command[i] = MAX_MOTOR_COMMAND[i];
            accumulated_enc_pulse_count_rate_error[i] = temp_accumulated_enc_pulse_count_rate_error[i];
        }
        
        if (motor_command[i] < MIN_MOTOR_COMMAND[i]) {
            motor_command[i] = MIN_MOTOR_COMMAND[i];
            accumulated_enc_pulse_count_rate_error[i] = temp_accumulated_enc_pulse_count_rate_error[i];
        }
    }
}
#endif  // ENCODER_ON && FEEDBACK_ON
// -----------------------------------------------------------------------------

// ROS -------------------------------------------------------------------------
#if !defined(SERIAL_COM_MODE)
ros::NodeHandle nh;
#endif  // SERIAL_COM_MODE
//------------------------------------------------------------------------------

// publish messages (callback) --------------------------------------------------------
#if !defined(SERIAL_COM_MODE)
void publish_msgs() {
    motor_command_ch0_current_msgs.data = motor_command[0];
    motor_command_ch0_current_pub.publish(&motor_command_ch0_current_msgs);
    motor_command_ch1_current_msgs.data = motor_command[1];
    motor_command_ch1_current_pub.publish(&motor_command_ch1_current_msgs);
#if defined(ENCODER_ON)
    enc_pulse_count_rate_ch0_current_msgs.data = enc_pulse_count_rate[0];
    enc_pulse_count_rate_ch0_current_pub.publish(&enc_pulse_count_rate_ch0_current_msgs);
    enc_pulse_count_rate_ch1_current_msgs.data = enc_pulse_count_rate[1];
    enc_pulse_count_rate_ch1_current_pub.publish(&enc_pulse_count_rate_ch1_current_msgs);
//    enc_pulse_count_ch0_current_msgs.data = enc_pulse_count[0];
//    enc_pulse_count_ch0_current_pub.publish(&enc_pulse_count_ch0_current_msgs);
//    enc_pulse_count_ch1_current_msgs.data = enc_pulse_count[1];
//    enc_pulse_count_ch1_current_pub.publish(&enc_pulse_count_ch1_current_msgs);
#if defined(FEEDBACK_ON)
    desired_enc_pulse_count_rate_ch0_current_msgs.data = desired_enc_pulse_count_rate[0];
    desired_enc_pulse_count_rate_ch0_current_pub.publish(&desired_enc_pulse_count_rate_ch0_current_msgs);
    desired_enc_pulse_count_rate_ch1_current_msgs.data = desired_enc_pulse_count_rate[1];
    desired_enc_pulse_count_rate_ch1_current_pub.publish(&desired_enc_pulse_count_rate_ch1_current_msgs);
#endif  // FEEDBACK_ON
#endif  // ENCODER_ON
    nh.spinOnce();
}
#endif  // SERIAL_COM_MODE
// -----------------------------------------------------------------------------

// control motor (callback) ----------------------------------------------------
void control_motor() {
#if defined(ENCODER_ON)
    // update encoder pulse count rate
    enc_pulse_count_rate[0] = enc0.getPulses();
    enc0.reset();
    enc_pulse_count_rate[1] = enc1.getPulses();
    enc1.reset();
#endif  // ENCODER_ON
    
    // update motor command
    limit_motor_command(motor_command);
    send_motor_command(motor_command);

#if defined(ENCODER_ON)
    // accumurate encoder pulse count rate
    enc_pulse_count[0] += enc_pulse_count_rate[0];
    enc_pulse_count[1] += enc_pulse_count_rate[1];
#endif  // ENCODER_ON
    
#if !defined(SERIAL_COM_MODE)
#if defined(ENCODER_ON)
    feedback_control();
#endif  // ENCODER_ON
#else
    switch (pc.getc()) {
        case 's':
            motor_command[0] = NEUTRAL_MOTOR_COMMAND[0];
            break;
        case 'a':
            motor_command[0] += 10;
            break;
        case 'd':
            motor_command[0] -= 10;
            break;
        case 'k':
            motor_command[1] = NEUTRAL_MOTOR_COMMAND[1];
            break;
        case 'j':
            motor_command[1] += 10;
            break;
        case 'l':
            motor_command[1] -= 10;
            break;
        default:
            ;
    }

#if defined(ENCODER_ON)
    pc.printf("%d, %d, %d, %d, %d, %d\n", motor_command[0], motor_command[1], enc_pulse_count_rate[0], enc_pulse_count_rate[1], enc_pulse_count[0], enc_pulse_count[1]);
#else
    pc.printf("%d, %d\n", motor_command[0], motor_command[1]);

#endif  // ENCODER_ON
    
#endif  // SERIAL_COM_MODE
}
// -----------------------------------------------------------------------------

int main() {
    // initialize serials
    out_saber.baud(38400);
#if defined(SERIAL_COM_MODE)
    pc.baud(9600);
#else  // ROS
//    nh.getHardware()->setBaud(38400);  // default: 57600
#endif  // SERIAL_COM_MODE

    // initialize motor drivers and ROS
    init_motor_drivers();
#if defined(SERIAL_COM_MODE)
    pc.printf("initialized motor driver\n");    
#else  // ROS
    nh.initNode();
    nh.advertise(motor_command_ch0_current_pub);
    nh.advertise(motor_command_ch1_current_pub);
#if !defined(FEEDBACK_ON)
    nh.subscribe(motor_command_ch0_sub);
    nh.subscribe(motor_command_ch1_sub);
#endif  // FEEDBACK_ON
#if defined(ENCODER_ON)
    nh.advertise(enc_pulse_count_rate_ch0_current_pub);
    nh.advertise(enc_pulse_count_rate_ch1_current_pub);
//    nh.advertise(enc_pulse_count_ch0_current_pub);
//    nh.advertise(enc_pulse_count_ch1_current_pub);
#if defined(FEEDBACK_ON)
    nh.advertise(desired_enc_pulse_count_rate_ch0_current_pub);
    nh.advertise(desired_enc_pulse_count_rate_ch1_current_pub);
    nh.subscribe(desired_enc_pulse_count_rate_ch0_sub);
    nh.subscribe(desired_enc_pulse_count_rate_ch1_sub);
    
    // obtain PI gains from ROS parameter server
    if (!nh.getParam("/dcmotor_control_sabertooth_rosserial/Kp_v", &Kp_v, 1)) Kp_v = 0.0;
    if (!nh.getParam("/dcmotor_control_sabertooth_rosserial/Ki_v", &Ki_v, 1)) Ki_v = 0.0;
#endif  // FEEDBACK_ON
#endif  // ENCODER_ON
#endif  // SERIAL_COM_MODE
    
    // event queues
    queue.call_every(CONTROL_PERIOD, &control_motor);
#if !defined(SERIAL_COM_MODE)
    queue.call_every(100, &publish_msgs);  // period of message publishing: 100 ms
#endif  // SERIAL_COM_MODE
    queue.dispatch();
}
