#include "mbed.h"
#include "mDot.h"
#include "MTSLog.h"
#include "MTSText.h"
#include <string>
#include <vector>

using namespace mts;

/* ********************************************** */
/* Create AppEUI identifier and Assigned AppKey   */
/* ********************************************** */

static const uint8_t DEVKEY[16] = { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF }; // Normal
static const uint8_t APPEUI[8]  = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00  }; // SMTC AppEUI - Most Significant Byte First
static uint8_t       config_frequency_sub_band = 0; // 0 = Enable all channels, 1 = 1st eight channels

/* ********************************************** */

static std::vector<uint8_t> AppKey (DEVKEY, DEVKEY + sizeof(DEVKEY)/sizeof(uint8_t) );
static std::vector<uint8_t> AppEUI (APPEUI, APPEUI + sizeof(APPEUI)/sizeof(uint8_t) ); 

InterruptIn button(PB_1);
DigitalOut  drive_high(PB_0);
DigitalIn   door_open(PA_7);

volatile bool msg_rdy;

void queue_message(){
    msg_rdy = true;
}
    

int main() {
    Serial debug(USBTX, USBRX);
    debug.baud(460800);
    drive_high = 1;
    
    msg_rdy = false;
    int32_t ret;
    int32_t next_tx;
    int32_t wait_time = 2;
    mDot* dot;
    std::vector<uint8_t> send_data;
    std::vector<uint8_t> recv_data;
    uint8_t recv = 0;
    uint8_t recv_mismatch = 0;
    uint8_t send_failure = 0;
    uint8_t iterations = 50;
    button.rise(&queue_message);
    
    send_data.push_back(0x01);
    send_data.push_back(0x02);

    dot = mDot::getInstance();

    dot->resetConfig();
    
    dot->setLogLevel(MTSLog::TRACE_LEVEL);
    //dot->setJoinByteOrder(1); // MSB
    dot->setTxDataRate(mDot::SF_10);
    //dot->setTxPower(20);
    
    while ((ret = dot->setFrequencySubBand(config_frequency_sub_band)) != mDot::MDOT_OK) {
        logError("failed to set frequency sub band: [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
    }
    while ((ret = dot->setPublicNetwork(true)) != mDot::MDOT_OK) {
        logError("failed to set Public Network: [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
    }
    /** Set network ID
     * for use with OTA & AUTO_OTA network join modes
     * setting network ID via this function sets network name to empty
     * @param id a vector of 8 bytes
     * @returns MDOT_OK if success
     */
    while ((ret = dot->setNetworkId(AppEUI)) != mDot::MDOT_OK) {
        logError("failed to set AppEUI: [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
    }
    while ((ret = dot->setNetworkKey(AppKey)) != mDot::MDOT_OK) {
        logError("failed to set AppEUI: [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
    }

    logInfo("enabling activity LED");
    dot->setActivityLedEnable(true);

    logInfo("joining network");
    while ((ret = dot->joinNetwork()) != mDot::MDOT_OK) {
        logError("failed to join network: [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
        wait_ms(dot->getNextTxMs() + 1);
    }
    logInfo("joined");

    dot->setAck(3); // Use Confirmed frames and try three times

    while (1) {

        send_data[0] = 192; //door_open;
        if ((ret = dot->send(send_data)) != mDot::MDOT_OK) {
            logError("failed to send: [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
            send_failure++;
        } else {
            logInfo("send data: %s", Text::bin2hexString(send_data).c_str());
            if ((ret = dot->recv(recv_data)) != mDot::MDOT_OK) {
                logError("failed to recv: [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
            } else {
                logInfo("recv data: %s", Text::bin2hexString(recv_data).c_str());
                if (recv_data[0] == 0x80) {
                    recv++;
                } else if (recv_data[0] == 0xFF) {
                    goto END;
                } else {
                    recv_mismatch++;
                }
            }
            recv_data.clear();
        }
        msg_rdy = true; // Set to 'true' for free running.
        wait(wait_time);
        while (!msg_rdy) wait(wait_time);
    }
END:    
    logInfo("Version: %s", dot->getId().c_str());
    logInfo("Recv: %d/%d", recv, iterations);
    logInfo("Recv Mismatch: %d/%d", recv_mismatch, iterations);
    logInfo("Send Failure: %d/%d", send_failure, iterations);
    logInfo("Dropped: %d/%d", iterations - (recv + recv_mismatch + send_failure), iterations);

    return 0;
}