#ifndef YCONP020_H
#define YCONP020_H

#include "Stream.h"
#include "Serial.h"
using namespace mbed;
#include <string>
#include <vector>
#include <queue>
using namespace std;
#include "GraphicsDisplay.h"
/// \class TextDisplay
/// \brief interface class for display text
///
/// Because it is derived from mbed::Stream, TextDisplay has function putc(),puts(),printf() and more


// for internal baffer
class YconP020GBuf : public GraphicsDisplay {
public:
    enum { MAX_X=200,MAX_Y=96 };
    YconP020GBuf(const char *name=NULL) : GraphicsDisplay(name) { setfontscale(); }
    virtual void pixel(int x, int y, int b);
    void setfontscale(uint8_t x=1, uint8_t y=1);
    void pixel1x1(int x, int y, int b);
    virtual void cls();
    uint8_t getbyte(size_t x,size_t y) const {return buffer[y][x];}
protected:
private:
    virtual int width() {return MAX_X;}
    virtual int height() {return MAX_Y;}
    uint8_t buffer[MAX_Y][MAX_X/8];
    uint8_t pixelsizex;
    uint8_t pixelsizey;
};

/// Y-con P020 Electric Paper Display
class YconP020 : public Stream {
static const size_t MAXRECBUFSIZE=1000;
public:
    typedef enum {
        NOERROR=0,FILEOPENERR=1,REGINDEXERR=2,FILENOTEXIST=3,DIROPENERR=4
    } error_t;
    
    /// @param UART PinName to Y-con P020
    YconP020(PinName tx, PinName rx);
 
    /// Similar to Serial::readable()
    bool readable() {return !recoutbuf.empty();}
    /// check ready to send command
    bool command_ready() const { return lastchar=='!'; }
    /// wait for ready to send command
    /// @return false timeout
    bool wait_command_ready(int ms=10000);
    /// change to command mode from normal(send file) mode
    /// @return flase timeout
    bool command_mode();

    //commands
    /// Y-con P020 command (1)?
    string help_str() {return get_stringparam("?");}
    /// Y-con P020 command (2)W, (3)B
    void clear_screen(bool white=true) {void_command(white? "W":"B");}
    /// Y-con P020 command (4)N
    void negative_screen() {void_command("N");}
    /// Y-con P020 command (5)D
    void dispstoredpict(int picnum) {set_param("D", picnum);} //(5)D
    /// Y-con P020 command (6)R
    error_t storepictfile(const string& filename, const int picnum);
    /// Y-con P020 command (7)I
    string infomation_str() {return get_stringparam("I");}
    /// Y-con P020 command (8)DEMO
    void start_demo() {void_command("DEMO",false);}    
    /// Y-con P020 command (8)DEMO stop (send ctrl+Z)
    void stop_demo();
    /// Y-con P020 command (9)INTERVAL
    int command_interval() {return get_intparam("INTERVAL","Interval");}
    /// Y-con P020 command (9)INTERVAL time
    void command_interval(int t) {set_param("INTERVAL", t);}
    /// Y-con P020 command (10)LASTWAIT
    int command_lastwait() {return get_intparam("LASTWAIT","Wait");}
    /// Y-con P020 command (10)LASTWAIT time
    void command_lastwait(int t) {return set_param("LASTWAIT", t);}
    /// Y-con P020 command (11)STANDBY
    bool command_standby() {return get_boolparam("STANDBY");}
    /// Y-con P020 command (11)STANDBY [0|1]
    void command_standby(bool f) {return set_param("STANDBY", f);}
    /// Y-con P020 command (12)LED
    bool command_led() {return get_boolparam("LED");}
    /// Y-con P020 command (12)LED [0|1]
    void command_led(bool f) {return set_param("LED", f);}
    /// Y-con P020 command (13)P37
    bool command_p37() {return get_boolparam("P37");}
    /// Y-con P020 command (13)P37 [0|1]
    void command_p37(bool f) {return set_param("P37", f);}
    /// Y-con P020 command (14)EXIT
    void command_exit() {void_command("EXIT",false);}
    /// Y-con P020 command (15)RESET
    void command_reset() {void_command("RESET",false);}
    /// Y-con P020 command (16)YSLAB
    string yslab_info() {return get_stringparam("YSLAB");}
        
    /// get file list of directories
    const vector<string>& getfilelist(const vector<string>& dirs);
    /// get file list of directory
    const vector<string>& getfilelist(const string& dir) {
        return getfilelist(vector<string>(1,dir));
    }
    /// return previous file list
    const vector<string>& getfilelist() const {return filelist;}
    /// get all file list
    const vector<string>& getfilelistall();
    /// display file
    error_t dispfile(const string& filename);

    /** output text to internal buffer. for example
     * @code
     * epd.text()->cls();
     * epd.setfontscale(2,3);
     * epd.text()->puts("Hello World\nYconP020\n");
     * epd.setfontscale();
     * epd.text()->printf("%d\n", epd.command_led());
     * epd.display_internalbuf();
     * @endcode
     */ 
    TextDisplay* text() {return &internalbuf;}
    /// set size of pixel
    void setfontscale(uint8_t x=1, uint8_t y=1) {internalbuf.setfontscale(x,y);}
    /// set pixel at internal buffer
    void pset(int x, int y, bool b=true) {internalbuf.pixel1x1(x,y,b);}
    /// clear internal buffer
    void clear_internalbuf() { internalbuf.cls(); }
    /// send internal buffer to Y-con P020
    void display_internalbuf();
    /// width of Y-con P020 (==200)
    int width() const { return YconP020GBuf::MAX_X; }
    /// height of Y-con P020 (==96)
    int height() const { return YconP020GBuf::MAX_Y; }
protected:       
    void uartint();
    void push_recbuf(char c);
    void push_recbuf(const string& s);
    
    error_t sendfile(const string& filename);

    //for command line intercept
    string localcomline;
    virtual int putc_intercept(int c);

    //sub-functions for functions of Y-con P020 command
    void send_command(const string& command, bool waitready=true);
    void void_command(const string& command, bool waitready=true);
    void set_param(const string& command, int param);
    //void set_param(const string& command, bool param); //int版で代用できるので要らない
    int get_intparam(const string& command, const string& retstr);
    bool get_boolparam(const string& command);
    string get_stringparam(const string& command);
        
private:
    Serial uart;
    volatile char lastchar;
    queue<char> recoutbuf;
    string recinbuf;
    bool use_recinbuf;
    vector<string> filelist;
    YconP020GBuf internalbuf;

    virtual int _getc();
    virtual int _putc(int c);
};

#endif