#include "mbed.h"
#include "YconP020.h"
#if DEVICE_LOCALFILESYSTEM == 0
#include "DirHandle.h" 
#endif
#include <cctype>
#include <functional>

int toint(const string::const_iterator& begin, const string::const_iterator& end)
{
    if(begin==end) return 0;
    int val=0;
    bool minus=false;
    string::const_iterator p=begin;
    if(*p=='-') {
        minus=true;
        p++;
    }
    while( p!=end) {
        if((*p)<'0' || '9'<(*p)) break;
        val = val*10+(*p)-'0';
        p++;
    }
    if(minus) val = 0-val;
    return val;
}
int toint(const string& s) {return toint(s.begin(),s.end());}
string tostring(int i)
{
    string s;
    bool minus=false;
    if(i<0) {
        minus=true;
        i=0-i;
    }
    do {
        s += (i%10)+'0';
        i/=10;
    } while(i!=0);
    if(minus) s += '-';
    reverse(s.begin(),s.end());
    return s;
}
template<class T>
T skipspace(T begin, T end)
{
    return find_if(begin,end, bind1st(not2(equal_to<char>()), ' '));
}

// class YconP020GBuf
void YconP020GBuf::pixel(int x, int y, int b)
{
    if(x<0 || x>=MAX_X/pixelsizex || y<0 || y>=MAX_Y/pixelsizey) return;
    for(int i=0; i<pixelsizex; i++) {
        for(int j=0; j<pixelsizey; j++) {
            pixel1x1(x*pixelsizex+i, y*pixelsizey+j, b);
        }
    }
}
void YconP020GBuf::setfontscale(uint8_t x, uint8_t y)
{
    if(x==0 || y==0) return;
    locate((_column*pixelsizex+x-1)/x, (_row*pixelsizey+y-1)/y);
    pixelsizex=x;
    pixelsizey=y;
}
void YconP020GBuf::cls()
{
    setfontscale();
    GraphicsDisplay::cls();
    locate(0,0);
}

void YconP020GBuf::pixel1x1(int x, int y, int b)
{
    if(x<0 || x>=MAX_X || y<0 || y>=MAX_Y) return;
    if(!b) buffer[MAX_Y-1-y][x/8] |= 1<<(7-(x%8));
    else buffer[MAX_Y-1-y][x/8] &= ~(1<<(7-(x%8)));
}

// class YconP020
YconP020::YconP020(PinName tx, PinName rx) : uart(tx,rx), use_recinbuf(false)
{
    uart.baud(115200);
    uart.attach(this,&YconP020::uartint);
}

bool YconP020::wait_command_ready(int ms)
{
    for(int i=0; i<ms; i++) {
        if(command_ready()) return true;
        wait_ms(1);
    }
    return false; //timeout
}
bool YconP020::command_mode()
{
    for(int i=0; i<10; i++) {
        if(command_ready()) return true;
        uart.putc('\r');
        wait_ms(300);
        if(command_ready()) return true;
        wait_ms(200);
        for(int i=0; i<3; i++) {
            uart.putc('+');
            wait_ms(200);
        }
        wait(1);
    }
    return false; //timeout
}

void YconP020::stop_demo()
{
    use_recinbuf=true;
    uart.putc(0x1a);
    wait_command_ready();
    use_recinbuf=false;
    recinbuf.clear();
}


const vector<string>& YconP020::getfilelist(const vector<string>& dirs)
{
    filelist.clear();
    for(vector<string>::const_iterator p=dirs.begin(); p!=dirs.end(); p++) {
        string dn;
        if((*p)[0]!='/') dn='/';
        dn += *p;
        DIR *dir = opendir(dn.c_str());
        if(dir==NULL) {
            filelist.push_back("ERROR: filesystem "+ *p +" can't open");
            return filelist;
        }
        string filename;
        struct dirent *file;
        while((file=readdir(dir))!=NULL) {
            filename=file->d_name;
            if(filename.size()<4) continue;
            string extstr;
            transform(filename.end()-4, filename.end(), back_inserter(extstr), toupper);
            if(extstr==".BMP") {
                filelist.push_back('/'+ *p +'/'+filename);
            }
        }
        closedir(dir);
    }
    sort(filelist.begin(),filelist.end());
    return filelist;
}
const vector<string>& YconP020::getfilelistall()
{
    vector<string> fslist;
    DIR *root = opendir("/");
    if(root==NULL) {
        filelist.clear();
        filelist.push_back("ERROR: any filesystem not found");
        return filelist;
    }
    struct dirent *file;
    while((file=readdir(root))!=NULL) {
        fslist.push_back(file->d_name);
    }
    closedir(root);
    return getfilelist(fslist);
}

YconP020::error_t YconP020::dispfile(const string& filename)
{
    use_recinbuf=true;
    command_mode();
    uart.puts("exit\r");
    while(lastchar!='>') wait_ms(1);
    lastchar='\0';
    error_t r=sendfile(filename);
    if(r==NOERROR) while(lastchar!='>') wait_ms(1);
    command_mode();
    use_recinbuf=false;
    recinbuf.clear();
    return r;
}

YconP020::error_t YconP020::storepictfile(const string& filename, const int picnum)
{
    if(picnum>6 || picnum<0) return REGINDEXERR;
    use_recinbuf=true;
    command_mode();
    uart.printf("R %d\r", picnum);
    wait_ms(500);
    error_t r=sendfile(filename);
    wait_command_ready();
    use_recinbuf=false;
    recinbuf.clear();
    return r;
}

const size_t BMPHEADERSIZE = 62;
const uint8_t BMPHEADER[BMPHEADERSIZE]
    = {0x42, 0x4d, 0xbd, 0x0a, 0x00, 0x00, 0x00, 0x00,
       0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x28, 0x00,
       0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x60, 0x00,
       0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00,
       0x00, 0x00, 0x80, 0x0a, 0x00, 0x00, 0x00, 0x00,
       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
       0x00, 0x00, 0xff, 0xff, 0xff, 0x00};
void YconP020::display_internalbuf()
{
    use_recinbuf=true;
    command_mode();
    uart.puts("exit\r");
    while(lastchar!='>') wait_ms(1);
    lastchar='\0';
    for(int i=0; i<BMPHEADERSIZE; i++) {
        uart.putc(BMPHEADER[i]);
    }
    for(int y=0; y<YconP020GBuf::MAX_Y; y++) {
        for(int x=0; x<YconP020GBuf::MAX_X/8; x++) uart.putc(internalbuf.getbyte(x,y));
        uart.putc(0x00);
        uart.putc(0x00);
        uart.putc(0x00);
    }
    while(lastchar!='>') wait_ms(1);
    command_mode();
    use_recinbuf=false;
    recinbuf.clear();
}
//protected:       
void YconP020::uartint()
{
    if(uart.readable()) {
        lastchar = uart.getc();
        push_recbuf(lastchar);
    }
}
void YconP020::push_recbuf(char c)
{
    if(use_recinbuf) {
        recinbuf += c;
        return;
    }
    __disable_irq();
    if(recoutbuf.size()>MAXRECBUFSIZE) recoutbuf.pop();
    recoutbuf.push(c);
    __enable_irq();
}
void YconP020::push_recbuf(const string& s)
{
    for_each(s.begin(),s.end(),bind1st(
        mem_fun<void,YconP020,char>(&YconP020::push_recbuf), this));
}
    
YconP020::error_t YconP020::sendfile(const string& filename)
{
    FILE *fp=fopen(filename.c_str(),"r");
    if(fp==NULL) return FILEOPENERR;
    int c;
    while((c=fgetc(fp)) != EOF) {
        uart.putc(c);
    }
    fclose(fp);
    return NOERROR;
}

// 出力を横取りして拡張コマンドを処理するための関数群
bool commandline_D(YconP020& epd, const string& comline, string& retstr)
{
    string::const_iterator cp=skipspace(comline.begin()+1,comline.end());
    if(cp==comline.end()) return false;
    switch(toupper(*cp)) {
    case 'F':
        if(++cp==comline.end()) return false;
        if(*cp==' ') {
            // "D F filename"
            cp=skipspace(cp,comline.end());
            if(cp!=comline.end()) {
                string fn(cp,comline.end());
                if(epd.dispfile(fn)!=YconP020::NOERROR) retstr += "\r\n"+fn+" not found\r\n!";
            } else retstr += "Filename is required\r\n";
        } else {
            // "D Fnumber"
            int i=toint(cp,comline.end());
            if(0<=i && i<epd.getfilelist().size()) epd.dispfile(epd.getfilelist()[i]);
            else retstr += "\r\nfile No."+string(cp,comline.end())+" not found\r\n!";
        }
        return true;
    case 'I':
        // "D I"
        epd.display_internalbuf();
        return true;
    default:
        return false;
    }
}
bool commandline_L(YconP020& epd, const string &comline, string& retstr)
{
    string::const_iterator cp=comline.begin()+1;
    if(cp==comline.end()) return false;
    if(toupper(*cp)!='S') return false;
    if(++cp==comline.end()) {
        // "LS"
    } else {
        if(*cp!=' ') return false;
        cp=skipspace(cp,comline.end());
        if(cp==comline.end()) {
            // "LS "
        } else if(*cp=='*') {
            // "LS *"
            epd.getfilelistall();
        } else {
            // "LS pathname"
            epd.getfilelist(string(cp,comline.end()));
        }
    }
    for(int i=0; i<epd.getfilelist().size(); i++) {
        retstr += tostring(i)+": "+epd.getfilelist()[i]+"\r\n";
    }
    return true;
}
void T_esc_num(YconP020& epd, string::const_iterator& p, const string::const_iterator& e)
{
    char c=*p++;
    const string::const_iterator semicolon=find(p,e,';');
    if(semicolon==e) return;
    const string::const_iterator comma=find(p,semicolon,',');
    int x,y;
    if(comma==semicolon) {
        x=0;
        y=0;
    } else {
        x=toint(p, comma);
        y=toint(comma+1,semicolon);
    }
    p=semicolon;
    if(c=='l') epd.text()->locate(x,y);
    if(x==0) x=1;
    if(y==0) y=1;
    if(c=='s') epd.setfontscale(x,y);
}
bool commandline_T(YconP020& epd, const string &comline)
{
    if(comline.size()>2 && comline[1]==' ') {
        for(string::const_iterator p=comline.begin()+2; p<comline.end(); p++) {
            if(*p=='\\') {
                if(++p==comline.end()) break;
                switch(*p) {
                    case 'c': epd.text()->cls();         break;
                    case 'f': epd.display_internalbuf(); break;
                    case 'n': epd.text()->putc('\n');    break;
                    case 'l': case 's':
                       T_esc_num(epd,p,comline.end());   break;
                    case '\\': epd.text()->putc('\\');   break;
                    default: epd.text()->putc(*p);
                }
            } else  epd.text()->putc(*p);
        }
        return true;
    }
    return false;
}
int YconP020::putc_intercept(int c)
{
    if(localcomline.size()) {
        push_recbuf(c);
        if(c!='\r') {
            if(c=='\b') localcomline.resize(localcomline.size()-1);
            else localcomline += c;
            return c;
        }
        push_recbuf('\n');
        bool command_done=false;
        string retstr;
        switch(toupper(localcomline[0])) {
            case 'D':
                command_done=commandline_D(*this, localcomline, retstr);
                break;
            case 'L':
                command_done=commandline_L(*this, localcomline, retstr);
                break;
            case 'T':
                command_done=commandline_T(*this, localcomline);
                break;
            case 'W':
                if(localcomline.size()>=2 && toupper(localcomline[1])=='I') {
                    clear_internalbuf();
                    text()->locate(0,0);
                    command_done=true;
                }
                break;
        }
        if(!retstr.empty()) push_recbuf(retstr);
        if(command_done) {
            //拡張コマンドとして処理した
            localcomline.clear();
            push_recbuf('!');
            lastchar='!';
            return c;
        }
        //拡張コマンドではなかったので Y-con P020に送る
        uart.puts(localcomline.c_str());
        uart.putc('\r');
        localcomline.clear();
        return c;
    }
    if(lastchar=='!' && c!='\r') {
        lastchar=' ';
        push_recbuf(c);
        localcomline=c;
    } else {
        uart.putc(c);
    }
    return c;
}

// Y-conP 020 のコマンドを関数化するためのヘルパー関数群
void YconP020::send_command(const string& command, bool waitready)
{
    use_recinbuf=true;
    command_mode();
    lastchar=' ';
    uart.puts(command.c_str());
    uart.putc('\r');
    if(waitready) wait_command_ready();
    use_recinbuf=false;
}    
void YconP020::void_command(const string& command, bool waitready)
{
    send_command(command, waitready);
    recinbuf.clear();
}
void YconP020::set_param(const string& command, int param)
{
    send_command(command+' '+tostring(param));
    recinbuf.clear();
}
int YconP020::get_intparam(const string& command, const string& retstr)
{
    send_command(command);
    int retval;
    string::iterator p=search(recinbuf.begin(),recinbuf.end(),
                                retstr.begin(),  retstr.end());
    if(p!=recinbuf.end()) p+=retstr.size()+1;
    retval=toint(p,recinbuf.end());
    recinbuf.clear();
    return retval;
}
bool YconP020::get_boolparam(const string& command)
{
    send_command(command);
    bool retval=false;
    if(recinbuf.find("Enable")!=string::npos) retval=true;
    recinbuf.clear();
    return retval;
}
string YconP020::get_stringparam(const string& command)
{
    send_command(command);
    string::iterator p=search(recinbuf.begin(),recinbuf.end(),
                               command.begin(), command.end());
    if(p!=recinbuf.end()) p+=command.size();
    string retval(p,find(p,recinbuf.end(),'!'));
    recinbuf.clear();
    return retval;
}

//private:
int YconP020::_getc()
{
    while(recoutbuf.empty()) {}
    __disable_irq();
    int c=recoutbuf.front();
    recoutbuf.pop();
    __enable_irq();
    return c;
}
int YconP020::_putc(int c)
{
    putc_intercept(c);
    return c;
}
