/**
 * @author Athanassios Mavrogeorgiadis
 * @author TI CC1101 library  developed by Athanassios Mavrogeorgiadis (tmav Electronics) as template based on TI C8051 SOURCE CODE swrc021f
 * @section LICENSE
 *
 * Copyright (c) 2010 ARM Limited
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * CC1101 Low-Power Sub-1 GHz RF Transceiver CC1101.
 *
 * Datasheet:
 *
 * http://www.ti.com/lit/ds/symlink/cc1101.pdf
 */


#ifndef CC1101_H
#define CC1101_H


/**
 * Includes
 */
#include "mbed.h"
#include "rtos.h"
#include "Radio.h"
#include "CC1101-Defines.h"


/**
 * CC1101 Low-Power Sub-1 GHz RF Transceiver.
 */
class CC1101 : public Radio
{
public:

    /**
     * Constructor.
     *
     * @param mosi mbed pin to use for MOSI line of SPI interface.
     * @param miso mbed pin to use for MISO line of SPI interface.
     * @param clk mbed pin to use for SCK line of SPI interface.
     * @param csn mbed pin to use for the negated chip select line of SPI interface.
     * @param interpt pin connected to an interrupt-capable input used for signaling that new radio data has been received
     * @param tx_led the indicator led for when data is transmitted
     * @param rx_led the indicator led for when data is received
     * @param xosc oscillator frequency that the CC1101 operates from. Either 26MHz or 27MHz
     */
    CC1101(PinName mosi, PinName miso, PinName clk, PinName csn, PinName tx_led, PinName rx_led, PinName interpt, unsigned int xosc = 27000000);


    /**
    * Default Constructor
    */
    CC1101();


    /*
    * Deconstructor
    */
    virtual ~CC1101();


    uint16_t scan(void);

    void init(void);


    /**
    * Get the chip's version number.
    */
    uint8_t version(void);

    void tiny_delay(void);


    /**
    * Implementation of setting device address
    */
    virtual void address(uint8_t address);


    /**
    * Get the current state of the CC1101 state machine.
    */
    uint8_t mode(void);


    /**
    * Read the contents of any CC1101's status register.
    */
    uint8_t status(uint8_t);


    /**
    * Get the current setting that the CC1101 is operating in.
    */
    uint8_t status(void);


    /**
    * Perform a soft-reset for the CC1101
    */
    void reset(void);


    uint8_t strobe(uint8_t);


    /**
     * This function returns the RSSI value based from the last packet received
     *
     * @return Return the RSSI value.
     */
    uint8_t rssi(void);


    /**
     * This function returns the LQI value based from the last packet received
     *
     * @return Return the LQI value.
     */
    uint8_t lqi(void);


    /**
     * This function reads and returns the contents of the PARTNUM register on the CC1101.
     */
    uint8_t partnum(void);


    /**
     * This function returns the operating frequency that the CC1101 is receiving/transmitting on.
     */
    //virtual uint32_t freq(void);
    virtual void freq(uint32_t);


    /**
     * This function is used to update the operating channel of the CC1101.
     *
     * @param channel Desired operating channel number.
     */
    virtual void channel(uint8_t);


    /**
     * This function gets the value of a single specified CCxxx0 register.
     *
     * @param addr Value of the accessed CCxxx0 register
     * @return Value of the accessed CCxxx0 register
     */
    uint8_t read_reg(uint8_t);


    /**
     * This function reads multiple CCxxx0 register, using SPI burst access.
     *
     * @param addr Value of the accessed CCxxx0 register
     * @param *buffer Pointer to a byte array which stores the values read from a
     *                corresponding range of CCxxx0 registers.
     * @param count Number of bytes to be read from the subsequent CCxxx0 registers.
     */
    void read_reg(uint8_t, uint8_t*, uint8_t);


    /**
     * Function for writing to a single CCxxx0 register
     *
     * @param addr Address of the first CCxxx0 register to be accessed.
     * @param value Value to be written to the specified CCxxx0 register.
     */
    void write_reg(uint8_t, uint8_t);


    /**
     * This function writes to multiple CCxxx0 register, using SPI burst access.
     *
     * @param addr Address of the first CCxxx0 register to be accessed.
     * @param *buffer Array of bytes to be written into a corresponding range of
     *                CCxx00 registers, starting by the address specified in _addr_.
     * @param count Number of bytes to be written to the subsequent CCxxx0 registers.
     */
    void write_reg(uint8_t, uint8_t*, uint8_t);


    /**
     * This function can be used to transmit a packet with packet length up to 63 bytes.
     *
     * @param *txBuffer Pointer to a buffer containing the data that are going to be transmitted
     * @param size The size of the txBuffer
     */
    virtual void put_pck(uint8_t*, uint8_t);


    /**
     * This function can be used to receive a packet of variable packet length (first byte in the packet
     * must be the length byte). The packet length should not exceed the RX FIFO size.
     *
     * @param *rxBuffer Pointer to the buffer where the incoming data should be stored
     * @param *length Pointer to a variable containing the size of the buffer where the incoming data should be
     *                stored. After this function returns, that variable holds the packet length.
     * @return Return value is 1 if CRC OK or else 0 if CRC NOT OK (or no packet was put in the RX FIFO due to filtering)
     */
    virtual bool get_pck(uint8_t*, uint8_t*);


    /**
     * This function flushes the RX FIFO buffer.
     */
    void flush_rx(void);


    /**
     * This function flushes the TX FIFO buffer.
     */
    void flush_tx(void);


    /**
     * This function change the state of CC1101 to RX mode.
     */
    void rx_mode(void);


    void tx_mode(void);

    void idle(void);

    virtual void powerUp(void);

protected:

    DigitalOut *_csn;
    /*
    DigitalOut _tx_led;
    DigitalOut _rx_led;
    */
    // InterruptIn *_rx_int;
    //RingBuffer rxBuf;
    //Serial _pc;

    void calibrate();

private:

    //static void threadStarter(void const*);

    /*
    static void led_tick(void const*);

    static void transmit_thread(void const*);
    static void receive_thread(void const*);
    //static void get_serial_data_thread(void const*);
    */

    void rssi(uint8_t);
    /*
        Thread _receive_thread;
        Thread _transmit_thread;
        Thread _tx_led_thread;
        Thread _rx_led_thread;
        //Thread _serial_thread;
        */

    void setup_spi(PinName = NC, PinName = NC, PinName = NC);
    void setup_pins(PinName = NC, PinName = NC);
    void setup_lights(void);
    void setup_chip(void);

    void assign_modem_params(void);
    void assign_packet_params(void);
    void assign_if_freq(uint32_t);
    void assign_channel_spacing(uint32_t);
    void assign_baud_rate(uint32_t);
    void set_rf_settings(void);
    void assign_freq_offset(uint8_t);

    // ISR routine for new radio data received over the air
    void isr_receive(void);
    void power_on_reset(void);
    void put_rf_settings(void);
    void compute_freq(void);


    rf_settings_t rfSettings;
    radio_state_t      _mode;
    pck_ctrl_t  _pck_control;
    modem_t     _modem;

    uint32_t    _crystal_freq;
    uint32_t    _carrier_freq;
    uint8_t     _offset_freq;
    uint32_t    _if_freq;

    uint32_t    _final_freq;
    uint32_t    _baud_rate;
    uint32_t    _channel_spacing;

    uint8_t         _lqi;
    uint8_t         _partnum;
    uint8_t         _chip_version;

    PinName     _so;
    PinName     _si;
    PinName     _sck;

// Variables used to calculate RSSI
    uint8_t rssi_dec;
    int16_t rssi_dBm;
    //uint8_t rssi_offset[CC_NUMBER_OF_SUB_BANDS];

    int16_t rssiTable[256];
    uint16_t channelNumber[256];
    uint8_t carrierSenseCounter;   // Counter used to keep track on how many time CS has been asserted in one sub band

    uint8_t activeBand; // After the scanFreqBands() function has run, this variable will contain the sub band where
// the strongest signal was found

    uint16_t activeChannel; // After the scanFreqBands() function has run, this variable will contain the channel number
// where the strongest signal was found

};
#endif
