#ifndef COMMUNICATION_LINK_H
#define COMMUNICATION_LINK_H


#include "mbed.h"
#include "cmsis_os.h"
#include "RTP.h"
#include "ThreadHelper.h"
#include "MailHelper.h"
#include "CommModule.h"


#define COMM_LINK_TX_QUEUE_SIZE 3
#define COMM_LINK_RX_QUEUE_SIZE 3
#define COMM_LINK_SIGNAL_START_THREAD   0x01
#define COMM_LINK_SIGNAL_TX_TRIGGER   0x02
#define COMM_LINK_SIGNAL_RX_TRIGGER   0x04
#define COMM_LINK_SIGNAL_MODULE_LINKED   0x08
#define COMM_LINK_BUFFER_SIZE   64


class CommLink
{
public:
    // Constructors
    CommLink();
    CommLink(PinName, PinName, PinName, PinName = NC, PinName = NC);

    // Deconstructor
    virtual ~CommLink() {}; // Don't forget to include deconstructor implementation in derived classes that frees memory

    // Class constants for the data queue sizes
    static const int TX_QUEUE_SIZE;
    static const int RX_QUEUE_SIZE;

    // The pure virtual methods for making CommLink an abstract class
    virtual void reset(void) = 0;
    virtual int32_t selfTest(void) = 0;
    virtual bool isConnected(void) = 0;
    
    void setModule(CommModule&);
    void sendPacket(RTP_t*);
    void receivePacket(RTP_t*);

protected:
    virtual int32_t sendData(uint8_t*, uint8_t) = 0;    // write data out to the radio device using SPI
    virtual int32_t getData(uint8_t*, uint8_t*) = 0;   // read data in from the radio device using SPI

    void ISR(void);
    void toggle_cs(void);
    void ready(void);   // Always call CommLink::ready() after derived class is ready for communication
    void setup_spi(void);

    // The data queues for temporarily holding received packets and packets that need to be transmitted
    osMailQId   _txQueue;
    osMailQId   _rxQueue;

    // ============== PIN NAMES ==============
    // SPI bus pins
    PinName     _miso_pin;
    PinName     _mosi_pin;
    PinName     _sck_pin;
    PinName     _cs_pin;    // CS pin
    PinName     _int_pin;   // Interrupt pin

    // ============== PIN OBJECTS ==============
    SPI         *_spi;      // SPI pointer
    DigitalOut  *_cs;       // Chip Select pointer
    InterruptIn *_int_in;    // Interrupt pin

private:
    // Used to help define the class's threads in the constructor
    friend void define_thread(osThreadDef_t&, void(*task)(void const *arg), osPriority, uint32_t, unsigned char*);

    // Data queues
    MailHelper<RTP_t, COMM_LINK_TX_QUEUE_SIZE>   _txQueueHelper;
    MailHelper<RTP_t, COMM_LINK_RX_QUEUE_SIZE>   _rxQueueHelper;
    
    // Thread definitions and IDs
    osThreadDef_t   _txDef;
    osThreadDef_t   _rxDef;
    osThreadId      _txID;
    osThreadId      _rxID;

    // The working threads for handeling RX/TX data queue operations
    static void txThread(void const*);
    static void rxThread(void const*);

    // Methods for initializing a transceiver's pins for communication
    void setup(void);
    void setup_pins(PinName = NC, PinName = NC, PinName = NC, PinName = NC, PinName = NC);
    void setup_cs(void);
    void setup_interrupt(void);
    
    // Used for tracking the number of link-level communication interfaces
    static unsigned int _nbr_links;

    //typedef void(*func_p)(void const*);
    //nc_p          _rx_handle;

    uint8_t buf[COMM_LINK_BUFFER_SIZE];

    CommModule     *_comm_module;

};

#endif  // COMMUNICATION_LINK_H
