// RoboCup dual-frequency band base station

#include "BaseStation.h"

uint8_t channel;

// Function for writing a number to the 7-segment display
void writeSegment(uint8_t val, DigitalOut& latch)
{
    // Outputs used as input values to the 7-segment binary decoder - uses latching inputs
    DigitalOut signal[4] = { RJ_7_SEG_PINS };

    // write out the new value
    for (int i=0; i<4; i++)
        signal[i] = ((1<<i) & (val)) & 0x0F;

    // latch the value
    for (int i=0; i<2; i++)
        latch = !latch;
}

void seg_task(void const *arg)
{
    // latch pin for 7-seg
    DigitalOut latch( RJ_7_SEG_LATCH_PIN, 0 );

    // Decimal point initialized to OFF
    DigitalOut decimal( RJ_7_SEG_DOT_PIN, 1 );

    channel = 8;
    writeSegment(channel, latch);

    // wait to be signaled before beginning
    osSignalWait(0x01, osWaitForever);

    // give a small delay to ensure the startup value stays lit for some time
    Thread::wait(500);

    // turn the decimal point off
    decimal = 0;

    while(1) {  // loop forever
        // send numerical value to 7-segment & hold for a while
        writeSegment(channel, latch);
        // channel = (channel > 9) ? 0 : channel; // reset value if too high
        Thread::wait(1000);
    }
}

int main()
{
    // RGB Status LED
    PwmOut rgb_led[3] = { RJ_RGB_LED_PINS };

    // Primary radio status LEDs
    DigitalOut r1_led[3] = { RJ_PRIMARY_RADIO_LEDS };

    // Secondary radio status LEDs
    DigitalOut r2_led[3] = { RJ_SECONDARY_RADIO_LEDS };

    // Used for controlling power to the RGB LED's shared annode lead
    DigitalOut rgb_pwr( RJ_RGB_LED_ANNODE, 0 );

    // Start 7-segment task
    Thread thread_seg_task(seg_task);

    // turn all LEDs off initially - values are inverted since LEDs are sinking the current
    for (int i=0; i<3; i++) {
        rgb_led[i] = 1;
        r1_led[i] = 1;
        r2_led[i] = 1;
    }

    // =========== Cyle primary & secondary radio status LEDs ===========
    // turn on all radio status LEDs
    for (int i=0; i<3; i++) {

        // initialze off at the start of every iteration
        for(int j=0; j<3; j++) {
            r1_led[j] = 1;
            r2_led[j] = 1;
        }

        if (i != 2) {
            // cycle 2 LEDs
            for (int j=i; j<2; j++) {
                r1_led[j] = 0;
                r2_led[j] = 0;
                Thread::wait(50);
            }
        } else {
            r1_led[i] = 0;
            r2_led[i] = 0;
            Thread::wait(50);
            r1_led[i] = 1;
            r2_led[i] = 1;
        }
    }

    // give power to all colors of the RGB LED and turn off decimal point
    rgb_pwr = 1;

    // 915MHz band radio object
    CC1101 cc1101(
        RJ_SPI_PINS,
        RJ_PRIMARY_RADIO_CS_PIN,
        RJ_PRIMARY_RADIO_LED_TX,
        RJ_PRIMARY_RADIO_LED_RX,
        RJ_PRIMARY_RADIO_INT_PIN
    );

    // set pointer for base class calls
    Radio *radio_p = &cc1101;

    channel = radio_p->channel();
    // tell the segment thread to begin its task
    thread_seg_task.signal_set(0x01);

    uint8_t color;
    if (radio_p->hasError()) {
        color = 0;
    } else {
        color = 1;
    }

    // fade the RGB LED up to the right status color at half power
    for (float i=1.0; i>0.5; i-=0.01) {
        rgb_led[color] = i;
        Thread::wait(20);
    }



    // loop forever ====================
    while(1) {

        //

        // delay
        Thread::wait(300);
    }
}