/* mbed Library - MobileLCD
 * Copyright (c) 2007, sford
 */

#include "MobileLCD.h"

#include "mbed.h"

#include "Terminal.h"

Terminal term(USBTX, USBRX); // tx, rx

using namespace mbed;

const unsigned char FONT6x8[97][8] = {
    0x06,0x08,0x08,0x00,0x00,0x00,0x00,0x00, // columns, rows, num_bytes_per_char
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // space 0x20
    0x20,0x20,0x20,0x20,0x20,0x00,0x20,0x00, // !
    0x50,0x50,0x50,0x00,0x00,0x00,0x00,0x00, // "
    0x50,0x50,0xF8,0x50,0xF8,0x50,0x50,0x00, // #
    0x20,0x78,0xA0,0x70,0x28,0xF0,0x20,0x00, // $
    0xC0,0xC8,0x10,0x20,0x40,0x98,0x18,0x00, // %
    0x40,0xA0,0xA0,0x40,0xA8,0x90,0x68,0x00, // &
    0x30,0x30,0x20,0x40,0x00,0x00,0x00,0x00, // '
    0x10,0x20,0x40,0x40,0x40,0x20,0x10,0x00, // (
    0x40,0x20,0x10,0x10,0x10,0x20,0x40,0x00, // )
    0x00,0x20,0xA8,0x70,0x70,0xA8,0x20,0x00, // *
    0x00,0x20,0x20,0xF8,0x20,0x20,0x00,0x00, // +
    0x00,0x00,0x00,0x00,0x30,0x30,0x20,0x40, // ,
    0x00,0x00,0x00,0xF8,0x00,0x00,0x00,0x00, // -
    0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00, // .
    0x00,0x08,0x10,0x20,0x40,0x80,0x00,0x00, // / (forward slash)
    0x70,0x88,0x88,0xA8,0x88,0x88,0x70,0x00, // 0 0x30
    0x20,0x60,0x20,0x20,0x20,0x20,0x70,0x00, // 1
    0x70,0x88,0x08,0x70,0x80,0x80,0xF8,0x00, // 2
    0xF8,0x08,0x10,0x30,0x08,0x88,0x70,0x00, // 3
    0x10,0x30,0x50,0x90,0xF8,0x10,0x10,0x00, // 4
    0xF8,0x80,0xF0,0x08,0x08,0x88,0x70,0x00, // 5
    0x38,0x40,0x80,0xF0,0x88,0x88,0x70,0x00, // 6
    0xF8,0x08,0x08,0x10,0x20,0x40,0x80,0x00, // 7
    0x70,0x88,0x88,0x70,0x88,0x88,0x70,0x00, // 8
    0x70,0x88,0x88,0x78,0x08,0x10,0xE0,0x00, // 9
    0x00,0x00,0x20,0x00,0x20,0x00,0x00,0x00, // :
    0x00,0x00,0x20,0x00,0x20,0x20,0x40,0x00, // ;
    0x08,0x10,0x20,0x40,0x20,0x10,0x08,0x00, // <
    0x00,0x00,0xF8,0x00,0xF8,0x00,0x00,0x00, // =
    0x40,0x20,0x10,0x08,0x10,0x20,0x40,0x00, // >
    0x70,0x88,0x08,0x30,0x20,0x00,0x20,0x00, // ?
    0x70,0x88,0xA8,0xB8,0xB0,0x80,0x78,0x00, // @ 0x40
    0x20,0x50,0x88,0x88,0xF8,0x88,0x88,0x00, // A
    0xF0,0x88,0x88,0xF0,0x88,0x88,0xF0,0x00, // B
    0x70,0x88,0x80,0x80,0x80,0x88,0x70,0x00, // C
    0xF0,0x88,0x88,0x88,0x88,0x88,0xF0,0x00, // D
    0xF8,0x80,0x80,0xF0,0x80,0x80,0xF8,0x00, // E
    0xF8,0x80,0x80,0xF0,0x80,0x80,0x80,0x00, // F
    0x78,0x88,0x80,0x80,0x98,0x88,0x78,0x00, // G
    0x88,0x88,0x88,0xF8,0x88,0x88,0x88,0x00, // H
    0x70,0x20,0x20,0x20,0x20,0x20,0x70,0x00, // I
    0x38,0x10,0x10,0x10,0x10,0x90,0x60,0x00, // J
    0x88,0x90,0xA0,0xC0,0xA0,0x90,0x88,0x00, // K
    0x80,0x80,0x80,0x80,0x80,0x80,0xF8,0x00, // L
    0x88,0xD8,0xA8,0xA8,0xA8,0x88,0x88,0x00, // M
    0x88,0x88,0xC8,0xA8,0x98,0x88,0x88,0x00, // N
    0x70,0x88,0x88,0x88,0x88,0x88,0x70,0x00, // O
    0xF0,0x88,0x88,0xF0,0x80,0x80,0x80,0x00, // P 0x50
    0x70,0x88,0x88,0x88,0xA8,0x90,0x68,0x00, // Q
    0xF0,0x88,0x88,0xF0,0xA0,0x90,0x88,0x00, // R
    0x70,0x88,0x80,0x70,0x08,0x88,0x70,0x00, // S
    0xF8,0xA8,0x20,0x20,0x20,0x20,0x20,0x00, // T
    0x88,0x88,0x88,0x88,0x88,0x88,0x70,0x00, // U
    0x88,0x88,0x88,0x88,0x88,0x50,0x20,0x00, // V
    0x88,0x88,0x88,0xA8,0xA8,0xA8,0x50,0x00, // W
    0x88,0x88,0x50,0x20,0x50,0x88,0x88,0x00, // X
    0x88,0x88,0x50,0x20,0x20,0x20,0x20,0x00, // Y
    0xF8,0x08,0x10,0x70,0x40,0x80,0xF8,0x00, // Z
    0x78,0x40,0x40,0x40,0x40,0x40,0x78,0x00, // [
    0x00,0x80,0x40,0x20,0x10,0x08,0x00,0x00, // \ (back slash)
    0x78,0x08,0x08,0x08,0x08,0x08,0x78,0x00, // ]
    0x20,0x50,0x88,0x00,0x00,0x00,0x00,0x00, // ^
    0x00,0x00,0x00,0x00,0x00,0x00,0xF8,0x00, // _
    0x60,0x60,0x20,0x10,0x00,0x00,0x00,0x00, // ` 0x60
    0x00,0x00,0x60,0x10,0x70,0x90,0x78,0x00, // a
    0x80,0x80,0xB0,0xC8,0x88,0xC8,0xB0,0x00, // b
    0x00,0x00,0x70,0x88,0x80,0x88,0x70,0x00, // c
    0x08,0x08,0x68,0x98,0x88,0x98,0x68,0x00, // d
    0x00,0x00,0x70,0x88,0xF8,0x80,0x70,0x00, // e
    0x10,0x28,0x20,0x70,0x20,0x20,0x20,0x00, // f
    0x00,0x00,0x70,0x98,0x98,0x68,0x08,0x70, // g
    0x80,0x80,0xB0,0xC8,0x88,0x88,0x88,0x00, // h
    0x20,0x00,0x60,0x20,0x20,0x20,0x70,0x00, // i
    0x10,0x00,0x10,0x10,0x10,0x90,0x60,0x00, // j
    0x80,0x80,0x90,0xA0,0xC0,0xA0,0x90,0x00, // k
    0x60,0x20,0x20,0x20,0x20,0x20,0x70,0x00, // l
    0x00,0x00,0xD0,0xA8,0xA8,0xA8,0xA8,0x00, // m
    0x00,0x00,0xB0,0xC8,0x88,0x88,0x88,0x00, // n
    0x00,0x00,0x70,0x88,0x88,0x88,0x70,0x00, // o
    0x00,0x00,0xB0,0xC8,0xC8,0xB0,0x80,0x80, // p 0x70
    0x00,0x00,0x68,0x98,0x98,0x68,0x08,0x08, // q
    0x00,0x00,0xB0,0xC8,0x80,0x80,0x80,0x00, // r
    0x00,0x00,0x78,0x80,0x70,0x08,0xF0,0x00, // s
    0x20,0x20,0xF8,0x20,0x20,0x28,0x10,0x00, // t
    0x00,0x00,0x88,0x88,0x88,0x98,0x68,0x00, // u
    0x00,0x00,0x88,0x88,0x88,0x50,0x20,0x00, // v
    0x00,0x00,0x88,0x88,0xA8,0xA8,0x50,0x00, // w
    0x00,0x00,0x88,0x50,0x20,0x50,0x88,0x00, // x
    0x00,0x00,0x88,0x88,0x78,0x08,0x88,0x70, // y
    0x00,0x00,0xF8,0x10,0x20,0x40,0xF8,0x00, // z
    0x10,0x20,0x20,0x40,0x20,0x20,0x10,0x00, // {
    0x20,0x20,0x20,0x00,0x20,0x20,0x20,0x00, // |
    0x40,0x20,0x20,0x10,0x20,0x20,0x40,0x00, // }
    0x40,0xA8,0x10,0x00,0x00,0x00,0x00,0x00, // ~
    0x70,0xD8,0xD8,0x70,0x00,0x00,0x00,0x00
}; // DEL

const unsigned char FONT8x8[97][8] = {
    0x08,0x08,0x08,0x00,0x00,0x00,0x00,0x00, // columns, rows, num_bytes_per_char
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // space 0x20
    0x30,0x78,0x78,0x30,0x30,0x00,0x30,0x00, // !
    0x6C,0x6C,0x6C,0x00,0x00,0x00,0x00,0x00, // "
    0x6C,0x6C,0xFE,0x6C,0xFE,0x6C,0x6C,0x00, // #
    0x18,0x3E,0x60,0x3C,0x06,0x7C,0x18,0x00, // $
    0x00,0x63,0x66,0x0C,0x18,0x33,0x63,0x00, // %
    0x1C,0x36,0x1C,0x3B,0x6E,0x66,0x3B,0x00, // &
    0x30,0x30,0x60,0x00,0x00,0x00,0x00,0x00, // '
    0x0C,0x18,0x30,0x30,0x30,0x18,0x0C,0x00, // (
    0x30,0x18,0x0C,0x0C,0x0C,0x18,0x30,0x00, // )
    0x00,0x66,0x3C,0xFF,0x3C,0x66,0x00,0x00, // *
    0x00,0x30,0x30,0xFC,0x30,0x30,0x00,0x00, // +
    0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x30, // ,
    0x00,0x00,0x00,0x7E,0x00,0x00,0x00,0x00, // -
    0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x00, // .
    0x03,0x06,0x0C,0x18,0x30,0x60,0x40,0x00, // / (forward slash)
    0x3E,0x63,0x63,0x6B,0x63,0x63,0x3E,0x00, // 0 0x30
    0x18,0x38,0x58,0x18,0x18,0x18,0x7E,0x00, // 1
    0x3C,0x66,0x06,0x1C,0x30,0x66,0x7E,0x00, // 2
    0x3C,0x66,0x06,0x1C,0x06,0x66,0x3C,0x00, // 3
    0x0E,0x1E,0x36,0x66,0x7F,0x06,0x0F,0x00, // 4
    0x7E,0x60,0x7C,0x06,0x06,0x66,0x3C,0x00, // 5
    0x1C,0x30,0x60,0x7C,0x66,0x66,0x3C,0x00, // 6
    0x7E,0x66,0x06,0x0C,0x18,0x18,0x18,0x00, // 7
    0x3C,0x66,0x66,0x3C,0x66,0x66,0x3C,0x00, // 8
    0x3C,0x66,0x66,0x3E,0x06,0x0C,0x38,0x00, // 9
    0x00,0x18,0x18,0x00,0x00,0x18,0x18,0x00, // :
    0x00,0x18,0x18,0x00,0x00,0x18,0x18,0x30, // ;
    0x0C,0x18,0x30,0x60,0x30,0x18,0x0C,0x00, // <
    0x00,0x00,0x7E,0x00,0x00,0x7E,0x00,0x00, // =
    0x30,0x18,0x0C,0x06,0x0C,0x18,0x30,0x00, // >
    0x3C,0x66,0x06,0x0C,0x18,0x00,0x18,0x00, // ?
    0x3E,0x63,0x6F,0x69,0x6F,0x60,0x3E,0x00, // @ 0x40
    0x18,0x3C,0x66,0x66,0x7E,0x66,0x66,0x00, // A
    0x7E,0x33,0x33,0x3E,0x33,0x33,0x7E,0x00, // B
    0x1E,0x33,0x60,0x60,0x60,0x33,0x1E,0x00, // C
    0x7C,0x36,0x33,0x33,0x33,0x36,0x7C,0x00, // D
    0x7F,0x31,0x34,0x3C,0x34,0x31,0x7F,0x00, // E
    0x7F,0x31,0x34,0x3C,0x34,0x30,0x78,0x00, // F
    0x1E,0x33,0x60,0x60,0x67,0x33,0x1F,0x00, // G
    0x66,0x66,0x66,0x7E,0x66,0x66,0x66,0x00, // H
    0x3C,0x18,0x18,0x18,0x18,0x18,0x3C,0x00, // I
    0x0F,0x06,0x06,0x06,0x66,0x66,0x3C,0x00, // J
    0x73,0x33,0x36,0x3C,0x36,0x33,0x73,0x00, // K
    0x78,0x30,0x30,0x30,0x31,0x33,0x7F,0x00, // L
    0x63,0x77,0x7F,0x7F,0x6B,0x63,0x63,0x00, // M
    0x63,0x73,0x7B,0x6F,0x67,0x63,0x63,0x00, // N
    0x3E,0x63,0x63,0x63,0x63,0x63,0x3E,0x00, // O
    0x7E,0x33,0x33,0x3E,0x30,0x30,0x78,0x00, // P 0x50
    0x3C,0x66,0x66,0x66,0x6E,0x3C,0x0E,0x00, // Q
    0x7E,0x33,0x33,0x3E,0x36,0x33,0x73,0x00, // R
    0x3C,0x66,0x30,0x18,0x0C,0x66,0x3C,0x00, // S
    0x7E,0x5A,0x18,0x18,0x18,0x18,0x3C,0x00, // T
    0x66,0x66,0x66,0x66,0x66,0x66,0x7E,0x00, // U
    0x66,0x66,0x66,0x66,0x66,0x3C,0x18,0x00, // V
    0x63,0x63,0x63,0x6B,0x7F,0x77,0x63,0x00, // W
    0x63,0x63,0x36,0x1C,0x1C,0x36,0x63,0x00, // X
    0x66,0x66,0x66,0x3C,0x18,0x18,0x3C,0x00, // Y
    0x7F,0x63,0x46,0x0C,0x19,0x33,0x7F,0x00, // Z
    0x3C,0x30,0x30,0x30,0x30,0x30,0x3C,0x00, // [
    0x60,0x30,0x18,0x0C,0x06,0x03,0x01,0x00, // \ (back slash)
    0x3C,0x0C,0x0C,0x0C,0x0C,0x0C,0x3C,0x00, // ]
    0x08,0x1C,0x36,0x63,0x00,0x00,0x00,0x00, // ^
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF, // _
    0x18,0x18,0x0C,0x00,0x00,0x00,0x00,0x00, // ` 0x60
    0x00,0x00,0x3C,0x06,0x3E,0x66,0x3B,0x00, // a
    0x70,0x30,0x3E,0x33,0x33,0x33,0x6E,0x00, // b
    0x00,0x00,0x3C,0x66,0x60,0x66,0x3C,0x00, // c
    0x0E,0x06,0x3E,0x66,0x66,0x66,0x3B,0x00, // d
    0x00,0x00,0x3C,0x66,0x7E,0x60,0x3C,0x00, // e
    0x1C,0x36,0x30,0x78,0x30,0x30,0x78,0x00, // f
    0x00,0x00,0x3B,0x66,0x66,0x3E,0x06,0x7C, // g
    0x70,0x30,0x36,0x3B,0x33,0x33,0x73,0x00, // h
    0x18,0x00,0x38,0x18,0x18,0x18,0x3C,0x00, // i
    0x06,0x00,0x06,0x06,0x06,0x66,0x66,0x3C, // j
    0x70,0x30,0x33,0x36,0x3C,0x36,0x73,0x00, // k
    0x38,0x18,0x18,0x18,0x18,0x18,0x3C,0x00, // l
    0x00,0x00,0x66,0x7F,0x7F,0x6B,0x63,0x00, // m
    0x00,0x00,0x7C,0x66,0x66,0x66,0x66,0x00, // n
    0x00,0x00,0x3C,0x66,0x66,0x66,0x3C,0x00, // o
    0x00,0x00,0x6E,0x33,0x33,0x3E,0x30,0x78, // p
    0x00,0x00,0x3B,0x66,0x66,0x3E,0x06,0x0F, // q
    0x00,0x00,0x6E,0x3B,0x33,0x30,0x78,0x00, // r
    0x00,0x00,0x3E,0x60,0x3C,0x06,0x7C,0x00, // s
    0x08,0x18,0x3E,0x18,0x18,0x1A,0x0C,0x00, // t
    0x00,0x00,0x66,0x66,0x66,0x66,0x3B,0x00, // u
    0x00,0x00,0x66,0x66,0x66,0x3C,0x18,0x00, // v
    0x00,0x00,0x63,0x6B,0x7F,0x7F,0x36,0x00, // w
    0x00,0x00,0x63,0x36,0x1C,0x36,0x63,0x00, // x
    0x00,0x00,0x66,0x66,0x66,0x3E,0x06,0x7C, // y
    0x00,0x00,0x7E,0x4C,0x18,0x32,0x7E,0x00, // z
    0x0E,0x18,0x18,0x70,0x18,0x18,0x0E,0x00, // {
    0x0C,0x0C,0x0C,0x00,0x0C,0x0C,0x0C,0x00, // |
    0x70,0x18,0x18,0x0E,0x18,0x18,0x70,0x00, // }
    0x3B,0x6E,0x00,0x00,0x00,0x00,0x00,0x00, // ~
    0x1C,0x36,0x36,0x1C,0x00,0x00,0x00,0x00
}; // DEL

const unsigned char FONT8x16[97][16] = {
    0x08,0x10,0x10,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // columns, rows, num_bytes_per_char
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // space 0x20
    0x00,0x00,0x18,0x3C,0x3C,0x3C,0x18,0x18,0x18,0x00,0x18,0x18,0x00,0x00,0x00,0x00, // !
    0x00,0x63,0x63,0x63,0x22,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // "
    0x00,0x00,0x00,0x36,0x36,0x7F,0x36,0x36,0x36,0x7F,0x36,0x36,0x00,0x00,0x00,0x00, // #
    0x0C,0x0C,0x3E,0x63,0x61,0x60,0x3E,0x03,0x03,0x43,0x63,0x3E,0x0C,0x0C,0x00,0x00, // $
    0x00,0x00,0x00,0x00,0x00,0x61,0x63,0x06,0x0C,0x18,0x33,0x63,0x00,0x00,0x00,0x00, // %
    0x00,0x00,0x00,0x1C,0x36,0x36,0x1C,0x3B,0x6E,0x66,0x66,0x3B,0x00,0x00,0x00,0x00, // &
    0x00,0x30,0x30,0x30,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // '
    0x00,0x00,0x0C,0x18,0x18,0x30,0x30,0x30,0x30,0x18,0x18,0x0C,0x00,0x00,0x00,0x00, // (
    0x00,0x00,0x18,0x0C,0x0C,0x06,0x06,0x06,0x06,0x0C,0x0C,0x18,0x00,0x00,0x00,0x00, // )
    0x00,0x00,0x00,0x00,0x42,0x66,0x3C,0xFF,0x3C,0x66,0x42,0x00,0x00,0x00,0x00,0x00, // *
    0x00,0x00,0x00,0x00,0x18,0x18,0x18,0xFF,0x18,0x18,0x18,0x00,0x00,0x00,0x00,0x00, // +
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x18,0x30,0x00,0x00, // ,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // -
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x00,0x00,0x00,0x00, // .
    0x00,0x00,0x01,0x03,0x07,0x0E,0x1C,0x38,0x70,0xE0,0xC0,0x80,0x00,0x00,0x00,0x00, // / (forward slash)
    0x00,0x00,0x3E,0x63,0x63,0x63,0x6B,0x6B,0x63,0x63,0x63,0x3E,0x00,0x00,0x00,0x00, // 0 0x30
    0x00,0x00,0x0C,0x1C,0x3C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x3F,0x00,0x00,0x00,0x00, // 1
    0x00,0x00,0x3E,0x63,0x03,0x06,0x0C,0x18,0x30,0x61,0x63,0x7F,0x00,0x00,0x00,0x00, // 2
    0x00,0x00,0x3E,0x63,0x03,0x03,0x1E,0x03,0x03,0x03,0x63,0x3E,0x00,0x00,0x00,0x00, // 3
    0x00,0x00,0x06,0x0E,0x1E,0x36,0x66,0x66,0x7F,0x06,0x06,0x0F,0x00,0x00,0x00,0x00, // 4
    0x00,0x00,0x7F,0x60,0x60,0x60,0x7E,0x03,0x03,0x63,0x73,0x3E,0x00,0x00,0x00,0x00, // 5
    0x00,0x00,0x1C,0x30,0x60,0x60,0x7E,0x63,0x63,0x63,0x63,0x3E,0x00,0x00,0x00,0x00, // 6
    0x00,0x00,0x7F,0x63,0x03,0x06,0x06,0x0C,0x0C,0x18,0x18,0x18,0x00,0x00,0x00,0x00, // 7
    0x00,0x00,0x3E,0x63,0x63,0x63,0x3E,0x63,0x63,0x63,0x63,0x3E,0x00,0x00,0x00,0x00, // 8
    0x00,0x00,0x3E,0x63,0x63,0x63,0x63,0x3F,0x03,0x03,0x06,0x3C,0x00,0x00,0x00,0x00, // 9
    0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x00,0x00,0x00,0x18,0x18,0x00,0x00,0x00,0x00, // :
    0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x00,0x00,0x00,0x18,0x18,0x18,0x30,0x00,0x00, // ;
    0x00,0x00,0x00,0x06,0x0C,0x18,0x30,0x60,0x30,0x18,0x0C,0x06,0x00,0x00,0x00,0x00, // <
    0x00,0x00,0x00,0x00,0x00,0x00,0x7E,0x00,0x00,0x7E,0x00,0x00,0x00,0x00,0x00,0x00, // =
    0x00,0x00,0x00,0x60,0x30,0x18,0x0C,0x06,0x0C,0x18,0x30,0x60,0x00,0x00,0x00,0x00, // >
    0x00,0x00,0x3E,0x63,0x63,0x06,0x0C,0x0C,0x0C,0x00,0x0C,0x0C,0x00,0x00,0x00,0x00, // ?
    0x00,0x00,0x3E,0x63,0x63,0x6F,0x6B,0x6B,0x6E,0x60,0x60,0x3E,0x00,0x00,0x00,0x00, // @ 0x40
    0x00,0x00,0x08,0x1C,0x36,0x63,0x63,0x63,0x7F,0x63,0x63,0x63,0x00,0x00,0x00,0x00, // A
    0x00,0x00,0x7E,0x33,0x33,0x33,0x3E,0x33,0x33,0x33,0x33,0x7E,0x00,0x00,0x00,0x00, // B
    0x00,0x00,0x1E,0x33,0x61,0x60,0x60,0x60,0x60,0x61,0x33,0x1E,0x00,0x00,0x00,0x00, // C
    0x00,0x00,0x7C,0x36,0x33,0x33,0x33,0x33,0x33,0x33,0x36,0x7C,0x00,0x00,0x00,0x00, // D
    0x00,0x00,0x7F,0x33,0x31,0x34,0x3C,0x34,0x30,0x31,0x33,0x7F,0x00,0x00,0x00,0x00, // E
    0x00,0x00,0x7F,0x33,0x31,0x34,0x3C,0x34,0x30,0x30,0x30,0x78,0x00,0x00,0x00,0x00, // F
    0x00,0x00,0x1E,0x33,0x61,0x60,0x60,0x6F,0x63,0x63,0x37,0x1D,0x00,0x00,0x00,0x00, // G
    0x00,0x00,0x63,0x63,0x63,0x63,0x7F,0x63,0x63,0x63,0x63,0x63,0x00,0x00,0x00,0x00, // H
    0x00,0x00,0x3C,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00, // I
    0x00,0x00,0x0F,0x06,0x06,0x06,0x06,0x06,0x06,0x66,0x66,0x3C,0x00,0x00,0x00,0x00, // J
    0x00,0x00,0x73,0x33,0x36,0x36,0x3C,0x36,0x36,0x33,0x33,0x73,0x00,0x00,0x00,0x00, // K
    0x00,0x00,0x78,0x30,0x30,0x30,0x30,0x30,0x30,0x31,0x33,0x7F,0x00,0x00,0x00,0x00, // L
    0x00,0x00,0x63,0x77,0x7F,0x6B,0x63,0x63,0x63,0x63,0x63,0x63,0x00,0x00,0x00,0x00, // M
    0x00,0x00,0x63,0x63,0x73,0x7B,0x7F,0x6F,0x67,0x63,0x63,0x63,0x00,0x00,0x00,0x00, // N
    0x00,0x00,0x1C,0x36,0x63,0x63,0x63,0x63,0x63,0x63,0x36,0x1C,0x00,0x00,0x00,0x00, // O
    0x00,0x00,0x7E,0x33,0x33,0x33,0x3E,0x30,0x30,0x30,0x30,0x78,0x00,0x00,0x00,0x00, // P 0x50
    0x00,0x00,0x3E,0x63,0x63,0x63,0x63,0x63,0x63,0x6B,0x6F,0x3E,0x06,0x07,0x00,0x00, // Q
    0x00,0x00,0x7E,0x33,0x33,0x33,0x3E,0x36,0x36,0x33,0x33,0x73,0x00,0x00,0x00,0x00, // R
    0x00,0x00,0x3E,0x63,0x63,0x30,0x1C,0x06,0x03,0x63,0x63,0x3E,0x00,0x00,0x00,0x00, // S
    0x00,0x00,0xFF,0xDB,0x99,0x18,0x18,0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00, // T
    0x00,0x00,0x63,0x63,0x63,0x63,0x63,0x63,0x63,0x63,0x63,0x3E,0x00,0x00,0x00,0x00, // U
    0x00,0x00,0x63,0x63,0x63,0x63,0x63,0x63,0x63,0x36,0x1C,0x08,0x00,0x00,0x00,0x00, // V
    0x00,0x00,0x63,0x63,0x63,0x63,0x63,0x6B,0x6B,0x7F,0x36,0x36,0x00,0x00,0x00,0x00, // W
    0x00,0x00,0xC3,0xC3,0x66,0x3C,0x18,0x18,0x3C,0x66,0xC3,0xC3,0x00,0x00,0x00,0x00, // X
    0x00,0x00,0xC3,0xC3,0xC3,0x66,0x3C,0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00, // Y
    0x00,0x00,0x7F,0x63,0x43,0x06,0x0C,0x18,0x30,0x61,0x63,0x7F,0x00,0x00,0x00,0x00, // Z
    0x00,0x00,0x3C,0x30,0x30,0x30,0x30,0x30,0x30,0x30,0x30,0x3C,0x00,0x00,0x00,0x00, // [
    0x00,0x00,0x80,0xC0,0xE0,0x70,0x38,0x1C,0x0E,0x07,0x03,0x01,0x00,0x00,0x00,0x00, // \ (back slash)
    0x00,0x00,0x3C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x3C,0x00,0x00,0x00,0x00, // ]
    0x08,0x1C,0x36,0x63,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // ^
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0x00,0x00,0x00, // _
    0x18,0x18,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // ` 0x60
    0x00,0x00,0x00,0x00,0x00,0x3C,0x46,0x06,0x3E,0x66,0x66,0x3B,0x00,0x00,0x00,0x00, // a
    0x00,0x00,0x70,0x30,0x30,0x3C,0x36,0x33,0x33,0x33,0x33,0x6E,0x00,0x00,0x00,0x00, // b
    0x00,0x00,0x00,0x00,0x00,0x3E,0x63,0x60,0x60,0x60,0x63,0x3E,0x00,0x00,0x00,0x00, // c
    0x00,0x00,0x0E,0x06,0x06,0x1E,0x36,0x66,0x66,0x66,0x66,0x3B,0x00,0x00,0x00,0x00, // d
    0x00,0x00,0x00,0x00,0x00,0x3E,0x63,0x63,0x7E,0x60,0x63,0x3E,0x00,0x00,0x00,0x00, // e
    0x00,0x00,0x1C,0x36,0x32,0x30,0x7C,0x30,0x30,0x30,0x30,0x78,0x00,0x00,0x00,0x00, // f
    0x00,0x00,0x00,0x00,0x00,0x3B,0x66,0x66,0x66,0x66,0x3E,0x06,0x66,0x3C,0x00,0x00, // g
    0x00,0x00,0x70,0x30,0x30,0x36,0x3B,0x33,0x33,0x33,0x33,0x73,0x00,0x00,0x00,0x00, // h
    0x00,0x00,0x0C,0x0C,0x00,0x1C,0x0C,0x0C,0x0C,0x0C,0x0C,0x1E,0x00,0x00,0x00,0x00, // i
    0x00,0x00,0x06,0x06,0x00,0x0E,0x06,0x06,0x06,0x06,0x06,0x66,0x66,0x3C,0x00,0x00, // j
    0x00,0x00,0x70,0x30,0x30,0x33,0x33,0x36,0x3C,0x36,0x33,0x73,0x00,0x00,0x00,0x00, // k
    0x00,0x00,0x1C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x1E,0x00,0x00,0x00,0x00, // l
    0x00,0x00,0x00,0x00,0x00,0x6E,0x7F,0x6B,0x6B,0x6B,0x6B,0x6B,0x00,0x00,0x00,0x00, // m
    0x00,0x00,0x00,0x00,0x00,0x6E,0x33,0x33,0x33,0x33,0x33,0x33,0x00,0x00,0x00,0x00, // n
    0x00,0x00,0x00,0x00,0x00,0x3E,0x63,0x63,0x63,0x63,0x63,0x3E,0x00,0x00,0x00,0x00, // o
    0x00,0x00,0x00,0x00,0x00,0x6E,0x33,0x33,0x33,0x33,0x3E,0x30,0x30,0x78,0x00,0x00, // p 0x70
    0x00,0x00,0x00,0x00,0x00,0x3B,0x66,0x66,0x66,0x66,0x3E,0x06,0x06,0x0F,0x00,0x00, // q
    0x00,0x00,0x00,0x00,0x00,0x6E,0x3B,0x33,0x30,0x30,0x30,0x78,0x00,0x00,0x00,0x00, // r
    0x00,0x00,0x00,0x00,0x00,0x3E,0x63,0x38,0x0E,0x03,0x63,0x3E,0x00,0x00,0x00,0x00, // s
    0x00,0x00,0x08,0x18,0x18,0x7E,0x18,0x18,0x18,0x18,0x1B,0x0E,0x00,0x00,0x00,0x00, // t
    0x00,0x00,0x00,0x00,0x00,0x66,0x66,0x66,0x66,0x66,0x66,0x3B,0x00,0x00,0x00,0x00, // u
    0x00,0x00,0x00,0x00,0x00,0x63,0x63,0x36,0x36,0x1C,0x1C,0x08,0x00,0x00,0x00,0x00, // v
    0x00,0x00,0x00,0x00,0x00,0x63,0x63,0x63,0x6B,0x6B,0x7F,0x36,0x00,0x00,0x00,0x00, // w
    0x00,0x00,0x00,0x00,0x00,0x63,0x36,0x1C,0x1C,0x1C,0x36,0x63,0x00,0x00,0x00,0x00, // x
    0x00,0x00,0x00,0x00,0x00,0x63,0x63,0x63,0x63,0x63,0x3F,0x03,0x06,0x3C,0x00,0x00, // y
    0x00,0x00,0x00,0x00,0x00,0x7F,0x66,0x0C,0x18,0x30,0x63,0x7F,0x00,0x00,0x00,0x00, // z
    0x00,0x00,0x0E,0x18,0x18,0x18,0x70,0x18,0x18,0x18,0x18,0x0E,0x00,0x00,0x00,0x00, // {
    0x00,0x00,0x18,0x18,0x18,0x18,0x18,0x00,0x18,0x18,0x18,0x18,0x18,0x00,0x00,0x00, // |
    0x00,0x00,0x70,0x18,0x18,0x18,0x0E,0x18,0x18,0x18,0x18,0x70,0x00,0x00,0x00,0x00, // }
    0x00,0x00,0x3B,0x6E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // ~
    0x00,0x70,0xD8,0xD8,0x70,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00  // DELL
};

void MobileLCD::command(int value) {
    _spi.write(value & 0xFF);
}

void MobileLCD::data(int value) {
    _spi.write(value | 0x100);
}

void MobileLCD::_select() {
    _cs = 0;
}

void MobileLCD::_deselect() {
    _cs = 1;
}

int MobileLCD::width() {
    return _width;
}

int MobileLCD::height() {
    return _height;
}

void MobileLCD::foreground(int v) {
    _foreground = v;
}

void MobileLCD::background(int v) {
    _background = v;
}


/*************************************************************************/

MobileLCD::MobileLCD(PinName mosi, PinName miso, PinName clk, PinName cs, PinName rst)
        : _spi(mosi, miso, clk)
        , _rst(rst)
        , _cs(cs) {

    _width = 130;
    _height = 130;
    foreground(0xFFFFFF);
    background(0x000000);
    reset();
}

void MobileLCD::reset() {

    _cs = 1;
    _rst = 0;
    _spi.format(9);
    _spi.frequency(5000000);

    wait_ms(1);

    _rst = 1;

    wait_ms(1);

    _select();

    command(DISCTL); // display control
    data(0x0);
    data(31);
    data(0x0);

    command(COMSCN);  // scan
    data(0x1);

    command(OSCON); // oscillator on
    command(SLPOUT); // sleep out
    command(PWRCTR); // power control
    data(0x0F);

    command(DISINV); // invert display

    command(VOLCTR); // Voltage control
    data(39);      // contrast setting: 0..63
    data(3);       // resistance ratio

    wait_ms(1);

    command(DATCTL); // data control
    data(0x0); // scan dirs
    data(0x0); // RGB
    data(0x2); // grayscale

    command(DISON);  // turn on the display

    _deselect();

    cls();
}

void MobileLCD::cls()
{
    fill(0, 0, 130, 130, _background);
}

void MobileLCD::_window(int x, int y, int width, int height) 
{
    int x1, x2, y1, y2;
    x1 = x + 0;
    y1 = y + 2;
    x2 = x1 + width - 1;
    y2 = y1 + height - 1;

    command(CASET); // column
    data(x1);
    data(x2);

    command(PASET); // page
    data(y1);
    data(y2);

    command(RAMWR); // start write to ram
}

void MobileLCD::fill(int x, int y, int width, int height, int colour)
{
    int i = 0;

    _select();
    _window(x, y, width, height);

    int r4 = (colour >> (16 + 4)) & 0xF;
    int g4 = (colour >> (8 + 4)) & 0xF;
    int b4 = (colour >> (0 + 4)) & 0xF;

    int d1 = (r4 << 4) | g4;
    int d2 = (b4 << 4) | r4;
    int d3 = (g4 << 4) | b4;

    for(i=0; i<(width*height+1)/2; i++) 
    {
        data(d1);
        data(d2);
        data(d3);
    }
    
    _deselect();
}

void MobileLCD::pixel(int x, int y, int colour) {
    fill(x, y, 1, 1, colour);
}

void MobileLCD::setpixel(int x, int y, int colour) {

    _select();

    int r4 = (colour >> (16 + 4)) & 0xF;
    int g4 = (colour >> (8 + 4)) & 0xF;
    int b4 = (colour >> (0 + 4)) & 0xF;

    int d1 = (r4 << 4) | g4;
    int d2 = (b4 << 4) | r4;
    int d3 = (g4 << 4) | b4;

    command(CASET); // column
    data(x);
    data(x);

    command(PASET); // page
    data(y);
    data(y);

    command(RAMWR); // start write to ram

    data(d1);
    data(d2);
    data(d3);

    _deselect();
}

void MobileLCD::setline(int x0, int y0, int x1, int y1, int color) {
    int dy = y1 - y0;
    int dx = x1 - x0;
    int stepx, stepy;

    if(dy < 0)
    {
        dy = -dy; 
        stepy = -1; 
    }
    else
        stepy = 1;
        
    if(dx < 0)
    {
        dx = -dx; 
        stepx = -1; 
    } 
    else
        stepx = 1;

    dy <<= 1; // dy is now 2*dy
    dx <<= 1; // dx is now 2*dx

    setpixel(x0, y0, color);
    if (dx > dy) 
    {
        int fraction = dy - (dx >> 1); // same as 2*dy - dx
        while (x0 != x1) 
        {
            if (fraction >= 0) 
            {
                y0 += stepy;
                fraction -= dx; // same as fraction -= 2*dx
            }
            x0 += stepx;
            fraction += dy; // same as fraction -= 2*dy
            setpixel(x0, y0, color);
        }
    }
    else
    {
        int fraction = dx - (dy >> 1);
        while (y0 != y1) 
        {
            if (fraction >= 0) 
            {
                x0 += stepx;
                fraction -= dy;
            }
            y0 += stepy;
            fraction += dx;
            setpixel(x0, y0, color);
        }
    }
}

void  MobileLCD::setrect(int x0, int y0, int x1, int y1, unsigned char tmpNum, int color) {

    int xmin, xmax, ymin, ymax;

    // check if the rectangle is to be filled
    if (tmpNum == FILL) 
    {
        // best way to create a filled rectangle is to define a drawing box
        // and loop two pixels at a time
        // calculate the min and max for x and y directions
        xmin = (x0 <= x1) ? x0 : x1;
        xmax = (x0 > x1) ? x0 : x1;
        ymin = (y0 <= y1) ? y0 : y1;
        ymax = (y0 > y1) ? y0 : y1;

        fill(xmin, ymin, (xmax - xmin), (ymax - ymin), color);
    }
    else
    {
        // best way to draw un unfilled rectangle is to draw four lines
        setline(x0, y0, x1, y0, color);
        setline(x0, y1, x1, y1, color);
        setline(x0, y0, x0, y1, color);
        setline(x1, y0, x1, y1, color);
    }
}

void MobileLCD::setcircle(int x0, int y0, int radius, int color) {

    int f = 1 - radius;
    int ddF_x = 0;
    int ddF_y = -2 * radius;
    int x = 0;
    int y = radius;

    setpixel(x0, y0 + radius, color);
    setpixel(x0, y0 - radius, color);
    setpixel(x0 + radius, y0, color);
    setpixel(x0 - radius, y0, color);
    
    while(x < y) 
    {
        if(f >= 0) 
        {
            y--;
            ddF_y += 2;
            f += ddF_y;
        }
        x++;
        ddF_x += 2;
        f += ddF_x + 1;
        
        setpixel(x0 + x, y0 + y, color);
        setpixel(x0 - x, y0 + y, color);
        setpixel(x0 + x, y0 - y, color);
        setpixel(x0 - x, y0 - y, color);
        setpixel(x0 + y, y0 + x, color);
        setpixel(x0 - y, y0 + x, color);
        setpixel(x0 + y, y0 - x, color);
        setpixel(x0 - y, y0 - x, color);
    }
}

void MobileLCD::bitblit(int x, int y, int width, int height, const char* bitstream, int fColor, int bColor) {
    _select();
    _window(x, y, width, height);
    for(int i=0; i<height*width/2; i++) {
    int byte1 = (i*2) / 8;
    int bit1 = (i*2) % 8;    
    int colour1 = ((bitstream[byte1] << bit1) & 0x80) ? fColor : bColor;
    int byte2 = (i*2+1) / 8;
    int bit2 = (i*2+1) % 8;    
    int colour2 = ((bitstream[byte2] << bit2) & 0x80) ? fColor : bColor;

    int r41 = (colour1 >> (16 + 4)) & 0xF;
    int g41 = (colour1 >> (8 + 4)) & 0xF;
    int b41 = (colour1 >> (0 + 4)) & 0xF;
    
    int r42 = (colour2 >> (16 + 4)) & 0xF;
    int g42 = (colour2 >> (8 + 4)) & 0xF;
    int b42 = (colour2 >> (0 + 4)) & 0xF;    
    int d1 = (r41 << 4) | g41;
    int d2 = (b41 << 4) | r42;
    int d3 = (g42 << 4) | b42;                
        data(d1); 
        data(d2); 
        data(d3); 
    }
    _deselect();
}

void MobileLCD::putchar(int x, int y, int size, int value, int fColor, int bColor) { 

    switch(size)
    {
        case SMALL    :    bitblit(x, y, 8, 8, (char*)&(FONT6x8[value - 0x1F][0]), fColor, bColor);
                           break;
        case MEDIUM   :    bitblit(x, y, 8, 8, (char*)&(FONT8x8[value - 0x1F][0]), fColor, bColor);
                           break;
        case LARGE    :    bitblit(x, y, 8, 16, (char*)&(FONT8x16[value - 0x1F][0]), fColor, bColor);
                           break;
    }

}