/*
  Copyright 2014 Johan Wikman

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

#include <sqmbed/interruptin.h>
#include <mbed.h>

namespace
{

uint32_t u_PTA;
uint32_t u_PTB;
uint32_t u_PTC;
uint32_t u_PTD;
uint32_t u_PTE;

}

namespace
{

void deliverInterrupts(uint32_t base, uint16_t value)
{
    int p = 0;

    while (value) {
        int bit = (value & 0x1);

        if (bit) {
            SqMbed::InterruptIn::deliver(static_cast<PinName>(base + p));
        }

        value >>= 1;
        ++p;
    }
}

uint32_t readAndReset(uint32_t* pValue)
{
    uint32_t value;
    int status = 0;

    do {
        value = __LDREXW(pValue);
        status = __STREXW(0, pValue);
    } while (status != 0); // Retry until no-one has modified in between.

    __DMB();

    return value;
}

}

namespace SqMbed
{

void InterruptIn::postponeInterrupt(PinName p)
{
    uint16_t value = static_cast<uint16_t>(p) & 0x1f;

    switch (static_cast<uint32_t>(p) >> GPIO_PORT_SHIFT) {
    case PTA0 >> GPIO_PORT_SHIFT:
        u_PTA |= (1 << value);
        break;

    case PTB0 >> GPIO_PORT_SHIFT:
        u_PTB |= (1 << value);
        break;

    case PTC0 >> GPIO_PORT_SHIFT:
        u_PTC |= (1 << value);
        break;

    case PTD0 >> GPIO_PORT_SHIFT:
        u_PTD |= (1 << value);
        break;

    case PTE0 >> GPIO_PORT_SHIFT:
        u_PTE |= (1 << value);
        break;

    default:
        printf("Unknown PinName: %d", p);
    }
}

void InterruptIn::flushInterrupts()
{
    uint32_t pta = readAndReset(&u_PTA);
    uint32_t ptb = readAndReset(&u_PTB);
    uint32_t ptc = readAndReset(&u_PTC);
    uint32_t ptd = readAndReset(&u_PTD);
    uint32_t pte = readAndReset(&u_PTE);

    if (pta != 0) {
        deliverInterrupts(PTA0, pta);
    }

    if (ptb != 0) {
        deliverInterrupts(PTB0, ptb);
    }

    if (ptc != 0) {
        deliverInterrupts(PTC0, ptc);
    }

    if (ptd != 0) {
        deliverInterrupts(PTD0, ptd);
    }

    if (pte != 0) {
        deliverInterrupts(PTE0, pte);
    }
}

void bindPlatform(HSQUIRRELVM vm)
{
    // Check PinNames.h
    SQBIND_CONSTANT(vm, PTA0);
    SQBIND_CONSTANT(vm, PTA1);
    SQBIND_CONSTANT(vm, PTA2);
    SQBIND_CONSTANT(vm, PTA3);
    SQBIND_CONSTANT(vm, PTA4);
    SQBIND_CONSTANT(vm, PTA5);
    SQBIND_CONSTANT(vm, PTA6);
    SQBIND_CONSTANT(vm, PTA7);
    SQBIND_CONSTANT(vm, PTA8);
    SQBIND_CONSTANT(vm, PTA9);
    SQBIND_CONSTANT(vm, PTA10);
    SQBIND_CONSTANT(vm, PTA11);
    SQBIND_CONSTANT(vm, PTA12);
    SQBIND_CONSTANT(vm, PTA13);
    SQBIND_CONSTANT(vm, PTA14);
    SQBIND_CONSTANT(vm, PTA15);
    SQBIND_CONSTANT(vm, PTA16);
    SQBIND_CONSTANT(vm, PTA17);
    SQBIND_CONSTANT(vm, PTA18);
    SQBIND_CONSTANT(vm, PTA19);
    SQBIND_CONSTANT(vm, PTA20);
    SQBIND_CONSTANT(vm, PTA21);
    SQBIND_CONSTANT(vm, PTA22);
    SQBIND_CONSTANT(vm, PTA23);
    SQBIND_CONSTANT(vm, PTA24);
    SQBIND_CONSTANT(vm, PTA25);
    SQBIND_CONSTANT(vm, PTA26);
    SQBIND_CONSTANT(vm, PTA27);
    SQBIND_CONSTANT(vm, PTA28);
    SQBIND_CONSTANT(vm, PTA29);
    SQBIND_CONSTANT(vm, PTA30);
    SQBIND_CONSTANT(vm, PTA31);
    SQBIND_CONSTANT(vm, PTB0);
    SQBIND_CONSTANT(vm, PTB1);
    SQBIND_CONSTANT(vm, PTB2);
    SQBIND_CONSTANT(vm, PTB3);
    SQBIND_CONSTANT(vm, PTB4);
    SQBIND_CONSTANT(vm, PTB5);
    SQBIND_CONSTANT(vm, PTB6);
    SQBIND_CONSTANT(vm, PTB7);
    SQBIND_CONSTANT(vm, PTB8);
    SQBIND_CONSTANT(vm, PTB9);
    SQBIND_CONSTANT(vm, PTB10);
    SQBIND_CONSTANT(vm, PTB11);
    SQBIND_CONSTANT(vm, PTB12);
    SQBIND_CONSTANT(vm, PTB13);
    SQBIND_CONSTANT(vm, PTB14);
    SQBIND_CONSTANT(vm, PTB15);
    SQBIND_CONSTANT(vm, PTB16);
    SQBIND_CONSTANT(vm, PTB17);
    SQBIND_CONSTANT(vm, PTB18);
    SQBIND_CONSTANT(vm, PTB19);
    SQBIND_CONSTANT(vm, PTB20);
    SQBIND_CONSTANT(vm, PTB21);
    SQBIND_CONSTANT(vm, PTB22);
    SQBIND_CONSTANT(vm, PTB23);
    SQBIND_CONSTANT(vm, PTB24);
    SQBIND_CONSTANT(vm, PTB25);
    SQBIND_CONSTANT(vm, PTB26);
    SQBIND_CONSTANT(vm, PTB27);
    SQBIND_CONSTANT(vm, PTB28);
    SQBIND_CONSTANT(vm, PTB29);
    SQBIND_CONSTANT(vm, PTB30);
    SQBIND_CONSTANT(vm, PTB31);
    SQBIND_CONSTANT(vm, PTC0);
    SQBIND_CONSTANT(vm, PTC1);
    SQBIND_CONSTANT(vm, PTC2);
    SQBIND_CONSTANT(vm, PTC3);
    SQBIND_CONSTANT(vm, PTC4);
    SQBIND_CONSTANT(vm, PTC5);
    SQBIND_CONSTANT(vm, PTC6);
    SQBIND_CONSTANT(vm, PTC7);
    SQBIND_CONSTANT(vm, PTC8);
    SQBIND_CONSTANT(vm, PTC9);
    SQBIND_CONSTANT(vm, PTC10);
    SQBIND_CONSTANT(vm, PTC11);
    SQBIND_CONSTANT(vm, PTC12);
    SQBIND_CONSTANT(vm, PTC13);
    SQBIND_CONSTANT(vm, PTC14);
    SQBIND_CONSTANT(vm, PTC15);
    SQBIND_CONSTANT(vm, PTC16);
    SQBIND_CONSTANT(vm, PTC17);
    SQBIND_CONSTANT(vm, PTC18);
    SQBIND_CONSTANT(vm, PTC19);
    SQBIND_CONSTANT(vm, PTC20);
    SQBIND_CONSTANT(vm, PTC21);
    SQBIND_CONSTANT(vm, PTC22);
    SQBIND_CONSTANT(vm, PTC23);
    SQBIND_CONSTANT(vm, PTC24);
    SQBIND_CONSTANT(vm, PTC25);
    SQBIND_CONSTANT(vm, PTC26);
    SQBIND_CONSTANT(vm, PTC27);
    SQBIND_CONSTANT(vm, PTC28);
    SQBIND_CONSTANT(vm, PTC29);
    SQBIND_CONSTANT(vm, PTC30);
    SQBIND_CONSTANT(vm, PTC31);
    SQBIND_CONSTANT(vm, PTD0);
    SQBIND_CONSTANT(vm, PTD1);
    SQBIND_CONSTANT(vm, PTD2);
    SQBIND_CONSTANT(vm, PTD3);
    SQBIND_CONSTANT(vm, PTD4);
    SQBIND_CONSTANT(vm, PTD5);
    SQBIND_CONSTANT(vm, PTD6);
    SQBIND_CONSTANT(vm, PTD7);
    SQBIND_CONSTANT(vm, PTD8);
    SQBIND_CONSTANT(vm, PTD9);
    SQBIND_CONSTANT(vm, PTD10);
    SQBIND_CONSTANT(vm, PTD11);
    SQBIND_CONSTANT(vm, PTD12);
    SQBIND_CONSTANT(vm, PTD13);
    SQBIND_CONSTANT(vm, PTD14);
    SQBIND_CONSTANT(vm, PTD15);
    SQBIND_CONSTANT(vm, PTD16);
    SQBIND_CONSTANT(vm, PTD17);
    SQBIND_CONSTANT(vm, PTD18);
    SQBIND_CONSTANT(vm, PTD19);
    SQBIND_CONSTANT(vm, PTD20);
    SQBIND_CONSTANT(vm, PTD21);
    SQBIND_CONSTANT(vm, PTD22);
    SQBIND_CONSTANT(vm, PTD23);
    SQBIND_CONSTANT(vm, PTD24);
    SQBIND_CONSTANT(vm, PTD25);
    SQBIND_CONSTANT(vm, PTD26);
    SQBIND_CONSTANT(vm, PTD27);
    SQBIND_CONSTANT(vm, PTD28);
    SQBIND_CONSTANT(vm, PTD29);
    SQBIND_CONSTANT(vm, PTD30);
    SQBIND_CONSTANT(vm, PTD31);
    SQBIND_CONSTANT(vm, PTE0);
    SQBIND_CONSTANT(vm, PTE1);
    SQBIND_CONSTANT(vm, PTE2);
    SQBIND_CONSTANT(vm, PTE3);
    SQBIND_CONSTANT(vm, PTE4);
    SQBIND_CONSTANT(vm, PTE5);
    SQBIND_CONSTANT(vm, PTE6);
    SQBIND_CONSTANT(vm, PTE7);
    SQBIND_CONSTANT(vm, PTE8);
    SQBIND_CONSTANT(vm, PTE9);
    SQBIND_CONSTANT(vm, PTE10);
    SQBIND_CONSTANT(vm, PTE11);
    SQBIND_CONSTANT(vm, PTE12);
    SQBIND_CONSTANT(vm, PTE13);
    SQBIND_CONSTANT(vm, PTE14);
    SQBIND_CONSTANT(vm, PTE15);
    SQBIND_CONSTANT(vm, PTE16);
    SQBIND_CONSTANT(vm, PTE17);
    SQBIND_CONSTANT(vm, PTE18);
    SQBIND_CONSTANT(vm, PTE19);
    SQBIND_CONSTANT(vm, PTE20);
    SQBIND_CONSTANT(vm, PTE21);
    SQBIND_CONSTANT(vm, PTE22);
    SQBIND_CONSTANT(vm, PTE23);
    SQBIND_CONSTANT(vm, PTE24);
    SQBIND_CONSTANT(vm, PTE25);
    SQBIND_CONSTANT(vm, PTE26);
    SQBIND_CONSTANT(vm, PTE27);
    SQBIND_CONSTANT(vm, PTE28);
    SQBIND_CONSTANT(vm, PTE29);
    SQBIND_CONSTANT(vm, PTE30);
    SQBIND_CONSTANT(vm, PTE31);
    SQBIND_CONSTANT(vm, LED_RED);
    SQBIND_CONSTANT(vm, LED_GREEN);
    SQBIND_CONSTANT(vm, LED_BLUE);
    SQBIND_CONSTANT(vm, LED1);
    SQBIND_CONSTANT(vm, LED2);
    SQBIND_CONSTANT(vm, LED3);
    SQBIND_CONSTANT(vm, LED4);
    SQBIND_CONSTANT(vm, SW2);
    SQBIND_CONSTANT(vm, SW3);
    SQBIND_CONSTANT(vm, USBTX);
    SQBIND_CONSTANT(vm, USBRX);
    SQBIND_CONSTANT(vm, D0);
    SQBIND_CONSTANT(vm, D1);
    SQBIND_CONSTANT(vm, D2);
    SQBIND_CONSTANT(vm, D3);
    SQBIND_CONSTANT(vm, D4);
    SQBIND_CONSTANT(vm, D5);
    SQBIND_CONSTANT(vm, D6);
    SQBIND_CONSTANT(vm, D7);
    SQBIND_CONSTANT(vm, D8);
    SQBIND_CONSTANT(vm, D9);
    SQBIND_CONSTANT(vm, D10);
    SQBIND_CONSTANT(vm, D11);
    SQBIND_CONSTANT(vm, D12);
    SQBIND_CONSTANT(vm, D13);
    SQBIND_CONSTANT(vm, D14);
    SQBIND_CONSTANT(vm, D15);
    SQBIND_CONSTANT(vm, I2C_SCL);
    SQBIND_CONSTANT(vm, I2C_SDA);
    SQBIND_CONSTANT(vm, A0);
    SQBIND_CONSTANT(vm, A1);
    SQBIND_CONSTANT(vm, A2);
    SQBIND_CONSTANT(vm, A3);
    SQBIND_CONSTANT(vm, A4);
    SQBIND_CONSTANT(vm, A5);

    SQBIND_CONSTANT(vm, PullNone);
    SQBIND_CONSTANT(vm, PullDown);
    SQBIND_CONSTANT(vm, PullUp);
    SQBIND_CONSTANT(vm, PullDefault);
}

}
