/*
  Copyright 2014 Johan Wikman

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

#include <sqmbed/trigger.h>
#include <sqmbed/pointerstorage.h>

namespace
{

class Triggers : public SqMbed::PointerStorage<SqMbed::Trigger>
{
public:
    static void notify()
    {
        // Do not store pointers here. Who knows, maybe the handler creates a
        // new InterruptIn and then both s_nSpace and s_ppInterrupts may change.
        for (uint32_t i = 0; i < s_nSpace; ++i) {
            SqMbed::Trigger* pTrigger = s_ppObjects[i];

            if (pTrigger) {
                pTrigger->call();
            }
        }
    }
};

}

namespace SqMbed
{

Trigger::Trigger(HSQUIRRELVM vm)
    : m_vm(vm)
    , m_triggered(false)
{
    m_object._type = OT_NULL;

    Triggers::add(this);
}

Trigger::~Trigger()
{
    Triggers::remove(this);
}

bool Trigger::reserve()
{
    return Triggers::reserve();
}

void Trigger::call()
{
    if (m_triggered) {
        m_triggered = false; // Should be atomic.

        int top = sq_gettop(m_vm);
        sq_pushobject(m_vm, m_object); // The function
        sq_pushroottable(m_vm);        // This
        sq_call(m_vm, 1, 0, 0);        // Only this as argument.
        sq_settop(m_vm, top);          // Restore the stack.
    }
}

// static
void Trigger::flushTriggers()
{
    Triggers::notify();
}

void Trigger::attach(HSQOBJECT o)
{
    if (m_object._type != OT_NULL) {
        sq_release(m_vm, &m_object);
        m_object._type = OT_NULL;
    }

    m_object = o;
    sq_addref(m_vm, &m_object);
}

// static
void Trigger::triggered()
{
    // TODO: Should be atomic.
    m_triggered = true;
}

}
