#ifndef SN_SnSDUtils
#define SN_SnSDUtils

#include <stdio.h>
#include <stdint.h>

#include "SnCommWin.h"
#include "SnBitUtils.h"
#include "SnPowerFrame.h"
#include "SnTempFrame.h"
#include "SnHeaderFrame.h"

class SnEventFrame;
class SnConfigFrame;
class SnClockSetFrame;
class SnHeartbeatFrame;

// a namespace-like class to handle the i/o with the SD card

class SnSDUtils {
 public:
    static const char* const kSDdir;
    static const char* const kSDsubDir;
    static const char* const kRunSeqListFilenm;
    static const uint16_t    kMaxSeqNum;
    static const uint16_t    kBadSeqNum;
    static const uint8_t     kFNBufSize=128;
    static const uint8_t     kIOvers; // file I/O version
    static const uint32_t    kMaxSizeOfFileHdr;
    
    typedef      int         (*InitSDFcn)(void);
    static       InitSDFcn   fgDoInit;
    static       bool        fgInitOk;
    
 private:

    static
    bool  InitSDCard(const bool force=false);
    
    static
    void  SetSDCardInitTo(const bool ok) { fgInitOk = ok; }
    
    static
    FILE* OpenSDFile(const char* name, const char* mode, const bool redoDir);
    
    static
    uint16_t GetSeqNum(const uint64_t macadr,
                       const uint32_t run);
    
    static
    const char* GetOutFileName(const uint64_t macadr,
                               const uint32_t run,
                               const uint16_t seq);

    static
    void  DeleteAllFiles(const char* dirname=kSDsubDir);

    static
    void  DeleteFilesOfRun(const uint32_t run);

    static
    void  DeleteFile(FILE*& f, const char* fname);
    
    static
    bool  DeleteDirIfEmpty(const char* dirname);

 private:   
    static char     fgCurFileName[kFNBufSize];
    static FILE*    fgCurFile;
    static uint16_t fgCurSeq;
    static bool     fgNeedToInit;
    
    
 public:
    SnSDUtils() {}
    virtual ~SnSDUtils() {}
    
    static
    bool IsInitOk() {
        return fgInitOk;
    }
    
    static
    void SetDoNeedToInit() { fgNeedToInit = true; }
    
    static
    int   CloseOutputFile(FILE* f) {
        // TODO: set current file pointer to 0, IF f is current file
        const int rt = (f!=0) ? fclose(f) : 0;
        return rt;
    }
    
    static
    FILE* OpenNewOutputFile(const uint64_t macadr,
                            const uint32_t run,
                            const uint16_t minseq,
                            const bool     useRSlist);
    
    static
    FILE* OpenExistingFile(const char* name, const bool setcurrent,
                           const bool redoDir);
    
    static
    bool ClearRunSeqList(const bool useRSlist);
    
    static 
    bool AddToRunSeqList(const uint32_t run,
                         const uint16_t seq,
                         const bool     useRSlist);

    static
    SnCommWin::ECommWinResult
    SendFilesInRunSeqList(SnCommWin* comm,
                          const uint32_t timeout,
                          char* const buf,
                          const uint32_t bsize,
                          const SnConfigFrame& curConf,
                          SnEventFrame& evt,
                          SnPowerFrame& pow);

    static
    SnCommWin::ECommWinResult
    SendFileWithRunSeq(SnCommWin* comm,
                       const uint32_t timeout,
                       char* const buf,
                       const uint32_t bsize,
                       const SnConfigFrame& curConf,
                       SnEventFrame& evt,
                       SnPowerFrame& pow,
                       const uint32_t run,
                       const uint16_t seq);


    static
    SnCommWin::ECommWinResult SendOneFile(const char* dfn,
                                          const bool determineDir,
                                          SnCommWin* comm,
                                          const uint32_t timeout,
                                          char* const buf,
                                          const uint32_t bsize,
                                          const SnConfigFrame& curConf,
                                          SnEventFrame& evt,
                                          SnPowerFrame& pow);

    static
    SnCommWin::ECommWinResult SendPartOfRun(SnCommWin* comm,
                                            const uint32_t timeout,
                                            char* const buf,
                                            const uint32_t bsize,
                                            const SnConfigFrame& curConf,
                                            SnEventFrame& evt,
                                            SnPowerFrame& pow,
                                            const uint32_t run,
                                            const uint16_t minseq,
                                            const uint16_t maxseq);
    
    static
    SnCommWin::ECommWinResult  SendAllOfRun(SnCommWin* comm,
                                            const uint32_t timeout,
                                            char* const buf,
                                            const uint32_t bsize,
                                            const SnConfigFrame& curConf,
                                            SnEventFrame& evt,
                                            SnPowerFrame& pow,
                                            const uint32_t runnum);
    
    static
    SnCommWin::ECommWinResult SendAllFiles(SnCommWin* comm,
                                           const uint32_t timeout,
                                           char* const buf,
                                           const uint32_t bsize,
                                           const SnConfigFrame& curConf,
                                                 SnEventFrame& evt,
                                                 SnPowerFrame& pow,
                                           const char* dirname=kSDsubDir);
    
    static
    DIR*        OpenOrMakeAllDirs(const char* dirname);
    static
    DIR*        OpenOrMakeDir(const char* dirname);
    
    static
    const char* GetCurFileName() { return fgCurFileName; }
    
    static
    FILE*       GetCurFile() { return fgCurFile; }
    
    static
    uint16_t    GetCurSeqNum() { return fgCurSeq; }

    static
    bool        GetRunSeqFromFilename(const char* fn,
                                      uint32_t& run,
                                      uint16_t& seq);

    static
    const char* GetSubDirFor(const uint32_t run, const uint16_t seq,
                             uint32_t& slen, const bool useSeq);

    static
    bool        GetFullFilename(const char* name, std::string& ffn);


    static
    bool        CopyFileToSD(const char* const infn,
                             const char* const outfn,
                             char* const buf,
                             const uint32_t bsize);

    static
    void        PrintFilesInDirs(const char* dirname);
    
    static
    void        GetDirProps(const char* dirname,
                            uint32_t& nfiles,
                            float& totbytes);

    static
    float       GetFreeBytes();

    static
    bool        WriteHeartbeatTo(FILE* file, const SnHeartbeatFrame& htbt);

    static
    bool        WriteTrigWaitWinTime(FILE* file,
                                     SnClockSetFrame& clkset,
                                     const bool isStart);

    static
    bool  WriteEventTo(FILE* efile, char* const evtBuf,
                       const SnEventFrame& evt,
                       const SnConfigFrame& conf);
    
    static
    bool  WriteConfig(FILE* efile,
                      const SnConfigFrame& conf);
    
    template<class T>
    static
    SnCommWin::ECommWinResult WritePowerTo(T& f,
                                           const SnPowerFrame& pow,
                                           uint32_t& pnum) {
        const bool rs = 
            SnHeaderFrame::WriteTo(f, SnHeaderFrame::kPowerCode, 
                                   SnPowerFrame::SizeOf(SnPowerFrame::kIOvers));
        const SnCommWin::ECommWinResult re = pow.WriteTo(f);
        ++pnum;
        return (rs) ? re : SnCommWin::kUndefFail;
    }
    
    template<class T>
    static
    SnCommWin::ECommWinResult WriteTempTo(T& f, const SnTempFrame& tmp) {
        const bool rs = 
            SnHeaderFrame::WriteTo(f, SnHeaderFrame::kTemperatureCode, 
                                   SnTempFrame::SizeOf() );
        const SnCommWin::ECommWinResult re = tmp.WriteTo(f);
        return (rs) ? re : SnCommWin::kUndefFail;
    }

    template<class T>
    static
    SnCommWin::ECommWinResult ReadBlockHeader(T& f,
                                              uint8_t& mcode,
                                              uint32_t& mlen) {
        return SnHeaderFrame::ReadFrom(f, mcode, mlen) ? 
            SnCommWin::kOkWithMsg : SnCommWin::kUndefFail;
    }
    
    template<class T>
    static
    SnCommWin::ECommWinResult WriteFileHeader(T& f, const uint64_t macadr,
                      const uint32_t run, const uint16_t seq) {
        // MUST INCREMENT kIOvers if these writes are altered
        f = SnBitUtils::WriteTo(f, kIOvers);
        f = SnBitUtils::WriteTo(f, macadr);
        f = SnBitUtils::WriteTo(f, run);
        f = SnBitUtils::WriteTo(f, seq);
        return SnCommWin::kOkMsgSent;
    }
    
    template<class T>
    static
    SnCommWin::ECommWinResult ReadFileHeader(T& f, uint64_t& macadr,
                     uint32_t& run, uint16_t& seq,
                     SnPowerFrame* pow=0) {
        SnCommWin::ECommWinResult res = SnCommWin::kOkWithMsg;
        uint8_t Rv=0;
        f = SnBitUtils::ReadFrom(f, Rv);
        f = SnBitUtils::ReadFrom(f, macadr);
        f = SnBitUtils::ReadFrom(f, run);
        f = SnBitUtils::ReadFrom(f, seq);
        if (Rv==2) {
            uint16_t v1, v2;
            f = SnBitUtils::ReadFrom(f, v1);
            f = SnBitUtils::ReadFrom(f, v2);
            if (pow!=0) {
                pow->Set(v1, v2, 0, 0, 0);
            }
        }
        return res;
    }
    
    static
    uint32_t SizeOfFileHeader(const uint8_t rv) {
        if (rv==2) {
            return kMaxSizeOfFileHdr;
        } else {
            return sizeof(uint8_t)+sizeof(uint64_t)+sizeof(uint32_t)+sizeof(uint16_t);
        }
    }
    
    friend class SnSDUtilsWhisperer; // to restrict access to specific functions
};

class SnSDUtilsWhisperer {
    static
    void  DeleteFile(FILE*& f, const char* fname) {
        SnSDUtils::DeleteFile(f, fname);
    }

    static
    void  DeleteAllFiles(const char* dirname=SnSDUtils::kSDsubDir) {
        SnSDUtils::DeleteAllFiles(dirname);
    }
    
    static
    void DeleteFilesOfRun(const uint32_t run) {
        SnSDUtils::DeleteFilesOfRun(run);
    }

    friend class SnCommWin; // the one who's allowed to use me
};

#endif // SN_SnSDUtils
