#include "SnUIDtoMac.h"

#include "SnCRCUtils.h"
#include "SnConstants.h"

void SnUIDtoMac::GetMacAddress( char mac[SnUIDtoMac::kNumMacBytes] ) {
    // this first gets the UID of the ARM chip (LPC1768)
    // it then tries to see if the UID is in the lookup table of known MAC addresses
    // (the lookup table is found in GetMacFromUIDtable)
    // if not, it constructs a fake MAC address given the UID (see GetMacFromCRCofUID)
    
    uint32_t uid[SnLPC1768UID::kUIDlen];
    const bool uidOk = SnLPC1768UID::GetUID(uid);
    if (uidOk) {

#ifdef DEBUG
        printf("got UID = %08X %08X %08X %08X\r\n",
            uid[0], uid[1], uid[2], uid[3]);
#endif
        
        const bool uidInLookupTable = GetMacFromUIDtable(uid, mac);
        if (uidInLookupTable==false) {
#ifdef DEBUG
            printf("UID not in lookup table. Constructing MAC.\r\n");
#endif
            GetMacFromCRCofUID(uid, mac);
        }
#ifdef DEBUG
          else {
            printf("UID found in lookup table.\r\n");
        }
#endif
        
    } else {
#ifdef DEBUG
        printf("mac/uid lookup failed.\r\n");
#endif    
        // otherwise we failed. use the default
        memmove(mac, kDefaultMacAdress, sizeof(kDefaultMacAdress));
    }
}

bool SnUIDtoMac::GetMacFromUIDtable(const uint32_t uid[SnLPC1768UID::kUIDlen],
                                    char mac[kNumMacBytes] ) {
    
    if        ( (uid[0]==0x0A015013) && (uid[1]==0xAE2A8CA0) && (uid[2]==0x52B77AA3) && (uid[3]==0xF5001E84) ) {
        // SST102, stn13
        mac[0]=0x00; mac[1]=0x02; mac[2]=0xF7; mac[3]=0xF2; mac[4]=0x24; mac[5]=0x4B;
        //fake mac address for testing
        //mac[0]=0x00; mac[1]=0x02; mac[2]=0xBB; mac[3]=0xBB; mac[4]=0xBB; mac[5]=0xBB;
        return true;
    } else if ( (uid[0]==0x0A01300B) && (uid[1]==0xAE2A9126) && (uid[2]==0x52B29DC5) && (uid[3]==0xF5001E84) ) {
        // SST104, stn14
        mac[0]=0x00; mac[1]=0x02; mac[2]=0xF7; mac[3]=0xF2; mac[4]=0x0A; mac[5]=0x9C;
        //fake mac address for testing
        //mac[0]=0x00; mac[1]=0x02; mac[2]=0xBB; mac[3]=0xBB; mac[4]=0xBB; mac[5]=0xBB;
        return true;
    } else if ( (uid[0]==0x0A000002) && (uid[1]==0xAE299840) && (uid[2]==0x52A26AFA) && (uid[3]==0xF5001E82) ) {
        // SST110, stn15
        mac[0]=0x00; mac[1]=0x02; mac[2]=0xF7; mac[3]=0xF1; mac[4]=0xF7; mac[5]=0xA8;
        //fake mac address for testing
        //mac[0]=0x00; mac[1]=0x02; mac[2]=0xBB; mac[3]=0xBB; mac[4]=0xBB; mac[5]=0xBB;
        return true;
    } else if ( (uid[0]==0x0A000010) && (uid[1]==0xAE2A9126) && (uid[2]==0x52B2A03B) && (uid[3]==0xF5001E80) ) {
        // SST109, stn17
        mac[0]=0x00; mac[1]=0x02; mac[2]=0xF7; mac[3]=0xF2; mac[4]=0x02; mac[5]=0xC1;
        //fake mac address for testing
        //mac[0]=0x00; mac[1]=0x02; mac[2]=0xBB; mac[3]=0xBB; mac[4]=0xBB; mac[5]=0xBB;
        return true;
    } else if ( (uid[0]==0x0DFFC013) && (uid[1]==0xAE2A9126) && (uid[2]==0x52B2B04A) && (uid[3]==0xF5001E87) ) {
        // SST111, stn18
        mac[0]=0x00; mac[1]=0x02; mac[2]=0xF7; mac[3]=0xF2; mac[4]=0x1A; mac[5]=0x8A;
        //fake mac address for testing
        //mac[0]=0x00; mac[1]=0x02; mac[2]=0xBB; mac[3]=0xBB; mac[4]=0xBB; mac[5]=0xBB;
        return true;
    } else if ( (uid[0]==0x0E00D01E) && (uid[1]==0xAE2A9126) && (uid[2]==0x52B3DEC1) && (uid[3]==0xF5001E80) ) {
        // SST105, stn19
        mac[0]=0x00; mac[1]=0x02; mac[2]=0xF7; mac[3]=0xF2; mac[4]=0x24; mac[5]=0x44;
        //fake mac address for testing
        //mac[0]=0x00; mac[1]=0x02; mac[2]=0xBB; mac[3]=0xBB; mac[4]=0xBB; mac[5]=0xBB;
        return true;
    } else if ( (uid[0]==0x17006008) && (uid[1]==0xAE299840) && (uid[2]==0x52A2BD7C) && (uid[3]==0xF5001E80) ) {
        // SST107, stn31
        mac[0]=0x00; mac[1]=0x02; mac[2]=0xF7; mac[3]=0xF1; mac[4]=0xF6; mac[5]=0x34;
        //fake mac address for testing
        //mac[0]=0x00; mac[1]=0x02; mac[2]=0xBB; mac[3]=0xBB; mac[4]=0xBB; mac[5]=0xBB;
        return true;
    } else if ( (uid[0]==0x12013009) && (uid[1]==0xAE2A8CA7) && (uid[2]==0x52C35FA6) && (uid[3]==0xF5001E85) ) {
        // SST113, stn30
        mac[0]=0x00; mac[1]=0x02; mac[2]=0xF7; mac[3]=0xF1; mac[4]=0xF2; mac[5]=0x12;
        //fake mac address for testing
        //mac[0]=0x00; mac[1]=0x02; mac[2]=0xBB; mac[3]=0xBB; mac[4]=0xBB; mac[5]=0xBB;
        return true;
    } else if ( (uid[0]==0x1200B00D) && (uid[1]==0xAE2A8CA7) && (uid[2]==0x52C36348) && (uid[3]==0xF5001E84) ) {
        // SST112, stn32
        mac[0]=0x00; mac[1]=0x02; mac[2]=0xF7; mac[3]=0xF1; mac[4]=0xF2; mac[5]=0x1A;
        //fake mac address for testing
        //mac[0]=0x00; mac[1]=0x02; mac[2]=0xBB; mac[3]=0xBB; mac[4]=0xBB; mac[5]=0xBB;
        return true;
    } else if ( (uid[0]==0x07FFA008) && (uid[1]==0xAE299840) && (uid[2]==0x52A25792) && (uid[3]==0xF5001E87) ) {
        // SST108, HCR Station 40. this mbed doesn't keep time properly.
        mac[0]=0x00; mac[1]=0x02; mac[2]=0xF7; mac[3]=0xF1; mac[4]=0xE9; mac[5]=0xED;
        //fake mac address for testing
        //mac[0]=0x00; mac[1]=0x02; mac[2]=0xBB; mac[3]=0xBB; mac[4]=0xBB; mac[5]=0xBB;
        return true;
    } else if ( (uid[0]==0x0A003002) && (uid[1]==0xAE299840) && (uid[2]==0x52A269C6) && (uid[3]==0xF5001E81) ) {
        // SST101, CR Station 41
        mac[0]=0x00; mac[1]=0x02; mac[2]=0xF7; mac[3]=0xF1; mac[4]=0xF7; mac[5]=0xC6;
        //fake mac address for testing
        //mac[0]=0x00; mac[1]=0x02; mac[2]=0xBB; mac[3]=0xBB; mac[4]=0xBB; mac[5]=0xBB;
        return true;
    } else if ( (uid[0]==0x07010008) && (uid[1]==0xAE299840) && (uid[2]==0x52A25466) && (uid[3]==0xF5001E82) ) {
        // SST114, bad SST - not deployed. here in case this mbed gets put on a different board
        mac[0]=0x00; mac[1]=0x02; mac[2]=0xF7; mac[3]=0xF1; mac[4]=0xE9; mac[5]=0xE1;
        //fake mac address for testing
        //mac[0]=0x00; mac[1]=0x02; mac[2]=0xBB; mac[3]=0xBB; mac[4]=0xBB; mac[5]=0xBB;
        return true;
    } else if ( (uid[0]==0x05FFB009) && (uid[1]==0xAEC584E3) && (uid[2]==0x56A2CCAA) && (uid[3]==0xF50020C1) ) {
        // 8Ch SST 201
        mac[0]=0x00; mac[1]=0x02; mac[2]=0xF7; mac[3]=0xF1; mac[4]=0xF6; mac[5]=0x34;
        //fake mac address for testing
        //mac[0]=0x00; mac[1]=0x02; mac[2]=0xCC; mac[3]=0xCC; mac[4]=0xCC; mac[5]=0xCC;
        return true;
    } else if ( (uid[0]==0x0BFF900A) && (uid[1]==0xAEBD1906) && (uid[2]==0x5642E55D) && (uid[3]==0xF5002145) ) {
        // 8Ch SST 202
        mac[0]=0x00; mac[1]=0x02; mac[2]=0xF7; mac[3]=0xF2; mac[4]=0xE2; mac[5]=0x4B;
        //fake mac address for testing
        //mac[0]=0x00; mac[1]=0x02; mac[2]=0xCC; mac[3]=0xCC; mac[4]=0xCC; mac[5]=0xCC;
        return true;
    } else if ( (uid[0]==0x0301100F) && (uid[1]==0xAEC510A3) && (uid[2]==0x56AA8DE5) && (uid[3]==0xF50020C4) ) {
        // 8Ch SST 203
        mac[0]=0x00; mac[1]=0x02; mac[2]=0xF7; mac[3]=0xF2; mac[4]=0xE7; mac[5]=0xB9;
        //fake mac address for testing
        //mac[0]=0x00; mac[1]=0x02; mac[2]=0xCC; mac[3]=0xCC; mac[4]=0xCC; mac[5]=0xCC;
        return true;
    } else if ( (uid[0]==0x0EFFB01E) && (uid[1]==0xAEBD1866) && (uid[2]==0x5641DCF6) && (uid[3]==0xF500214B) ) {
        // 8Ch SST 204
        mac[0]=0x00; mac[1]=0x02; mac[2]=0xF7; mac[3]=0xF2; mac[4]=0xE1; mac[5]=0xCE;
        //fake mac address for testing
        //mac[0]=0x00; mac[1]=0x02; mac[2]=0xCC; mac[3]=0xCC; mac[4]=0xCC; mac[5]=0xCC;
        return true;
    } else if ( (uid[0]==0x08FFF002) && (uid[1]==0xAEBD1866) && (uid[2]==0x5641C554) && (uid[3]==0xF5002147) ) {
        // 8Ch SST 205
        mac[0]=0x00; mac[1]=0x02; mac[2]=0xF7; mac[3]=0xF2; mac[4]=0xDA; mac[5]=0x83;
        //fake mac address for testing
        //mac[0]=0x00; mac[1]=0x02; mac[2]=0xCC; mac[3]=0xCC; mac[4]=0xCC; mac[5]=0xCC;
        return true;
    } else if ( (uid[0]==0x02FF7015) && (uid[1]==0xAEC510A3) && (uid[2]==0x56AA8BA8) && (uid[3]==0xF50020C1) ) {
        // 8Ch SST 207
        mac[0]=0x00; mac[1]=0x02; mac[2]=0xF7; mac[3]=0xF2; mac[4]=0xEC; mac[5]=0x55;
        //fake mac address for testing
        //mac[0]=0x00; mac[1]=0x02; mac[2]=0xCC; mac[3]=0xCC; mac[4]=0xCC; mac[5]=0xCC;
        return true;
    } else if ( (uid[0]==0x02FF5014) && (uid[1]==0xAEC510A3) && (uid[2]==0x56AA8C15) && (uid[3]==0xF50020C2) ) {
        // 8Ch SST 208
        mac[0]=0x00; mac[1]=0x02; mac[2]=0xF7; mac[3]=0xF2; mac[4]=0xED; mac[5]=0xFF;
        //fake mac address for testing
        //mac[0]=0x00; mac[1]=0x02; mac[2]=0xCC; mac[3]=0xCC; mac[4]=0xCC; mac[5]=0xCC;
        return true;
    }
    
    // not in the "lookup table"
    return false;
}

void SnUIDtoMac::GetMacFromCRCofUID(const uint32_t uid[SnLPC1768UID::kUIDlen],
                                    char mac[kNumMacBytes] ) {
    // take 2 bytes from the start of the UID and then add on a 32bit CRC of the
    // while UID in order to get a fake 48bit "mac address"
    
    const uint32_t crc = SnCRCUtils::GetCRC32for(uid, SnLPC1768UID::kUIDlen);
 #ifdef DEBUG
    printf("Got UID crc = %u\r\n", crc);
    printf("len=%d, byte=%d\r\n", int(SnLPC1768UID::kUIDlen),
        int(SnLPC1768UID::kUIDlen*sizeof(char)));
 #endif
    union {
        uint32_t x;
        char c[sizeof(uint32_t)];
    } d;
    d.x = uid[0];
    mac[0] = d.c[0];
    mac[1] = d.c[1];
    d.x = crc;
    memmove(mac+2, d.c, SnLPC1768UID::kUIDlen*sizeof(char));
#ifdef DEBUG
    printf("generated MAC = %02X %02X %02X %02X %02X %02X\r\n",
        mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
#endif   
}