#ifndef MOTOR_H
#define MOTOR_H

#include "mbed.h"

#define LEFT_MOTOR 0
#define RIGHT_MOTOR 1

class Motor {
  public: 
    
    Motor(PinName s, PinName d, PinName ms0, PinName ms1, PinName ms2, PinName en, char L_R);
    ~Motor();    
    
    inline void one_step();
    inline void one_step(char new_dir);
    inline void enable();
    inline void disable();
    inline void set_dir(char new_dir);
    inline void set_msLevel(char newLevel);
    inline char get_msLevel();
    
    enum {
        MS_ONE = 1,
        MS_TWO = 2,
        MS_FOUR = 4,
        MS_EIGHT = 8,
        MS_SIXTEEN = 16,
        MS_THIRTYTWO = 32
    };
    
    
  private:
  
    DigitalOut* step_pin;
    DigitalOut* dir_pin; 
    DigitalOut* ms0_pin;
    DigitalOut* ms1_pin;
    DigitalOut* ms2_pin;
    DigitalOut* enable_pin;
    
    char IN, OUT;
    char msLevel;
    
};

Motor::Motor(PinName s, PinName d, PinName ms0, PinName ms1, PinName ms2, PinName en, char L_R) {
    
    step_pin = new DigitalOut(s);
    dir_pin  = new DigitalOut(d);
    ms0_pin  = new DigitalOut(ms0);
    ms1_pin  = new DigitalOut(ms1);
    ms2_pin  = new DigitalOut(ms2);
    enable_pin = new DigitalOut(en);
    
    switch(L_R) {
        case LEFT_MOTOR:
            IN = 1; OUT = 0;
        break;
        case RIGHT_MOTOR:
            IN = 0; OUT = 1;
        break;   
        default:
            // ERROR
        break;
    }
    
    set_dir(IN);
    set_msLevel(MS_THIRTYTWO);
    enable();
}

Motor::~Motor() {
    delete step_pin;
    delete dir_pin;    
}

inline void Motor::enable() {
    *enable_pin = 0;
}

inline void Motor::disable() {
    *enable_pin = 1;
}


inline void Motor::set_dir(char new_dir) {
    *dir_pin = new_dir;
}

inline void Motor::one_step() {
    *step_pin = 0;
    *step_pin = 1;
    wait_us(6);
    *step_pin = 0;
}

inline void Motor::one_step(char new_dir) {
    set_dir(new_dir);
    one_step();
}

inline void Motor::set_msLevel(char newLevel) {
    
    int level = MS_ONE;
    
    switch(newLevel) {
        case MS_ONE:
            level = 0;
            msLevel = MS_ONE;
            break;
        case MS_TWO:
            level = 1;
            msLevel = MS_TWO;
            break;
        case MS_FOUR:
            level = 2;
            msLevel = MS_FOUR;
            break;
        case MS_EIGHT:
            level = 3;
            msLevel = MS_EIGHT;
            break;
        case MS_SIXTEEN:
            level = 4;
            msLevel = MS_SIXTEEN;
            break;
        case MS_THIRTYTWO:
            level = 5;
            msLevel = MS_THIRTYTWO;
            break;
        default:
            // Invalid microstep level!
            break;
    }
    
    *ms0_pin = level & 0x1;
    *ms1_pin = (level >> 1) & 0x1;
    *ms2_pin = (level >> 2) & 0x1;

}

inline char Motor::get_msLevel() {
    return msLevel;
}

#endif