//GPIB_9914.h

//NAT7210 - mbed I/O Port design
BusInOut DATA(p9, p10, p11, p12, p13, p14, p15, p16); // LSB p9 - MSB p16 data bus
DigitalOut RD(p18);
DigitalOut WR(p19);
BusOut RS(p21, p22, p23); //LSB p21 - MSB p23 address bus

//Flags & Macros
#define     _LOW        0
#define     _HIGH       1
#define     _OFF        0
#define     _ON         1
#define     SET_READ    DATA.input();
#define     SET_WRITE   DATA.output();

//NAT7210 9914 Mode READ Registers
#define     ISR0        0           //Interrupt Status Register 0
#define     ISR1        1           //Interrupt Status Register 1
#define     ADSR        2           //Address Register
#define     BSR         3           //Bus Status Register
#define     ISR2        4           //Interrupt Status Register 2(Page-In)
#define     SPSR        5           //Serial Poll Status Register(Page-In)
#define     CPTR        6           //Command Pass Through Register
#define     DIR         7           //Data In Register

//NAT7210 9914 Mode WRITE Registers
#define     IMR0        0           //Interrupt Mask Register 0
#define     IMR1        1           //Interrupt Mask Register 1
#define     IMR2        2           //Interrupt Mask register 2(Page-In)
#define     EOSR        2           //End-Of-String Register(Page-In)
#define     BCR         2           //Bus Control Register(Page-In)
#define     ACCR        2           //Accessory Read Register(Page-In)
#define     AUXCR       3           //Auxiliary Command Register
#define     ADR         4           //Address Register
#define     SPMR        5           //Serial Poll Mode Register
#define     PPR         6           //Parallel Poll Register
#define     CDOR        7           //Command/data Out Register

#define     AUXMR       5           //7210 Auxiliary Mode Register
#define     _sw9914     0x15        //Switch To 9914 Mode(7210 Mode AUXMR)

//ACCR Accessory Read register Map
#define     ICR         0x20
#define     ACCA        0x80
#define     ACCB        0xA0
#define     ACCE        0xC0
#define     ACCF        0xD0
#define     ACCI        0xE0

//AUXCR Auxiliary Commands
#define     _clr_swrst  0x00        //Clear Software Reset
#define     _set_swrst  0x80        //Set Software Reset
#define     _nonvalid   0x01        //Nonvalid Release DAC Holdoff
#define     _valid      0x81        //Valid Release DAC Holdoff
#define     _rhdf       0x02        //Release RFD Holdoff
#define     _clr_hdfa   0x03        //Clear Holdoff On All data
#define     _set_hdfa   0x83        //Set Holdoff On All Data
#define     _clr_hdfe   0x04        //Clear Holdoff On END Only
#define     _set_hdfe   0x84        //Holdoff On END Only
#define     _nbaf       0x05        //New Byte Available False
#define     _clr_fget   0x06        //Clear Force Group Execute Trigger
#define     _set_fget   0x86        //Set Force Group Execute Trigger
#define     _clr_rtl    0x07        //Clear return To Local
#define     _set_rtl    0x87        //Set Return To Local
#define     _feoi       0x08        //Send EOI With The Next Byte
#define     _clr_lon    0x09        //Clear Listen Only
#define     _set_lon    0x89        //Set Listen only
#define     _clr_ton    0x0A        //Clear Talk Only
#define     _set_ton    0x8A        //Set Talk Only
#define     _gts        0x0B        //Go To Standby
#define     _tca        0x0C        //Take Control Asynchronously
#define     _tcs        0x0D        //Take Control Synchronously
#define     _clr_rpp    0x0E        //Clear Request Parallel Poll
#define     _set_rpp    0x8E        //Set Request Parallel Poll
#define     _clr_sic    0x0F        //Clear Send Interface Clear
#define     _set_sic    0x8F        //Set Send Interface Clear
#define     _clr_sre    0x10        //Clear Send Remote Enable
#define     _set_sre    0x90        //Set Send Remote Enable
#define     _rqc        0x11        //Request Control
#define     _rlc        0x12        //Release Control
#define     _clr_dai    0x13        //Clear Disable IMR2, IMR1, And IMR0 Interrupts
#define     _set_dai    0x93        //Set Disable IMR2, IMR1 And IMR0 Interrupts
#define     _pts        0x14        //Pass Through Next Secondary
#define     _clr_stdl   0x15        //Clear Short T1 Delay
#define     _set_std1   0x95        //Set Short T1 Delay
#define     _clr_shdw   0x16        //Clear Shadow Handshaking
#define     _set_shdw   0x96        //Set Shadow Handshaking
#define     _clr_vstdl  0x17        //Clear Very Short T1 Delay
#define     _set_vstdl  0x97        //Set Very Short T1 Delay
#define     _clr_rsv2   0x18        //Clear Request Service bit 2
#define     _set_rsv2   0x98        //Set Request Service bit 2
#define     _sw7210     0x99        //Switch To 7210 Mode
#define     _reqf       0x1A        //Request rsv False
#define     _reqt       0x9A        //Request rsv True
#define     _ch_rst     0x1C        //Chip Reset
#define     _clr_ist    0x1D        //Clear Parallel Poll Flag
#define     _set_ist    0x9D        //Set Parallel Poll Flag
#define     _piimr2     0x1E        //Page-In Interrupt Mask Register 2
#define     _pibcr      0x1F        //Page-In Bus Control Register
#define     _clrpi      0x9C        //Clear_Page-In Register
#define     _pieosr     0x9E        //Page-In End-Of-String Register
#define     _piaccr     0x9F        //Page-In Accessory Register      

//ISR0 Flags
#define     _BI         0x20
#define     _BO         0x10

#define     _CR         0x0D        //Delemiter(CR) --- ignore
#define     _LF         0x0A        //Delimiter(LF)

//GPIB Command
#define     _GTL        0x01        //
#define     _SDC        0x04        //
#define     _PPC        0x05        //
#define     _GET        0x08        //
#define     _TCT        0x09        //
#define     _LLO        0x11        //Local Lock Out
#define     _DCL        0x14        //
#define     _PPU        0x15        //
#define     _SPE        0x18        //
#define     _SPD        0x19        //
#define     _UNL        0x3F        //Unlisten
#define     _UNT        0x5F        //Untalk

#define     _BUFLEN        16

char    databyte;
char    databuff[_BUFLEN];

//Data write to Internal Register
//IN: RS = NAT7210 Internal Register Address
//IN: DATA = data byte to write
void GPIB_write() {
    WR = _LOW;
    wait_us(100);
    WR = _HIGH;
}

//IN: RS = NAT7210 Internal Register Address
//OUT: databyte = get data byte
void GPIB_read() {
    SET_READ;
    RD = _LOW;
    wait_us(100);
    databyte = DATA;
    wait_us(100);
    RD = _HIGH;
    SET_WRITE;
}

void auxcr(unsigned char aux_cmd){
    RS = AUXCR;
    DATA = aux_cmd;
    GPIB_write();
    if(aux_cmd == _gts) wait_ms(10); 
}

void GPIB_in(unsigned char gpib_reg){
    RS = gpib_reg;
    GPIB_read();
}

//Data Output to GPIB bus
//IN: DATA = Data byte to out
void GPIB_DOUT(){
    RS = CDOR;
    GPIB_write();   //Data out
__WAIT_DOUT:
    wait_ms(1);
    GPIB_in(ISR0);
    if ((databyte & _BO) == _BO) return; //Wait for buffer out
    else {
        //lcd.locate(0,0);
        //lcd.printf("Waitng BO-1     ");
        goto __WAIT_DOUT;
    }
}

void GPIB_BOWT(){
__WAIT_DOWT:
    GPIB_in(ISR0);
    if ((databyte & _BO) == _BO) return; //Wait for buffer out
    else {
        //lcd.locate(0,0);
        //lcd.printf("Waitng BO-2     ");
        goto __WAIT_DOWT;
    }
}

char GPIB_BIWT(){
__WAIT_BIWT:
    GPIB_in(ISR0);
    if((databyte & _BI) == 0) {    //wait for Data available
        //lcd.locate(0,1);
        //lcd.printf("Waiting BI      ");
        goto __WAIT_BIWT;
    }
    GPIB_in(DIR);    //Get data byte
    return databyte;
}

void GPIB_out(unsigned char gpib_cmd){
    DATA = gpib_cmd;
    GPIB_DOUT();  //Out to GPIB data bus
}

void GPIB_OLA(unsigned char listener){
    GPIB_out(0x20 + listener);
}

void GPIB_OTA(unsigned char talker){
    GPIB_out(0x40 + talker);
}

void GPIB_DCL(){
    auxcr(_tca); //ATNL
    GPIB_out(_UNL);
    GPIB_out(_UNT);
    GPIB_out(_DCL);
    auxcr(_gts); //ATNH
}

void GPIB_SDC(unsigned char listener){
    auxcr(_tca); //ATNL
    GPIB_out(_UNL);
    auxcr(_set_ton); //MTA
    GPIB_OLA(listener);
    GPIB_out(_SDC);   
    auxcr(_gts); //ATNH
}

void GPIB_GPOUT(char *send_buffer){
int i;
    for(i = 0; i < (_BUFLEN - 1); i++){
        if(send_buffer[i] == NULL) goto __GPOUT_EXIT;
        else {
            GPIB_out(send_buffer[i]);
        }
    }
__GPOUT_EXIT:
    GPIB_out(_CR); //Delimiter
    auxcr(_feoi);
    GPIB_out(_LF);   
}

int GPIB_GPIN(char *rcv_buffer){
int i;
    auxcr(_set_hdfa);
    for(i = 0; i < (_BUFLEN - 1); i++){
        rcv_buffer[i] = GPIB_BIWT(); //Get byte
        if(rcv_buffer[i] == _LF) { //Delimiter?
            rcv_buffer[i] = NULL;
            goto __GPIN_EXIT;            
        } else if(rcv_buffer[i] == _CR) { //Ignor
            i--;
        }
        auxcr(_rhdf); //Next
    }                
__GPIN_EXIT:
    auxcr(_rhdf);
    auxcr(_clr_hdfa);
    return i;
} 

void GPIB_IFC(){
    auxcr(_tca); //ATNL
    wait_ms(1);
    auxcr(_set_sic);
    wait_us(100);
    auxcr(_clr_sic);
    wait_ms(1);
    auxcr(_gts); //ATNH
}
    
void GPIB_OUTPUT(char *send_buffer, unsigned char listener){  
    auxcr(_tca); //ATNL
    GPIB_out(_UNL);
    auxcr(_set_ton); //MTA
    GPIB_OLA(listener);
    auxcr(_gts); //ATNH
    GPIB_BOWT(); //wait talker
    GPIB_GPOUT(send_buffer);
}

int GPIB_ENTER(char *rcv_buffer, unsigned char talker){
    auxcr(_tca); //ATNL
    GPIB_out(_UNL);
    GPIB_OTA(talker);
    auxcr(_set_lon); //MLA
    auxcr(_gts);
    return GPIB_GPIN(rcv_buffer);
}
    
void GPIB_REMOTE(unsigned char listener){
    auxcr(_set_sre); //REN
    auxcr(_tca); //ATNL
    GPIB_out(_UNL);
    auxcr(_set_ton); //MTA
    GPIB_OLA(listener);
    auxcr(_gts);
}

void GPIB_LOCAL(unsigned char listener){
    auxcr(_tca); //ATNL
    GPIB_out(_UNL);
    auxcr(_set_ton); //MTA
    GPIB_OLA(listener);
    GPIB_out(_GTL);
    auxcr(_gts); //ATNH
}    
    
void GPIB_TRIGER(unsigned char listener){
    auxcr(_tca); //ATNL
    GPIB_out(_UNL);
    auxcr(_set_ton); //MTA
    GPIB_OLA(listener);
    GPIB_out(_GET);
    auxcr(_gts); //ATNH
}

void GPIB_Init(unsigned char my_address){
    SET_WRITE;
         
    RD = _HIGH;
    WR = _HIGH;
    
    RS = AUXMR;
    DATA = _sw9914;      //Switch To 9914 Mode
    GPIB_write();
    
    //RS = AUXCR;
    //DATA = _ch_rst;     //Chip Reset and pon
    //GPIB_write();
    
    RS = AUXCR;
    DATA = _set_swrst;   //Set Software Reset
    GPIB_write();
    
    wait_ms(1);
    
    RS = AUXCR;
    DATA = _clr_swrst;   //Clear Software Reset
    GPIB_write();
    
    RS = ADR;
    DATA = my_address;  //Set My Address
    GPIB_write();

    RS = AUXCR;
    DATA = _set_dai;    //Set Disable All Interrupt
    GPIB_write();
    
    RS = ISR0;          //Clear Interrupt Status Register 0
    GPIB_read();
    
    RS = ISR1;          //Clear Interrupt Status Register 1
    GPIB_read();        
    
    RS = SPMR;          //Clear Serial Poll register
    DATA = 0;
    GPIB_write();

    RS = PPR;           //Clear Parallel Poll register
    DATA = 0;
    GPIB_write();    
        
    RS = IMR0;          //Disable interrupt IMR0
    DATA = 0;
    GPIB_write();

    RS = IMR1;          //Disable interrupt IMR1
    DATA = 0;
    GPIB_write();

    RS = DIR;           //Clear Data In Register
    GPIB_read();

    RS = CDOR;          //Clear Command/Data Out Register
    DATA = 0;
    GPIB_write();
    
    RS = AUXCR;         //Set T1 Delay
    DATA = _set_std1;
    GPIB_write(); 
}