/*******************************************************************************
* RenBED DC Motor Drive for RenBuggy                                           *
* Copyright (c) 2014 Jon Fuge                                                  *
*                                                                              *
* Permission is hereby granted, free of charge, to any person obtaining a copy *
* of this software and associated documentation files (the "Software"), to deal*
* in the Software without restriction, including without limitation the rights *
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell    *
* copies of the Software, and to permit persons to whom the Software is        *
* furnished to do so, subject to the following conditions:                     *
*                                                                              *
* The above copyright notice and this permission notice shall be included in   *
* all copies or substantial portions of the Software.                          *
*                                                                              *
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR   *
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,     *
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE  *
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER       *
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,*
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN    *
* THE SOFTWARE.                                                                *
*                                                                              *
* DCMotorDrive.h                                                               *
*                                                                              *
* V1.0 06/01/2014 First issue of code                      Jon Fuge            *
*******************************************************************************/

#ifndef _DCMOTORDRIVE_C
#define _DCMOTORDRIVE_C

#include "mbed.h"
#include "DCMotorDrive.h"

DCMotorDrive::DCMotorDrive(PinName MotorOut, PinName BrakeOut, PinName SensorIn):
    _MotorPin(MotorOut), _BrakePin(BrakeOut), _SensorIn(SensorIn)
{
    _MotorPin.period_ms(PWM_PERIOD);
    SetMotorPwmAndRevolutions(0,0);
    _SensorIn.mode(PullUp);
    _SensorIn.fall(this, &DCMotorDrive::Counter);
    _BrakePin = 1;
    PulseClock.start();
    PulseClock.reset();
    ResetOdometer();
}

void DCMotorDrive::SetMotorPwm(int PwmValue)
{
    if (PwmValue <= 0)
    {
        _MotorPin.pulsewidth_us(0);
        _BrakePin = 1;
    } else
    {
        _BrakePin = 0;
        timeout.attach(this, &DCMotorDrive::Stall, MOTOR_STALL_TIME);
        _MotorPin.pulsewidth_us(PwmValue);
    }
}

void DCMotorDrive::SetMotorPwmAndRevolutions(int PwmValue, int MaxRevolutions)
{
    SetMotorPwm(PwmValue);
    RevolutionLimit = MaxRevolutions;
}

int DCMotorDrive::GetLastPulseTime(void)
{
    return(LastPulseTime);
}

int DCMotorDrive::GetRevolutionsLeft(void)
{
    return(RevolutionLimit);
}
void DCMotorDrive::ResetOdometer(void)
{
    RotationCounter = 0;
}

int DCMotorDrive::ReadOdometer(void)
{
    return(RotationCounter);
}

void DCMotorDrive::Counter(void)
{
    LastPulseTime = PulseClock.read_us();
    PulseClock.reset();
    timeout.detach();
    timeout.attach(this, &DCMotorDrive::Stall, MOTOR_STALL_TIME);
    RotationCounter++;
    if (RevolutionLimit != 0)
        if (--RevolutionLimit == 0)
        {
            SetMotorPwmAndRevolutions(0,0);
            _BrakePin = 1;
            timeout.detach();
        }
}

void DCMotorDrive::Stall(void)
{
    SetMotorPwmAndRevolutions(0,0);
    _BrakePin = 1;
    timeout.detach();
}

#endif