/**
 * Program an AVR with an mbed.
 * The mbed is connected to an ATMEGA328 via ISP.
 * The PC has a serial connection to the mbed.
 * The AVR program on the PC is send via the serial connection to the mbed using a python script.
 * The mbed then programs the ATMEGA328 via ISP.
 */

#include "AVR910_Serial.h"
#include "mbed.h"

// serial connection to a pc
Serial pc(USBTX, USBRX);
// ISP programmer
AVR910 mAVRISP(PB_15, PB_14, PB_13, PB_2); //mosi, miso, sclk, nreset.

int main()
{
    int success  = -1;
    int response =  0;

    //Read the vendor code [0x1E == Atmel].
    response = mAVRISP.readVendorCode();

    if (response == ATMEL_VENDOR_CODE) {
        pc.printf("Microcontroller is an Atmel [0x%02x]\n", response);
    } else if (response == DEVICE_LOCKED) {
        pc.printf("Device is locked\n");
        return -1;
    } else {
        pc.printf("Microcontroller is not an Atmel\n");
        return -1;
    }

    //Read part family and flash size - see datasheet for code meaning.
    response = mAVRISP.readPartFamilyAndFlashSize();

    if (response == 0xFF) {
        pc.printf("Device code erased or target missing\n");
    } else if (response == 0x01) {
        pc.printf("Device locked\n");
        return -1;
    } else {
        pc.printf("Part family and flash size code is: 0x%02x\n", response);
    }

    //Read part number.
    response = mAVRISP.readPartNumber();

    if (response == 0xFF) {
        pc.printf("Device code erased or target missing\n");
    } else if (response == 0x02) {
        pc.printf("Device locked\n");
        return -1;
    } else {
        pc.printf("Part number code is: 0x%02x\n", response);
    }

    // signal the python script it can start sending the file size
    printf("@");

    int hs, ls;
    while (!pc.readable());
    hs = pc.getc();
    while (!pc.readable());
    ls = pc.getc();
    int dataSize = (hs<<8) + ls;

    // signal the python script it can start sending the file
    printf("#");

    char *myFile;
    myFile = new char [dataSize];
    int readChar=0;
    while (readChar < dataSize) {
        if (pc.readable()) {
            myFile[readChar]=pc.getc();
            readChar++;
        }
    }

    // program the atmega
    success = mAVRISP.programData(myFile, dataSize, ATMEGA328P_PAGESIZE, ATMEGA328P_NUM_PAGES);

    if (success < 0) {
        printf("Programming failed.\n");
    } else {
        printf("Programming was successful!\n");
    }
    
    // signal the python script it can stop
    printf("$");

}

/* Python script to be run on the pc:
import serial
import sys
import os

fileName=sys.argv[1]
fileSize=os.path.getsize(fileName)

serdev='/dev/ttyACM0'
s = serial.Serial(serdev)

while True:
    if s.readable():
        c = s.read()
        if '@' in c:
            hs = (fileSize>>8) & 0xff
            s.write(chr(hs))
            ls = fileSize & 0xff
            s.write(chr(ls))
        elif '#' in c:
            with open(fileName, "rb") as f:
                byte = f.read(1)
                while byte != "":
                    s.write(byte)
                    byte = f.read(1)
        elif '$' in c:
            break
        else:
            sys.stdout.write(c)
*/

/* avr test file:
#define F_CPU 1000000UL // 1 MHz
#include <avr/io.h>
#include <util/delay.h>

int main(void)
{
// Set Port D pins as all outputs
    DDRD = 0xff;

// Set all Port D pins as HIGH
    while (1) {
        PORTD = 0xFF;
        _delay_ms(500.);
        PORTD = 0x00;
        _delay_ms(500.);
    }
    return 1;
}

The avr test file needs to be compiled, linked (make + make hex) to get a .hex file.
This hex needs to be transformed into a .bin file via hex2bin.
Make sure the mbed is up and running with the code in this file.
Then uploaded the .bin via the python script: python scriptName programName.bin
A reset of the mbed board may be needed.
*/