#include "mbed.h"
#include "nrf24l01.h"

Serial pc(USBTX, USBRX); // tx, rx

DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);
DigitalOut led4(LED4);
DigitalIn     mode(p20);

DigitalOut     nRF_CSN(p8);
DigitalOut     nRF_CE(p11);
DigitalIn     nRF_IRQ(p12);
SPI                    nRF_spi(p5, p6, p7);

int main() {
    uint8_t dump[35];
    unsigned char scratch[5];
    pc.baud(115200);
    pc.printf("mbed: nRF24L01 tranceiver\r\n");

    #define _NRF24L01P_SPI_MAX_DATA_RATE     10000000
    nRF_spi.frequency(_NRF24L01P_SPI_MAX_DATA_RATE/5);     // 2Mbit, 1/5th the maximum transfer rate for the SPI bus
  nRF_spi.format(8,0);                                   // 8-bit, ClockPhase = 0, ClockPolarity = 0

    //scratch[0] = 0x01;
    //nrf24l01_write_register(nrf24l01_RF_SETUP, scratch , 1);    // -18 dbm, air rate 1 mbps
    
    pc.printf("Reading all registers:\r\n");
    nrf24l01_get_all_registers(dump);
    for(int i=0; i<sizeof(dump); i++){
        pc.printf("0x%02x\r\n",dump[i]);
    }
    
    /* "fork" */
    if(mode){
        // TX Mode
        pc.printf("Set to transmitter mode p20 high\r\n");
        //vars
        unsigned char data; //register to hold letter sent and received
        unsigned int count; //counter for for loop
        // init
        nrf24l01_initialize_debug(false, 1, true); //initialize the 24L01 to the debug configuration as TX, 1 data byte, and auto-ack enabled
        //main program loop
        while(1)
        {
            //check UART status register to see if data has been received.  if so, process
            while(pc.readable())
            {
                data = pc.getc(); //get data from UART
                nrf24l01_write_tx_payload(&data, 1, true); //transmit received char over RF

                //wait until the packet has been sent or the maximum number of retries has been active
                while(!(nrf24l01_irq_pin_active() && (nrf24l01_irq_tx_ds_active() || nrf24l01_irq_max_rt_active())));

                //check to see if the maximum number of retries has been hit.  if not, wait for the RX device
                // to send the char back.  if so, assume the packet is lost and send "*" back to UART
                if(!nrf24l01_irq_max_rt_active())
                {
                    nrf24l01_irq_clear_all(); //clear all interrupts in the 24L01
                    nrf24l01_set_as_rx(true); //change the device to an RX to get the character back from the other 24L01

                    //wait a while to see if we get the data back (change the loop maximum and the lower if
                    //  argument (should be loop maximum - 1) to lengthen or shorten this time frame
                    for(count = 0; count < 25000; count++)
                    {
                        //check to see if the data has been received.  if so, get the data and exit the loop.
                        //  if the loop is at its last count, assume the packet has been lost and set the data
                        //  to go to the UART to "?".  If neither of these is true, keep looping.
                        if((nrf24l01_irq_pin_active() && nrf24l01_irq_rx_dr_active()))
                        {
                            nrf24l01_read_rx_payload(&data, 1); //get the payload into data
                            break;
                        }
                        
                        //if loop is on its last iteration, assume packet has been lost.
                        if(count == 24999)
                            data = '?';
                    }
                    
                    nrf24l01_irq_clear_all(); //clear interrupts again
                    pc.printf("%c", data); //print the received data (or ? if none) to the screen
                
                    wait_us(130); //wait for receiver to come from standby to RX
                    nrf24l01_set_as_tx(); //resume normal operation as a TX
                }
                else
                {
                    nrf24l01_flush_tx(); //get the unsent character out of the TX FIFO
                    nrf24l01_irq_clear_all(); //clear all interrupts
                    pc.printf("*"); //print "*" to the screen to show that the receiver did not receive the packet
                }
                                        
                led1 = !led1; //toggle the on-board LED as visual indication that the loop has completed
            }
        }
/* ########################################################################## */        
    }else{
        // RX Mode
        pc.printf("Set to receiver mode p20 low\r\n");
        unsigned char data; //register to hold letter received and sent
        // Init
        nrf24l01_initialize_debug(true, 1, true); //initialize the 24L01 to the debug configuration as RX, 1 data byte, and auto-ack enabled        
        //main program loop
        while(1)
        {
            //wait until a packet has been received
            while(!(nrf24l01_irq_pin_active() && nrf24l01_irq_rx_dr_active()));
            
            nrf24l01_read_rx_payload(&data, 1); //read the packet into data
            nrf24l01_irq_clear_all(); //clear all interrupts in the 24L01
            
            pc.putc(data);
            wait_us(130); //wait for the other 24L01 to come from standby to RX
            
            nrf24l01_set_as_tx(); //change the device to a TX to send back from the other 24L01
            nrf24l01_write_tx_payload(&data, 1, true); //transmit received char over RF
            
            //wait until the packet has been sent or the maximum number of retries has been reached
            while(!(nrf24l01_irq_pin_active() && (nrf24l01_irq_tx_ds_active() || nrf24l01_irq_max_rt_active())));

            nrf24l01_irq_clear_all(); //clear interrupts again
            nrf24l01_set_as_rx(true); //resume normal operation as an RX

            led1 = !led1; //toggle the on-board LED as visual indication that the loop has completed
        }
    }
}
