#include "mbed.h"

DigitalOut myled(LED1);
DigitalOut convst(p18);     // Conversion Start (active low)
DigitalOut cs(p16);         // Chip Select (active low)
DigitalIn  eoc(p20);        // End of Conversion (active low)
SPI spi(p11, p12, p13);     // mosi, miso, sclk
Serial pc(USBTX, USBRX);    // tx, rx

#define AIN0  0u
#define AIN1  1u
#define AIN2  2u
#define AIN3  3u
#define AIN4  4u
#define AIN5  5u
#define AIN6  6u
#define AIN7  7u
#define AIN8  8u
#define AIN9  9u
#define AIN10 10u
#define AIN11 11u
#define AIN12 12u
#define AIN13 13u
#define AIN14 14u
#define AIN15 15u

#define SCAN_TO_N               0u // Scans channels 0 through N.
#define SCAN_N_TO               1u // Scans channels N through the highest numbered channel.
#define SCAN_N_AVG              2u // Scans channel N repeatedly. The averaging register sets the number of results.
#define SCAN_OFF                3u // No scan. Converts channel N once only

#define CLKSEL_INT_INT_CNVST    0u // Internal clock, internal aqcuisition, convst
#define CLKSEL_INT_EXT_CNVST    1u // Internal clock, external aqcuisition convst, convst
#define CLKSEL_INT_INT_ADC      2u // Internal clock, internal aqcuisition, extra channel
#define CLKSEL_EXT_EXT_ADC      3u // External clock, external aqcuisition sclk, extra channel

#define REFSEL_INT_SLEEP        0u // Reference off after scan; need wake-up delay
#define REFSEL_EXT              1u // Reference off; no wake-up delay.
#define REFSEL_INT_NOSLEEP      2u // Reference always on; no wake-up delay.
#define REFSEL_RESERVED         3u // Reserved. Do not use.

#define NAVG_4                  0u // Performs four conversions and returns the average for each requested result.
#define NAVG_8                  1u // Performs eight conversions and returns the average for each requested result.
#define NAVG_16                 2u // Performs 16 conversions and returns the average for each requested result.
#define NAVG_32                 3u // Performs 32 conversions and returns the average for each requested result.
                            
#define NSCAN_4                 0u // Scans channel N and returns four results.
#define NSCAN_8                 1u // Scans channel N and returns eight results.
#define NSCAN_12                2u // Scans channel N and returns 12 results.
#define NSCAN_16                3u // Scans channel N and returns 16 results.

uint8_t compile_reset(bool fifo_only){
    if(fifo_only)
        return 0x18;
    else
        return 0x10;
}

uint8_t compile_conversion(uint8_t ch, uint8_t scan){
    return 0x80 | ((ch&0xF)<<3)|((scan&0x3)<<1);
}

uint8_t compile_setup(uint8_t clk, uint8_t ref){
    return 0x40 | ((clk&0x3)<<4)|((ref&0x3)<<2);
}

uint8_t compile_avg(bool avgon, uint8_t navg, uint8_t nscan){
    uint8_t y = 0x20;
    if(avgon) y |= 0x10;
    y |= ((navg&0x3)<<2)|(nscan&0x3);
    return y;
}

int main() {
    int data[4];
    convst = 1;
    cs = 1;

    // The max11626 uses normal 8 bit spi for the registers.
    // However, >8 bit reads can be used to read the data (mosi would be 0 during read).
    spi.format(8,3);
    spi.frequency(10000);
    
    // MAX11626 addressing.
    // Addressing is performed by the leading zeros.
    // - no leading zero (0x80) would be the conversion register.
    // - one leading zeros (0x40) would be the setup register.
    // - two leading zeros (0x20) would be the averaging register.
    // - 3 leading zeros (0x10) would be the reset register.
    // Transfers are MSB first.
    // Reading data is always safe when you set MOSI to 0.

    // Let the scope settle after reset
    wait(1);

    while(1) {
        // Write reset full.
        cs = 0;
        spi.write(compile_reset(0)); 
        cs = 1;
        
        wait_us(100);
        
        // Write conversion
        cs = 0;
        spi.write(compile_conversion(AIN3, SCAN_TO_N)); 
        cs = 1;

        wait_us(100);
        
        convst = 0;
        wait_us(100);
        convst = 1;
        
        wait_us(100);
        
        cs = 0;
        // Read 4
        for(int i=0; i<4; i++){
            data[i] = 0;
            data[i] = spi.write(0);
            data[i] <<= 8;
            data[i] |= spi.write(0);
        }
        cs = 1;
        
        pc.printf("%05d, %05d, %05d, %05d\n",data[0],data[1],data[2],data[3]);
        wait(0.2);
        
        myled = !myled;
    }
}
