 
 /**
 * @file    xifo.h
 * @brief   xifo circular buffer
 * @details xifo supplies object oriented circular buffer with 32 bit size elements. \n
 *                     To use either as FIFO (First In First Out) or as FILO (First In Last Out)
 *
 * @author    Jeroen Lodder
 * @date        April 2013
 * @version 2
 *
 * @{
 */
 
 /**
 *    Code has been tested on Cortex M0 (LPC1114)
 *  Performance table, number of core clocks
 *    Measured with a timer before and after
 *    r1 = timer->TC
 *        test_function 
 *  r2 = timer->TC
 *
 *  Function    Speed        Worst Case
 *
 *  write            69            71
 *  read_mr        59      59
 *  read_lr        63      70
 *  pop_mr        76      78
 *    pop_lr        45      45
 *
 */

#ifndef _xifo_H_
#define _xifo_H_

#include <inttypes.h>

#if defined(__arm__)  || defined(__DOXYGEN__)
#pragma anon_unions                /**< Allow unnamed unions */
#endif

/**
 * @brief   Circular Buffer object.
 * @details This struct holds the object of a circular buffer
 */
typedef struct  {
/* Pointers: */
        uint32_t *startpool;  /**< @brief First element in pool */
        uint32_t *endpool;        /**< @brief Last element in pool */
        uint32_t *read;                /**< @brief Read pointer */
        uint32_t *write;            /**< @brief Write pointer */
/* Variables: */
        uint32_t full;        /**< @brief Flag indicating buffer is full */
    uint32_t elementcount;/**< @brief Number of elements used */
        uint32_t size;                /**< @brief Size of buffer */
/* Locally used in functions to prevent stack use: */
        /**< @brief union to prevent lvalue typecasting */  
        union {                         
            uint32_t temp;            /**< @brief temp variable and padding for even sized block */
            uint32_t *ptemp;         /**< @brief temp variable and padding for even sized block */
        };
}xifo_t;

/**
 * @brief   Circular Buffer memory pool type.
 */
typedef unsigned int xifo_pool_t;

#ifdef __cplusplus
extern "C" {
#endif
/* xifo Common */
void xifo_init(xifo_t *c, uint32_t size, uint32_t *startpool);
void xifo_clear(xifo_t *c);
uint32_t xifo_write(xifo_t *c, uint32_t data);
/* FIFO use */
uint32_t xifo_read_lr(xifo_t *c, uint32_t index);
uint32_t xifo_pop_lr(xifo_t *c);
/* LIFO use */
uint32_t xifo_read_mr(xifo_t *c, uint32_t index);
uint32_t xifo_pop_mr(xifo_t *c);
/* Extractors */
uint32_t xifo_get_size(xifo_t *c);
uint32_t xifo_get_used(xifo_t *c);
uint32_t xifo_get_full(xifo_t *c);
uint32_t xifo_get_free(xifo_t *c);
#ifdef __cplusplus
}
#endif
#endif //_xifo_H_

/** @} */
