 /**    
 * @file    main.cpp
 * @brief   Bipolar Steppr Motor Control with dual H Bridge
 * @details Software plays stepper motor lookup tables for dual h bridge control.
 *                  
 *
 * @Author  Jeroen Lodder
 * @Date    September 2013
 * @version 1
 *
 */

#include "mbed.h"

/* IO declarations */
    /* LED coil indicators */
    BusOut leds(LED1, LED2, LED3, LED4);
    /* H Bridge output IO */
    BusOut motor(p21, p22, p23, p24, p25, p26, p27, p28);
    /* MOSFET io 
     *  FET         L1  L2  H1  H2      H3  H4  L3  L4
     *  Motor       MA                  MB          
     *  mbed        p21 p22 p23 p24     p25 p26 p27 p28
     * ------------------------ V+
     * |      |       |      |       
     * \H1    \H2     \H3    \H4 
     * |--MA--|       |--MB--|       
     * \L1    \L2     \L3    \L4 
     * |      |       |      |       
     * ------------------------ V- 
     */

/* Single phase control */
const uint8_t motortable_singlephase[4]     = { 
    0x09    ,
    0x90    ,
    0x06    ,
    0x60    
};

/* Dual phase motor control */
const uint8_t motortable_dualphase[4]       = { 
    0x99 ,
    0x96 ,
    0x66 ,
    0x69
};

/* Half step motor control */
const uint8_t motortable_half_step[8]           =    {  
    0x09 ,
    0x99 ,
    0x90 ,
    0x96 ,
    0x06 ,
    0x66 ,
    0x60 ,
    0x69
};

/* LED coil indicators tables */
const uint8_t ledtable_singlephase[4]   =   {   0x4,0x2,0x8,0x1 };
const uint8_t ledtable_dualphase[4]     =   {   0x6,0xA,0x9,0x5 };
const uint8_t ledtable_half_step[8]     = { 0x4,0x6,0x2,0xA,0x8,0x9,0x1,0x5 };

/* Fet driver definitions */
    /* Turn off time fet driver in micro seconds (uS) */
    #define TURN_ON_TIME_FET_DRIVER 750

/* Control definitions */
    /* Mutliplier of speed wait time setting */ 
    #define SPEED_SCALING   0.05

int main(void) {
/* Control declarations */
    /* Direction LUT are played */
    DigitalIn direction( p20 ); 
    /* Step pulse input if step mode */
    DigitalIn step( p18 );
    /* Speed potmeter input */
    AnalogIn    speed( p19 );

/* config variables */
    /* Autostep or stepmode with step input */
    uint8_t autostep = 1;
    /* Halfstep or Dual phase mode */
    uint8_t halfstep = 0;
 
    
/* Stepper motor control loop */    
    while(1) {
        if( halfstep ){
            if(direction){
                for(int i=0; i<8; i++) {
                    leds    = ledtable_half_step[i];
                    motor =  0;
                    wait_us(TURN_ON_TIME_FET_DRIVER);
                    motor =  motortable_half_step[i];
                    /* Delay */
                    if(!autostep){                      
                        while(!step);
                        wait(0.001);
                        while(step);
                    }else{
                        wait((speed*SPEED_SCALING)+0.001);
                    }
                }
            }else{
                for(int i=7; i>-1; i--) {
                    leds    = ledtable_half_step[i];
                    motor =  0;
                    wait_us(TURN_ON_TIME_FET_DRIVER);
                    motor =  motortable_half_step[i];
                    /* Delay */
                    if(!autostep){                      
                        while(!step);
                        wait(0.001);
                        while(step);
                    }else{
                        wait((speed*SPEED_SCALING)+0.001);
                    }
                }
            }
        }else{ /* Half step */
            if(direction){
                for(int i=0; i<4; i++) {
                    leds    = ledtable_dualphase[i];
                    motor =  0;
                    wait_us(TURN_ON_TIME_FET_DRIVER);
                    motor =  motortable_dualphase[i];
                    /* Delay */
                    if(!autostep){                      
                        while(!step);
                        wait(0.001);
                        while(step);
                    }else{
                        wait((speed*SPEED_SCALING)+0.001);
                    }
                }
            }else{
                for(int i=3; i>-1; i--) {
                    leds    = ledtable_dualphase[i];
                    motor = 0;
                    wait_us(TURN_ON_TIME_FET_DRIVER);
                    motor =  motortable_dualphase[i];
                    /* Delay */
                    if(!autostep){                      
                        while(!step);
                        wait(0.001);
                        while(step);
                    }else{
                        wait((speed*SPEED_SCALING)+0.001);
                    }
                }
            }
        }
    }
}
