#include "mbed.h"
#include "main.h"
#include "sx1272-hal.h"
#include "debug.h"

/* CONFIGURATION */

#define NODE_ADDR 0x411A

/* Set this flag to '1' to display debug messages on the console */
#define DEBUG_MESSAGE   1

#define RX_MODE 1

#define TX_MODE !RX_MODE

/* Set the mode RX_MODE or TX_MODE */
#define MODE TX_MODE

/* Set the delay (according the duty cycle for the ISM band of the frequency RF_FREQUENCY */

#define DUTY_CYCLE 0.01
//#define DELAY_NEXT_TX_IN_MILLISEC ((int)((timeOnAirInMilli / DUTY_CYCLE)-timeOnAirInMilli))

#define DELAY_NEXT_TX_IN_MILLISEC 5000

/* Set this flag to '1' to use the LoRa modulation or to '0' to use FSK modulation */
#define USE_MODEM_LORA  1
#define USE_MODEM_FSK   !USE_MODEM_LORA

#define RF_FREQUENCY                                    868100000 // Hz
#define TX_OUTPUT_POWER                                 14        // 14 dBm

#if USE_MODEM_LORA == 1

    #define LORA_BANDWIDTH                              0         // [0: 125 kHz,
                                                                  //  1: 250 kHz,
                                                                  //  2: 500 kHz,
                                                                  //  3: Reserved]
    #define LORA_SPREADING_FACTOR                       12         // [SF7..SF12]
    #define LORA_CODINGRATE                             1         // [1: 4/5,
                                                                  //  2: 4/6,
                                                                  //  3: 4/7,
                                                                  //  4: 4/8]
    #define LORA_PREAMBLE_LENGTH                        8         // Same for Tx and Rx
    #define LORA_SYMBOL_TIMEOUT                         5         // Symbols
    #define LORA_FIX_LENGTH_PAYLOAD_ON                  false
    #define LORA_FHSS_ENABLED                           false  
    #define LORA_NB_SYMB_HOP                            4     
    #define LORA_IQ_INVERSION_ON                        false
    #define LORA_CRC_ENABLED                            true
    
#elif USE_MODEM_FSK == 1

    #define FSK_FDEV                                    25000     // Hz
    #define FSK_DATARATE                                19200     // bps
    #define FSK_BANDWIDTH                               50000     // Hz
    #define FSK_AFC_BANDWIDTH                           83333     // Hz
    #define FSK_PREAMBLE_LENGTH                         5         // Same for Tx and Rx
    #define FSK_FIX_LENGTH_PAYLOAD_ON                   false
    #define FSK_CRC_ENABLED                             true
    
#else
    #error "Please define a modem in the compiler options."
#endif

#define RX_TIMEOUT_VALUE                                1000000000   // in us
#define BUFFER_SIZE                                     17        // Define the payload size here


Serial pc(USBTX,USBRX);

#if( defined ( TARGET_KL25Z ) || defined ( TARGET_LPC11U6X ) )
DigitalOut led(LED2);
#else
DigitalOut led(LED1);
#endif

/*
 *  Global variables declarations
 */
typedef enum
{
    LOWPOWER = 0,
    IDLE,
    
    RX,
    RX_TIMEOUT,
    RX_ERROR,
    
    TX,
    TX_TIMEOUT,
    
    CAD,
    CAD_DONE
}AppStates_t;

volatile AppStates_t State = LOWPOWER;

/*!
 * Radio events function pointer
 */
static RadioEvents_t RadioEvents;

/*
 *  Global variables declarations
 */
SX1272MB2xAS Radio( NULL );
uint16_t BufferSize = BUFFER_SIZE;
uint8_t Buffer[BUFFER_SIZE];

int16_t RssiValue = 0;
int8_t SnrValue = 0;

ModemType Modem;

uint16_t SeqNum = 0;

// Address of the Source Node
uint16_t NodeSrcAddr = NODE_ADDR;

void hexprintf(Serial* pc, const uint8_t* buffer, uint16_t size)
{
    for(uint16_t count = 0; count < size; count++)
    {
        pc->printf("%02x", buffer[count]);
    }
}
const uint8_t Msg[] = "TEST**##abcdefghijklmnopqrstuvwxyz#0123456789#ABCDEFGHIJKLMNOPQRSTUVWXYZ#0123456789#abcdefghijklmnopqrstuvwxyz#0123456789#ABCDEFGHIJKLMNOPQRSTUVWXYZ#0123456789#abcdefghijklmnopqrstuvwxyz#0123456789#ABCDEFGHIJKLMNOPQRSTUVWXYZ#0123456789#abcdefghijklmnopqrs#"; /* abc.. is for padding */

int main() 
{

    pc.format(8, Serial::None, 1);
    pc.baud(115200);
    
    debug( ">INFO RX/TX Modem \r\n\n" );
    
    // Initialize Radio driver
    RadioEvents.TxDone = OnTxDone;
    RadioEvents.RxDone = OnRxDone;
    RadioEvents.RxError = OnRxError;
    RadioEvents.TxTimeout = OnTxTimeout;
    RadioEvents.RxTimeout = OnRxTimeout;
    Radio.Init( &RadioEvents );
    
    // verify the connection with the board
    while( Radio.Read( REG_VERSION ) == 0x00  )
    {
        debug( "Radio could not be detected!\n\r", NULL );
        wait( 1 );
    }
            
    debug_if( ( DEBUG_MESSAGE & ( Radio.DetectBoardType( ) == SX1272MB2XAS ) ) , "\n\r > Board Type: SX1272MB2xAS < \n\r" );
  
    Radio.SetChannel( RF_FREQUENCY ); 

#if USE_MODEM_LORA == 1
    
    debug_if( LORA_FHSS_ENABLED, "\n\n\r             > LORA FHSS Mode < \n\n\r");
    debug_if( !LORA_FHSS_ENABLED, "\n\n\r             > LORA Mode < \n\n\r");

    Radio.SetTxConfig( MODEM_LORA, TX_OUTPUT_POWER, 0, LORA_BANDWIDTH,
                         LORA_SPREADING_FACTOR, LORA_CODINGRATE,
                         LORA_PREAMBLE_LENGTH, LORA_FIX_LENGTH_PAYLOAD_ON,
                         LORA_CRC_ENABLED, LORA_FHSS_ENABLED, LORA_NB_SYMB_HOP, 
                         LORA_IQ_INVERSION_ON, 2000000 );
    
    Radio.SetRxConfig( MODEM_LORA, LORA_BANDWIDTH, LORA_SPREADING_FACTOR,
                         LORA_CODINGRATE, 0, LORA_PREAMBLE_LENGTH,
                         LORA_SYMBOL_TIMEOUT, LORA_FIX_LENGTH_PAYLOAD_ON, 0,
                         LORA_CRC_ENABLED, LORA_FHSS_ENABLED, LORA_NB_SYMB_HOP, 
                         LORA_IQ_INVERSION_ON, true );
                         
#elif USE_MODEM_FSK == 1

    debug("\n\n\r              > FSK Mode < \n\n\r");
    Radio.SetTxConfig( MODEM_FSK, TX_OUTPUT_POWER, FSK_FDEV, 0,
                         FSK_DATARATE, 0,
                         FSK_PREAMBLE_LENGTH, FSK_FIX_LENGTH_PAYLOAD_ON,
                         FSK_CRC_ENABLED, 0, 0, 0, 2000000 );
    
    Radio.SetRxConfig( MODEM_FSK, FSK_BANDWIDTH, FSK_DATARATE,
                         0, FSK_AFC_BANDWIDTH, FSK_PREAMBLE_LENGTH,
                         0, FSK_FIX_LENGTH_PAYLOAD_ON, 0, FSK_CRC_ENABLED,
                         0, 0, false, true );
                         
#else

#error "Please define a modem in the compiler options."

#endif
     led = 0;
#if MODE == RX_MODE
     debug_if( DEBUG_MESSAGE, ">INFO Starting RX loop\r\n" ); 

    // start RX
    Radio.Rx( RX_TIMEOUT_VALUE );
    State = LOWPOWER;
    
#elif MODE == TX_MODE
    debug_if( DEBUG_MESSAGE, ">INFO Starting TX loop\r\n" ); 
    State = TX;
    
#else
#error "Please define a valid mode (RX_MODE, TX_MODE)."
#endif

// TODO ECHO_MODE : RX_MODE then TX_MODE

    while( 1 )
    {
        switch( State )
        {
        case RX:
                if( BufferSize > 0 )
                {
                    led = !led; // Invert LED

                    
                    // TODO ADD  tmst | number | Internal timestamp of "RX finished" event (32b unsigned)

                    pc.printf(">INFO RX modem=%d size=%d rssi=%d snr=%d freq=%d bw=%d sf=%d cr=%d buffer=",
                        Modem, BufferSize, RssiValue, SnrValue, RF_FREQUENCY, LORA_BANDWIDTH, LORA_SPREADING_FACTOR, LORA_CODINGRATE );
                    hexprintf(&pc, Buffer, BufferSize);
                    pc.printf("\r\n");

                    pc.printf("RX;%d;%d;%d;%d;%d;%d;%d;%d;",
                        Modem, BufferSize, RssiValue, SnrValue, RF_FREQUENCY, LORA_BANDWIDTH, LORA_SPREADING_FACTOR, LORA_CODINGRATE );
                    hexprintf(&pc, Buffer, BufferSize);
                    pc.printf("\r\n");
//                        wait_ms( 10 ); 
                    Radio.Rx( RX_TIMEOUT_VALUE );
                }
            
            State = LOWPOWER;
            break;
        case RX_TIMEOUT:
            Radio.Rx( RX_TIMEOUT_VALUE );  
            State = LOWPOWER;
            break;
        case RX_ERROR:
            // We have received a Packet with a CRC error
            pc.printf(">INFO RX_ERROR size=%d\r\n", BufferSize); 
            pc.printf("RX_ERROR;%d\r\n", BufferSize); 
            Radio.Rx( RX_TIMEOUT_VALUE );  
            State = LOWPOWER;
            break;
        case TX_TIMEOUT:
            pc.printf(">TX: TIMEOUT\r\n");
        case TX:
            int timeOnAirInMilli = Radio.TimeOnAir(Modem,BufferSize)/1000;
            if(DELAY_NEXT_TX_IN_MILLISEC!=0) {
                pc.printf(">INFO: wait %.4f sec before TX\r\n", DELAY_NEXT_TX_IN_MILLISEC/1000.0);
                wait_ms( DELAY_NEXT_TX_IN_MILLISEC );
            }
            led = !led; // Invert LED
            memcpy(( char* )Buffer, ( char* ) Msg, BufferSize);
            ++SeqNum;
            
            TemperatureValue = Radio.Read( REG_TEMP );
            
            /* refresh counters in payload (big endian, for readability) */
            Buffer[4] = (uint8_t)(SeqNum >> 8); /* MSB */
            Buffer[5] = (uint8_t)(SeqNum & 0x00FF); /* LSB */
            Buffer[6] = (uint8_t)(NodeSrcAddr >> 8); /* MSB */
            Buffer[7] = (uint8_t)(NodeSrcAddr & 0x00FF); /* LSB */
            Buffer[8] = (uint8_t)(TemperatureValue >> 8); /* MSB */
            Buffer[9] = (uint8_t)(TemperatureValue & 0x00FF); /* LSB */

            /* send packet */
            pc.printf(">TX: temp=%d src=%4x seqnum=%u size=%d timeOnAir=%.4f\r\n", TemperatureValue, NodeSrcAddr, SeqNum, BufferSize, timeOnAirInMilli/1000.0);
            wait_ms( 10 ); 
            Radio.Send( Buffer, BufferSize );
            State = LOWPOWER;
            break;            
        case LOWPOWER:
            break;
        default:
            State = LOWPOWER;
            break;
        }    
    }
}

void OnTxDone( void )
{
    Radio.Sleep( );
    State = TX;
    debug_if( DEBUG_MESSAGE, "> OnTxDone\n\r" );
}

void OnRxDone( uint8_t *payload, uint16_t size, int16_t rssi, int8_t snr)
{
    Radio.Sleep( );
    BufferSize = size;
    memcpy( Buffer, payload, BufferSize );
    RssiValue = rssi;
    SnrValue = snr;
    State = RX;
    debug_if( DEBUG_MESSAGE, "> OnRxDone\n\r" );
}

void OnTxTimeout( void )
{
    Radio.Sleep( );
    State = TX_TIMEOUT;
    debug_if( DEBUG_MESSAGE, "> OnTxTimeout\n\r" );
}

void OnRxTimeout( void )
{
    Radio.Sleep( );
    Buffer[ BufferSize ] = 0;
    State = RX_TIMEOUT;
    debug_if( DEBUG_MESSAGE, "> OnRxTimeout\n\r" );
}

void OnRxError( void )
{
    Radio.Sleep( );
    State = RX_ERROR;
    debug_if( DEBUG_MESSAGE, "> OnRxError\n\r" );
}
